/* Copyright (c) 2016 ARM Ltd., MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*
*
*  PCF8576 Universal LCD driver library
*
*  @author  Toyomasa Watarai
*  @version 1.0
*  @date    28-March-2016
*
*  Library for "PCF8576 Universal LCD driver for low multiplex rates"
*
*/
 
#ifndef PCF8576_H
#define PCF8576_H

#include "mbed.h"
#include "char_pattern.h"

#define PCF8576_CMD_MODE_SET    0x40
#define PCF8576_CMD_LOAD_DATA   0x00
#define PCF8576_CMD_DEVICE_SEL  0x60
#define PCF8576_CMD_BANK_SEL    0x78
#define PCF8576_CMD_BINK        0x70

#define PCF8576_DEFAULT_SLAVE_ADDRESS 0x70

/**
* PCF8576 Universal LCD driver library example
*
* @code
* #include "PCF8576.h"
* PCF8576 lcd(D14, D15);
* 
* int main() {
*     lcd.print("mbed");
*     while(1) {
*     }
* }
* @endcode
*/
class PCF8576 {
public:
    /**
    * PCF8576 constructor
    *
    * @param sda SDA pin
    * @param sdl SCL pin
    * @param addr Slave address of the I2C peripheral (default: 0x70)
    */
    PCF8576(PinName sda, PinName scl, int addr = PCF8576_DEFAULT_SLAVE_ADDRESS);
 
    /**
     * Create a PCF8576 instance which is connected to specified I2C pins
     * with specified address
     *
     * @param i2c_obj I2C object (instance)
     * @param addr Slave address of the I2C-bus peripheral (default: 0x70)
     */
    PCF8576(I2C &i2c_obj, int addr = PCF8576_DEFAULT_SLAVE_ADDRESS);
 
    /**
    * PCF8576 destructor
    */
    ~PCF8576();
 
    /** Initializa PCF8576 device
     *
     *  Configure LCD setting
     *
     */
    void initialize(void);

    /**
     * Print string
     *
     * @param str Poiner to print string
     */
    void print(char *str);
    
    /**
     * Print level-bar pictogram icon
     *
     * @param icon Level-bar pictogram icon from 0 (all off) to 0xF (all on)
     */
    void icon(uint32_t icon);

private:
    I2C  m_i2c;
    int  m_addr;
    char m_lcd_buf[14];
    char m_C5_mask;
    char m_C6_mask;

};

#endif
