/**
 ******************************************************************************
 * @file    MB85RSxx_SPI.h
 * @author  Toyomasa Watarai
 * @version V1.0.0
 * @date    24 April 2017
 * @brief   This file contains the class of an MB85RSxx FRAM library with SPI interface
 ******************************************************************************
 * @attention
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef MBED_MB85RSxx_SPI_H
#define MBED_MB85RSxx_SPI_H

#include "mbed.h"

#define MB85_WREN   0x06
#define MB85_WRDI   0x04
#define MB85_RDSR   0x05
#define MB85_WRSR   0x01
#define MB85_READ   0x03
#define MB85_WRITE  0x02
#define MB85_RDID   0x9F
#define MB85_FSTRD  0x0B
#define MB85_SLEEP  0xB9

#define MB85_DENSITY_64K    0x3
#define MB85_DENSITY_256K   0x5
#define MB85_DENSITY_512K   0x6
#define MB85_DENSITY_1M     0x7
#define MB85_DENSITY_2M     0x8

/**  Interface for accessing Fujitsu MB85RSxx FRAM
 *
 * @code
 * #include "mbed.h"
 * #include "MB85RCxx_I2C.h"
 * 
 * Serial pc(USBTX, USBRX);
 * 
 * #if defined(TARGET_LPC1768)
 * MB85RSxx_SPI _spi(p5, p6, p7, p8); // mosi, miso, sclk, cs
 * #elif defined(TARGET_LPC1114)
 * MB85RSxx_SPI _spi(dp2, dp1, dp6, dp9); // mosi, miso, sclk, cs
 * #else // Arduino R3 Shield form factor
 * MB85RSxx_SPI fram(D11, D12, D13, D10); // mosi, miso, sclk, cs
 * #endif
 * 
 * int main()
 * {
 *     uint8_t buf[16];
 *     uint32_t address;
 * 
 *     fram.write_enable();
 *     fram.fill(0, 0, 256);
 * 
 *     for (int i = 0; i < 16; i++) {
 *         buf[i] = i;
 *     }
 *     fram.write(0, buf, 16);
 * 
 *     for (address = 0; address < 0x80; address += 16) {
 *         fram.read(address, buf, 16);
 *         pc.printf("%08X : ", address);
 *         for (int i = 0; i < 16; i++) {
 *             pc.printf("%02X ", buf[i]);
 *         }
 *         pc.printf("\n");
 *     }
 * }
 * 
 * @endcode
 */

/** MB85RSxx_SPI class
 *
 *  MB85RSxx_SPI: A library to access Fujitsu MB85RSxx FRAM
 *
 */ 
class MB85RSxx_SPI
{
public:

    /** Create an MB85RSxx_SPI instance
     *  which is connected to specified SPI pins and chip select pin
     *
     * @param mosi SPI Master Out, Slave In pin
     * @param miso SPI Master In, Slave Out pin
     * @param sclk SPI Clock pin
     * @param cs Chip Select pin
     */
    MB85RSxx_SPI(PinName mosi, PinName miso, PinName sclk, PinName cs);

    /** Destructor of MB85RSxx_SPI
     */
    virtual ~MB85RSxx_SPI();
    
    /** Set the SPI bus clock frequency
     *
     * @param hz SPI bus clock frequency
     *
     */
    void frequency(int hz);
    
    /** Read device ID from MB85RSxx FRAM
     *
     * @param device_id Pointer to the byte-array to read data in to
     *
     * @returns memory dencity
     */
    void read_device_id(uint8_t* device_id);

    /** Read status register from MB85RSxx FRAM
     *
     * @returns Status register value
     */
    uint8_t read_status();

    /** Read data from memory address
     *
     * @param address Memory address
     * @param data Pointer to the byte-array to read data in to
     * @param length Number of bytes to read
     *
     */
    void read(uint32_t address, uint8_t* data, uint32_t length);

    /** Read byte data from memory address
     *
     * @param address Memory address
     *
     * @returns Read out data
     */
    uint8_t read(uint32_t address);

    /** Write data to memory address
     *
     * @param address Memory address
     * @param data Pointer to the byte-array data to write
     * @param length Number of bytes to write
     *
     */
    void write(uint32_t address, uint8_t* data, uint32_t length);

    /** Write data to memory address
     *
     * @param address Memory address
     * @param data data to write out to memory
     *
     */
    void write(uint32_t address, uint8_t data);

    /** Fill data to memory address
     *
     * @param address Memory address
     * @param data data to fill out to memory
     * @param length Number of bytes to write
     *
     */
    void fill(uint32_t address, uint8_t data, uint32_t length);

    /** Enable write access
     */
    void write_enable();

    /** Disable write access
     */
    void write_disable();

private:
    SPI _spi;
    DigitalOut _cs;
    int _address_bits;

};

#endif
