/**
 ******************************************************************************
 * @file    MB85RSxx_SPI.cpp
 * @author  Toyomasa Watarai
 * @version V1.0.0
 * @date    24 April 2017
 * @brief   MB85RSxx_SPI class implementation
 ******************************************************************************
 * @attention
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include "mbed.h"
#include "MB85RSxx_SPI.h"

MB85RSxx_SPI::MB85RSxx_SPI(PinName mosi, PinName miso, PinName sclk, PinName cs)
    :
    _spi(mosi, miso, sclk),
    _cs(cs)
{
    uint8_t buf[4];

    _cs = 1;
    _spi.format(8, 0);
    read_device_id(buf);
    if ((buf[2] & 0x1F) > MB85_DENSITY_512K) {
        _address_bits = 24;
    } else {
        _address_bits = 16;
    }
}

MB85RSxx_SPI::~MB85RSxx_SPI()
{
}

void MB85RSxx_SPI::frequency(int hz)
{
    _spi.frequency(hz);
}

void MB85RSxx_SPI::read_device_id(uint8_t* device_id)
{
    _cs = 0;
    _spi.write(MB85_RDID);
    for (int i = 0; i < 4; i++) {
        *device_id++ = (uint8_t)_spi.write(0);
    }
    _cs = 1;
}

uint8_t MB85RSxx_SPI::read_status()
{
    _cs = 0;
    _spi.write(MB85_RDSR);
    uint8_t st = (uint8_t)_spi.write(0);
    _cs = 1;
    
    return st;
}

void MB85RSxx_SPI::read(uint32_t address, uint8_t* data, uint32_t length)
{
    _cs = 0;    
    _spi.write(MB85_READ);
    if (_address_bits == 24) {
        _spi.write((uint8_t)((address >> 16) & 0xFF));
    }
    _spi.write((uint8_t)((address >>  8) & 0xFF));
    _spi.write((uint8_t)((address >>  0) & 0xFF));
    for (uint32_t i = 0; i < length; i++) {
        *data++ = (uint8_t)_spi.write(0);
    }
    _cs = 1;
}

uint8_t MB85RSxx_SPI::read(uint32_t address)
{
    uint8_t data;

    _cs = 0;
    _spi.write(MB85_READ);
    if (_address_bits == 24) {
        _spi.write((uint8_t)((address >> 16) & 0xFF));
    }
    _spi.write((uint8_t)((address >>  8) & 0xFF));
    _spi.write((uint8_t)((address >>  0) & 0xFF));
    data = (uint8_t)_spi.write(0);
    _cs = 1;
    
    return data;
}

void MB85RSxx_SPI::write(uint32_t address, uint8_t* data, uint32_t length)
{
    _cs = 0;
    _spi.write(MB85_WRITE);
    if (_address_bits == 24) {
        _spi.write((uint8_t)((address >> 16) & 0xFF));
    }
    _spi.write((uint8_t)((address >>  8) & 0xFF));
    _spi.write((uint8_t)((address >>  0) & 0xFF));
    for (uint32_t i = 0; i < length; i++) {
        _spi.write(*data++);
    }
    _cs = 1;
}

void MB85RSxx_SPI::write(uint32_t address, uint8_t data)
{
    _cs = 0;
    _spi.write(MB85_WRITE);
    if (_address_bits == 24) {
        _spi.write((uint8_t)((address >> 16) & 0xFF));
    }
    _spi.write((uint8_t)((address >>  8) & 0xFF));
    _spi.write((uint8_t)((address >>  0) & 0xFF));
    _spi.write(data);
    _cs = 1;
}

void MB85RSxx_SPI::fill(uint32_t address, uint8_t data, uint32_t length)
{
    _cs = 0;
    _spi.write(MB85_WRITE);
    if (_address_bits == 24) {
        _spi.write((uint8_t)((address >> 16) & 0xFF));
    }
    _spi.write((uint8_t)((address >>  8) & 0xFF));
    _spi.write((uint8_t)((address >>  0) & 0xFF));
    for (uint32_t i = 0; i < length; i++) {
        _spi.write(data);
    }
    _cs = 1;
}

void MB85RSxx_SPI::write_enable()
{
    _cs = 0;
    _spi.write(MB85_WREN);
    _cs = 1;
}

void MB85RSxx_SPI::write_disable()
{
    _cs = 0;
    _spi.write(MB85_WRDI);
    _cs = 1;
}
