/**
 ******************************************************************************
 * @file    MB85RCxx_I2C.cpp
 * @author  Toyomasa Watarai
 * @version V1.0.0
 * @date    22 April 2017
 * @brief   MB85RCxx_I2C class implementation
 ******************************************************************************
 * @attention
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include "mbed.h"
#include "MB85RCxx_I2C.h"

MB85RCxx_I2C::MB85RCxx_I2C(PinName sda, PinName scl, char slave_adr)
    :
    _i2c_p(new I2C(sda, scl)), 
    _i2c(*_i2c_p),
    _address(slave_adr)
{
    _i2c.frequency(400000);
}

MB85RCxx_I2C::~MB85RCxx_I2C()
{
    if (NULL != _i2c_p)
        delete  _i2c_p;
}

int MB85RCxx_I2C::read_device_id(char* device_id)
{
    _i2c.write(0xF8, &_address, 1, true);
    _i2c.read(0xF9, device_id, 3);
    
    return (device_id[1] & 0x0F);
}

void MB85RCxx_I2C::read(uint32_t address, char *data, uint32_t length)
{
    char byte_address[2];
    char i2c_adrs = (_address | ((address >> 15) & 0x02));
    
    byte_address[0] = ((address >> 8) & 0xFF);
    byte_address[1] = ((address >> 0) & 0xFF);
    _i2c.write(i2c_adrs, byte_address, 2, true);
    _i2c.read(i2c_adrs, data, length);
}

void MB85RCxx_I2C::write(uint32_t address, char *data, uint32_t length)
{
    char byte_address[2];
    char i2c_adrs = (_address | ((address >> 15) & 0x02));
    
    byte_address[0] = ((address >> 8) & 0xFF);
    byte_address[1] = ((address >> 0) & 0xFF);
    _i2c.write(i2c_adrs, byte_address, 2, true);
    for (uint32_t i = 0; i < length; i++) {
        _i2c.write(*data++);
    }
}

void MB85RCxx_I2C::write(uint32_t address, char data)
{
    char byte_address[2];
    char i2c_adrs = (_address | ((address >> 15) & 0x02));
    
    byte_address[0] = ((address >> 8) & 0xFF);
    byte_address[1] = ((address >> 0) & 0xFF);
    _i2c.write(i2c_adrs, byte_address, 2, true);
    _i2c.write(data);
}

void MB85RCxx_I2C::fill(uint32_t address, uint8_t data, uint32_t length)
{
    char byte_address[2];
    char i2c_adrs = (_address | ((address >> 15) & 0x02));
    
    byte_address[0] = ((address >> 8) & 0xFF);
    byte_address[1] = ((address >> 0) & 0xFF);
    _i2c.write(i2c_adrs, byte_address, 2, true);
    for (uint32_t i = 0; i < length; i++) {
        _i2c.write(data);
    }
}
