/**
 ******************************************************************************
 * @file    MB85RCxx_I2C.h
 * @author  Toyomasa Watarai
 * @version V1.0.0
 * @date    22 April 2017
 * @brief   This file contains the class of an MB85RCxx FRAM library with I2C interface
 ******************************************************************************
 * @attention
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef MBED_MB85RCxx_I2C_H
#define MBED_MB85RCxx_I2C_H

#include "mbed.h"
 
#define MB85_DENSITY_64K    0x3
#define MB85_DENSITY_256K   0x5
#define MB85_DENSITY_512K   0x6
#define MB85_DENSITY_1M     0x7
#define MB85_DENSITY_2M     0x8
#define MB85RC_I2C_ADDRESS  0xA0
#define I2C_WRITE           0
#define I2C_READ            1

/**  Interface for accessing Fujitsu MB85RCxx FRAM
 *
 * @code
 * #include "mbed.h"
 * #include "MB85RCxx_I2C.h"
 * 
 * Serial pc(USBTX, USBRX);
 * 
 * #if defined(TARGET_LPC1768)
 * MB85RCxx_I2C fram(p28, p27);
 * #else
 * MB85RCxx_I2C fram(D14, D15);
 * #endif
 * 
 * int main()
 * {
 *     char buf[16];
 *     uint32_t address;
 *     
 *     fram.fill(0, 0, 256);
 * 
 *     for (int i = 0; i < 16; i++) {
 *         buf[i] = i;
 *     }
 *     fram.write(0, buf, 16);
 * 
 *     for (address = 0; address < 0x80; address += 16) {
 *         fram.read(address, buf, 16);
 *         pc.printf("%08X : ", address);
 *         for (int i = 0; i < 16; i++) {
 *             pc.printf("%02X ", buf[i]);    
 *         }
 *         pc.printf("\n");
 *     }
 * }
 * 
 * @endcode
 */

/** MB85RCxx_I2C class
 *
 *  MB85RCxx_I2C: A library to access Fujitsu MB85RCxx_I2C FRAM
 *
 */ 
class MB85RCxx_I2C
{
public:

    /** Create an MB85RCxx_I2C instance
     *  which is connected to specified I2C pins with specified address
     *
     * @param sda I2C data line pin
     * @param scl I2C clock line pin
     * @param slave_adr (option) I2C slave address (default: 0xA0)
     */
    MB85RCxx_I2C(PinName sda, PinName scl, char slave_adr = MB85RC_I2C_ADDRESS);

    /** Destructor of MB85RCxx_I2C
     */
    virtual ~MB85RCxx_I2C();

    /** Read device ID from MB85RCxx FRAM
     *
     * @param device_id Pointer to the byte-array to read data in to
     *
     * @returns memory dencity
     */
    int read_device_id(char* device_id);
    
    /** Read data from memory address
     *
     * @param address Memory address
     * @param data Pointer to the byte-array to read data in to
     * @param length Number of bytes to read
     *
     */
    void read(uint32_t address, char *data, uint32_t length);

    /** Write data to memory address
     *
     * @param address Memory address
     * @param data Pointer to the byte-array data to write
     * @param length Number of bytes to write
     *
     */
    void write(uint32_t address, char *data, uint32_t length);

    /** Write data to memory address
     *
     * @param address Memory address
     * @param data data to write out to memory
     *
     */
    void write(uint32_t address, char data);

    /** Fill data to memory address
     *
     * @param address Memory address
     * @param data data to fill out to memory
     * @param length Number of bytes to write
     *
     */
    void fill(uint32_t address, uint8_t data, uint32_t length);

private:
    I2C         *_i2c_p;
    I2C         &_i2c;
    char        _address;
};

#endif
