/* Cypress FM25W256 F-RAM component library
 *
 * Copyright (c) 2016 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *  *
 *  @author  Toyomasa Watarai
 *  @version 1.0
 *  @date    5-March-2016
 *
 *  http://www.cypress.com/products/nonvolatile-ram
 *  http://www.cypress.com/documentation/datasheets/fm25w256-256-kbit-32-k-8-serial-spi-f-ram?source=search&keywords=FM25W256&cat=technical_documents
 *
 */

#include "mbed.h"
#include "FM25W256.h"

FM25W256::FM25W256(PinName mosi, PinName miso, PinName clk, PinName cs)
    : _spi(mosi, miso, clk), _cs(cs)
{
    _spi.format(8, 0);
    _spi.frequency(FM25W256_CLK);
    _cs = 1;
    wait_ms(1); // tPU = 1ms (min.)
}

FM25W256::FM25W256(SPI &spi, PinName cs)
    : _spi(spi), _cs(cs) 
{
    _spi.format(8, 0);
    _spi.frequency(FM25W256_CLK);
    _cs = 1;
    wait_ms(1); // tPU = 1ms (min.)
}

void FM25W256::write(uint16_t address, uint8_t data)
{
    _cs = 0;
    _spi.write(CMD_WREN);
    _cs = 1;

    _cs = 0;
    _spi.write(CMD_WRITE);
    _spi.write(address >> 8);
    _spi.write(address & 0xFF);
    _spi.write(data);
    _cs = 1;
}

void FM25W256::write(uint16_t address, uint8_t *data, uint16_t size)
{
    _cs = 0;
    _spi.write(CMD_WREN);
    _cs = 1;

    _cs = 0;
    _spi.write(CMD_WRITE);
    _spi.write(address >> 8);
    _spi.write(address & 0xFF);
    while(size--) {
        _spi.write(*data++);
    }
    _cs = 1;
}

uint8_t FM25W256::read(uint16_t address)
{
    uint8_t data;

    _cs = 0;
    _spi.write(CMD_READ);
    _spi.write(address >> 8);
    _spi.write(address & 0xFF);
    data = _spi.write(0);
    _cs = 1;

    return data;
}

void FM25W256::read(uint16_t address, uint8_t *buf, uint16_t size)
{
    _cs = 0;
    _spi.write(CMD_READ);
    _spi.write(address >> 8);
    _spi.write(address & 0xFF);
    while (size--) {
        *buf++ = _spi.write(0);
    }
    _cs = 1;
}

void FM25W256::wirte_status(uint8_t data)
{
    _cs = 0;
    _spi.write(CMD_WREN);
    _cs = 1;

    _cs = 0;
    _spi.write(CMD_WRSR);
    _spi.write(data);
    _cs = 1;
}

uint8_t FM25W256::read_status()
{
    uint8_t data;
    _cs = 0;
    _spi.write(CMD_RDSR);
    data = _spi.write(0);
    _cs = 1;
    
    return data;
}

void FM25W256::set_write_protect(E_WP bp)
{
    // Set WPEN, BP0 and BP1
    wirte_status((1 << 7) | (bp << 2));
}

void FM25W256::clear_write_protect()
{
    wirte_status(0);
}
    