/* * * * * * * * * * * * * * * * * * * * * * * * * * *
 * This drives the popular QDSP-6064 bubble display. *
 *                                                   *
 * Created by: Michael Dushkoff (mad1841@rit.edu)    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * */
#include "bubble.h"

/*
 * 
 */
void BubbleDisplay::init(PinName m0, PinName m1, PinName m2, PinName m3,
                  PinName m4, PinName m5, PinName m6, PinName m7,
                  PinName m8, PinName m9, PinName m10, PinName m11){
    // Set up pins
    _cat1 = new DigitalOut(m0);
    _anE = new DigitalOut(m1);
    _anC = new DigitalOut(m2);
    _cat3 = new DigitalOut(m3);
    _anDP = new DigitalOut(m4);
    _cat4 = new DigitalOut(m5);
    _anG = new DigitalOut(m6);
    _anD = new DigitalOut(m7);
    _anF = new DigitalOut(m8);
    _cat2 = new DigitalOut(m9);
    _anB = new DigitalOut(m10);
    _anA = new DigitalOut(m11);
    
    // Set the initial display number
    _seg = 0;
    
    // Set the initial character values
    _chrs[0] = ' ';
    _chrs[1] = ' ';
    _chrs[2] = ' ';
    _chrs[3] = ' ';
    
    // Initialize the Ticker
    _freq = DEF_DISPL_FREQ;
    _cycler.attach(this,&BubbleDisplay::cycle,1.0/(_freq));
}

/*
 * This is the default BubbleDisplay constructor that
 * maps the pins in a simple way for the LPC11U24.
 */ 
BubbleDisplay::BubbleDisplay(){
#if defined(TARGET_LPC1768) || defined(TARGET_LPC11U24)
    init(p21,p22,p23,p24,p25,p26,p36,p35,p34,p33,p30,p29);
#endif
}

/*
 * This allows a user to map the pins of the bubble display
 * to any possible pin that they desire.
 */
BubbleDisplay::BubbleDisplay(PinName m0, PinName m1, PinName m2, PinName m3,
              PinName m4, PinName m5, PinName m6, PinName m7,
              PinName m8, PinName m9, PinName m10, PinName m11){
    init(m0,m1,m2,m3,m4,m5,m6,m7,m8,m9,m10,m11);
}

/*
 * Default destructor
 */
BubbleDisplay::~BubbleDisplay(){
    // Clean up all of our pointers
    delete _cat1;
    delete _anE;
    delete _anC;
    delete _cat3;
    delete _anDP;
    delete _cat4;
    delete _anG;
    delete _anD;
    delete _anF;
    delete _cat2;
    delete _anB;
    delete _anA;
}

/*
 * This sets the cycle frequency to a specific value instead of the
 * default 10kHz.
 */
 void BubbleDisplay::setFreq(double freq){
     // Detach the cycler and then reattach it with the new frequency
     _freq=freq;
     _cycler.detach();
     _cycler.attach(this,&BubbleDisplay::cycle,1.0/(_freq));
}

/*
 * This cycles through the four seven segment displays on the
 * bubble display by switching on and off the correct cathodes.
 */
void BubbleDisplay::cycle(){
    // Set anodes to current display
    (*_anDP) = ((dispTabl[_chrs[_seg]] >> 7) & (0x01));
    (*_anA) = ((dispTabl[_chrs[_seg]] >> 6) & (0x01));
    (*_anB) = ((dispTabl[_chrs[_seg]] >> 5) & (0x01));
    (*_anC) = ((dispTabl[_chrs[_seg]] >> 4) & (0x01));
    (*_anD) = ((dispTabl[_chrs[_seg]] >> 3) & (0x01));
    (*_anE) = ((dispTabl[_chrs[_seg]] >> 2) & (0x01));
    (*_anF) = ((dispTabl[_chrs[_seg]] >> 1) & (0x01));
    (*_anG) = ((dispTabl[_chrs[_seg]] >> 0) & (0x01));
    
    switch(_seg){
        case 0:
            // Switch appropriate cathodes
            (*_cat1) = 0;
            (*_cat2) = 1;
            (*_cat3) = 1;
            (*_cat4) = 1;
            _seg = 1;
            break;
        case 1:
            // Switch appropriate cathodes
            (*_cat1) = 1;
            (*_cat2) = 0;
            (*_cat3) = 1;
            (*_cat4) = 1;
            _seg = 2;
            break;
        case 2:
            // Switch appropriate cathodes
            (*_cat1) = 1;
            (*_cat2) = 1;
            (*_cat3) = 0;
            (*_cat4) = 1;
            _seg = 3;
            break;
        default:
            // Switch appropriate cathodes
            (*_cat1) = 1;
            (*_cat2) = 1;
            (*_cat3) = 1;
            (*_cat4) = 0;
            _seg = 0;
            break;
    }
}

/*
 * This writes a sequence of characters from left to right
 * to the seven-segment displays.
 */
void BubbleDisplay::write(char c1, char c2, char c3, char c4){
    _chrs[0] = c1;
    _chrs[1] = c2;
    _chrs[2] = c3;
    _chrs[3] = c4;
}

/*
 * This writes a sequence of characters from left to right
 * to the seven-segment displays.
 */
void BubbleDisplay::write(char* c){
    _chrs[0] = c[0];
    _chrs[1] = c[1];
    _chrs[2] = c[2];
    _chrs[3] = c[3];
}

/*
 * This writes a sequence of integer number
 * to the seven-segment displays.
 */
void BubbleDisplay::write(uint16_t num){
    num %= 10000;
    _chrs[0] = (num/1000);
    _chrs[1] = ((num/100)%10);
    _chrs[2] = (((num/10)%100)%10);
    _chrs[3] = (((num%1000)%100)%10);
}
