/* Includes */
#include "mbed.h" /* Mbed include */

/* Lora includes */
#include "PinMap.h"
#include "sx1276-mbed-hal.h"

/* Serial communication include */
#include "BufferedSerial.h"

/* Set this flag to '1' to display debug messages on the console */
#define DEBUG_MESSAGE   1

/* Set this flag to '1' to use the LoRa modulation or to '0' to use FSK modulation */
#define USE_MODEM_LORA  1
#define USE_MODEM_FSK   !USE_MODEM_LORA
#define RF_FREQUENCY            RF_FREQUENCY_915_0 // Hz
#define TX_OUTPUT_POWER         14                  // 14 dBm

#if USE_MODEM_LORA == 1

#define LORA_BANDWIDTH          125000  // LoRa default, details in SX1276::BandwidthMap
#define LORA_SPREADING_FACTOR   LORA_SF7
#define LORA_CODINGRATE         LORA_ERROR_CODING_RATE_4_5

#define LORA_PREAMBLE_LENGTH    8       // Same for Tx and Rx
#define LORA_SYMBOL_TIMEOUT     5       // Symbols
#define LORA_FIX_LENGTH_PAYLOAD_ON  false
#define LORA_FHSS_ENABLED       false  
#define LORA_NB_SYMB_HOP        4     
#define LORA_IQ_INVERSION_ON    false
#define LORA_CRC_ENABLED        true
    
#endif 

#define RX_TIMEOUT_VALUE    0       // In ms
#define TX_TIMEOUT_VALUE    1000000 // In ms

//#define BUFFER_SIZE       32        // Define the payload size here
#define BUFFER_SIZE         64      // Define the payload size here

typedef struct {
        uint8_t header; // Header for identification of updated informations - 0 0 p temp LSM6DSL LSM303AGR
        int time; // Time between transmissions
        float p;  // Pressure of LPS22HB
        float temperatureLPS22HB; // Temperature from LPS22HB
        int32_t ag[3]; // Acceleration of the accelerometer and gyroscope LSM6DSL 
        int32_t w[3]; // Angular velocity of LSM6DSL
        int32_t a[3]; // Acceleration of the accelerometer LSM303AGR
        int32_t m[3]; // Heading of LSM303AGR
}Pkg1;

typedef struct {
        uint8_t header; // Header for identification of updated informations - 0 1 InternalCommunication HTS221
        int time; // Time between transmissions
        bool drogueStatus; // Drogue parachute status provided by Avionics
        bool mainStatus; //Main parachute status provided by Avionics
        bool mainStatusCOTS; // Main parachute status provided by COTS Altimeter
        bool drogueStatusCOTS; // Drogue status provided by COTS Altimeter
        float pressureBar; // Pressure by COTS Altimeter
        float temperature; // Temperature by COTS Altimeter
        int16_t timeStamp; // Timestamp from COTS Altimeter
        int16_t aglAlt; // AGL Altitude from COTS Altimeter
        int8_t battery; // Battery voltage reading from COTS Altimeter
        float humidity; // Humidity of HTS221
        float temperatureHTS221; // Temperature from HTS221
        //uint8_t filler[25];
}Pkg2;

typedef struct {
        uint8_t header; // Header for identification of updated informations - 1 0 GPS 
        int time; // Time between transmissions
        unsigned long timeOfWeek; //GPS time of week
        long timeOfWeekFracPart; // GPS time of week fractional part
        unsigned char gpsFix; // GPS fix
        long ecefx; // GPS X posiition
        long ecefy; // GPS Y posistion
        long ecefz; // GPS Z postion
        unsigned long positionAcc3D; // GPS 3D position accuracy
        long ecefvx; // GPS X velocity
        long ecefvy; // GPS Y velocity
        long ecefvz; // GPS Z velocity
        unsigned long speedAcc; // GPS speed accuracy
        unsigned char numbSat; // GPS number of satellites conected
        //uint8_t filler[8];
}Pkg3;


union Data {
    Pkg1 pkg1;
    Pkg2 pkg2;
    Pkg3 pkg3;
};

Data data;

/* LoRa modem instances and configurations */

static RadioEvents_t RadioEvents; // Calback functions struct

SX1276Generic *Radio; // Definition of a Radio object

bool received = false; // Flag to indicate the end of reception

/* Configuration function */
void SystemClock_Config(void);

/* Callback functions prototypes */

// Brief Function to be executed on Radio Tx Done event
void OnTxDone(void *radio, void *userThisPtr, void *userData);

// Brief Function to be executed on Radio Rx Done event
void OnRxDone(void *radio, void *userThisPtr, void *userData, uint8_t *payload, uint16_t size, int16_t rssi, int8_t snr );

// Brief Function executed on Radio Tx Timeout event
void OnTxTimeout(void *radio, void *userThisPtr, void *userData);

// Brief Function executed on Radio Rx Timeout event
void OnRxTimeout(void *radio, void *userThisPtr, void *userData);

// Brief Function executed on Radio Rx Error event
void OnRxError(void *radio, void *userThisPtr, void *userData);

// Brief Function executed on Radio Fhss Change Channel event
void OnFhssChangeChannel(void *radio, void *userThisPtr, void *userData, uint8_t channelIndex);

// Brief Function executed on CAD Done event
void OnCadDone(void *radio, void *userThisPtr, void *userData);

/* Serial communication to debug program */
BufferedSerial *ser;

int main() {
    SystemClock_Config(); /* Synchronize clock for TX and RX boards */
    
    /* Serial configuration */
    if (DEBUG_MESSAGE) {
        ser = new BufferedSerial(USBTX, USBRX);
        ser->baud(115200);
        ser->format(8);
    }
    
    /* General Header*/
    if (DEBUG_MESSAGE)
        ser->printf("Telemetry Rx inicial version program\r\n\r\n");
    
    Radio = new SX1276Generic(NULL, MURATA_SX1276,
            LORA_SPI_MOSI, LORA_SPI_MISO, LORA_SPI_SCLK, LORA_CS, LORA_RESET,
            LORA_DIO0, LORA_DIO1, LORA_DIO2, LORA_DIO3, LORA_DIO4, LORA_DIO5,
            LORA_ANT_RX, LORA_ANT_TX, LORA_ANT_BOOST, LORA_TCXO);
    
    if (DEBUG_MESSAGE) {        
        ser->printf("SX1276 Simple receiver aplication\r\n" );
        ser->printf("Frequency: %.1f\r\n", (double)RF_FREQUENCY/1000000.0);
        ser->printf("TXPower: %d dBm\r\n",  TX_OUTPUT_POWER);
        ser->printf("Bandwidth: %d Hz\r\n", LORA_BANDWIDTH);
        ser->printf("Spreading factor: SF%d\r\n", LORA_SPREADING_FACTOR);
    }
    
    // Initialize Radio driver
    RadioEvents.TxDone = OnTxDone;
    RadioEvents.RxDone = OnRxDone;
    RadioEvents.RxError = OnRxError;
    RadioEvents.TxTimeout = OnTxTimeout;
    RadioEvents.RxTimeout = OnRxTimeout; 
    
    // Initializes the radio    
    while (Radio->Init( &RadioEvents ) == false) {
        if (DEBUG_MESSAGE)
            ser->printf("Radio could not be detected!\r\n");
        wait( 1 );
    }
    
    // Display the board type
    switch(Radio->DetectBoardType()) {
        case SX1276MB1LAS:
            if (DEBUG_MESSAGE)
                ser->printf(" > Board Type: SX1276MB1LAS <\r\n");
            break;
        case SX1276MB1MAS:
            if (DEBUG_MESSAGE)
                ser->printf(" > Board Type: SX1276MB1LAS <\r\n");
        case MURATA_SX1276:
            if (DEBUG_MESSAGE)
                ser->printf(" > Board Type: MURATA_SX1276_STM32L0 <\r\n");
            break;
        case RFM95_SX1276:
            if (DEBUG_MESSAGE) 
                ser->printf(" > HopeRF RFM95xx <\r\n");
            break;
        default:
            if (DEBUG_MESSAGE)
                ser->printf(" > Board Type: unknown <\r\n");
    }
    
    Radio->SetChannel(RF_FREQUENCY ); // Sets the frequency of the communication
    
    // Debug message of the state of fhss
    if (LORA_FHSS_ENABLED) {
        if (DEBUG_MESSAGE)
            ser->printf("             > LORA FHSS Mode <\r\n");
    }    
    if (!LORA_FHSS_ENABLED) {
        if (DEBUG_MESSAGE)
            ser->printf("             > LORA Mode <\r\n");
    }
    // Sets the configuration of the transmission     
    Radio->SetTxConfig( MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                         LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                         LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                         LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP, 
                         LORA_IQ_INVERSION_ON, 2000 );
    
    // Sets the configuration of the reception
    Radio->SetRxConfig( MODEM_LORA, LORA_BANDWIDTH, LORA_SPREADING_FACTOR,
                         LORA_CODINGRATE, 0, LORA_PREAMBLE_LENGTH,
                         LORA_SYMBOL_TIMEOUT, LORA_FIX_LENGTH_PAYLOAD_ON, 0,
                         LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP, 
                         LORA_IQ_INVERSION_ON, true );
    if (DEBUG_MESSAGE)
        ser->printf("Starting Receive loop\r\n"); 
        
    Radio->Rx(RX_TIMEOUT_VALUE); // Puts the device in reception mode continuously
    
    while( 1 )
    {   
        //After the receiving, puts the device again in receive mode 
        if (received == true) {
            received = false;
            Radio->Rx(RX_TIMEOUT_VALUE); 
        }
    }
    
}


void SystemClock_Config(void)
{
#ifdef B_L072Z_LRWAN1_LORA
    /* 
     * The L072Z_LRWAN1_LORA clock setup is somewhat differnt from the Nucleo board.
     * It has no LSE.
     */
    RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
    RCC_OscInitTypeDef RCC_OscInitStruct = {0};

    /* Enable HSE Oscillator and Activate PLL with HSE as source */
    RCC_OscInitStruct.OscillatorType      = RCC_OSCILLATORTYPE_HSI;
    RCC_OscInitStruct.HSEState            = RCC_HSE_OFF;
    RCC_OscInitStruct.HSIState            = RCC_HSI_ON;
    RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_DEFAULT;
    RCC_OscInitStruct.PLL.PLLState        = RCC_PLL_ON;
    RCC_OscInitStruct.PLL.PLLSource       = RCC_PLLSOURCE_HSI;
    RCC_OscInitStruct.PLL.PLLMUL          = RCC_PLLMUL_6;
    RCC_OscInitStruct.PLL.PLLDIV          = RCC_PLLDIV_3;

    if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK) {
        // Error_Handler();
    }

    /* Set Voltage scale1 as MCU will run at 32MHz */
    __HAL_RCC_PWR_CLK_ENABLE();
    __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE1);

    /* Poll VOSF bit of in PWR_CSR. Wait until it is reset to 0 */
    while (__HAL_PWR_GET_FLAG(PWR_FLAG_VOS) != RESET) {};

    /* Select PLL as system clock source and configure the HCLK, PCLK1 and PCLK2
    clocks dividers */
    RCC_ClkInitStruct.ClockType = (RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2);
    RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
    RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
    RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;
    RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
    if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_1) != HAL_OK) {
        // Error_Handler();
    }
#endif
}

void OnTxDone(void *radio, void *userThisPtr, void *userData)
{
    Radio->Sleep( );
    if (DEBUG_MESSAGE)
        ser->printf("> OnTxDone\r\n");
}

void OnRxDone(void *radio, void *userThisPtr, void *userData, uint8_t *payload, uint16_t size, int16_t rssi, int8_t snr)
{
    Radio->Sleep( );
    received = true;
    uint8_t state = payload[0] >> 6;
    
    //ser->printf("The state is %x\r\n", state);
    ser->printf("> OnRxDone: RssiValue=%d dBm, SnrValue=%d, State=%d\r\n", rssi, snr, state);
    //for(int i = 0; i < BUFFER_SIZE; i++){
    //    ser->printf("%x ", payload[i]);   
    //}
    ser->printf("\r\n");
    if(state == 3){
        memcpy(&data.pkg1, payload, BUFFER_SIZE);
        //ser->printf("Header: %x, time: %d, p: %f, tempLPS22HB: %f, ag: %d; %d; %d, w: %d; %d; %d, a: %d; %d; %d, m: %d; %d; %d\r\n", pkg1.header, pkg1.time, pkg1.p, pkg1.temperatureLPS22HB, pkg1.ag[0], pkg1.ag[1], pkg1.ag[2], pkg1.w[0], pkg1.w[1], pkg1.w[2], pkg1.a[0], pkg1.a[1], pkg1.a[2], pkg1.m[0], pkg1.m[1], pkg1.m[2]);   
        ser->printf("%x,%d,%f,%f,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d\r\n", data.pkg1.header, data.pkg1.time, data.pkg1.p, data.pkg1.temperatureLPS22HB, data.pkg1.ag[0], data.pkg1.ag[1], data.pkg1.ag[2], data.pkg1.w[0], data.pkg1.w[1], data.pkg1.w[2], data.pkg1.a[0], data.pkg1.a[1], data.pkg1.a[2], data.pkg1.m[0], data.pkg1.m[1], data.pkg1.m[2]);
    }
    else if(state == 1){
        memcpy(&data.pkg2, payload, BUFFER_SIZE);
        //ser->printf("Header: %x, time: %d, parachuteStatus: %d%d%d%d, pressureBar: %f, temperature: %f, timeStamp: %d, aglAlt: %d, battery: %d, humidity: %f, tempHTS221: %f\r\n", pkg2.header, pkg2.time, pkg2.drogueStatus, pkg2.mainStatus, pkg2.mainStatusCOTS, pkg2.drogueStatusCOTS, pkg2.pressureBar, pkg2.temperature, pkg2.timeStamp, pkg2.aglAlt, pkg2.battery, pkg2.humidity, pkg2.temperatureHTS221);
        ser->printf("%x,%d,%d%d%d%d,%f,%f,%d,%d,%d,%f,%f\r\n", data.pkg2.header, data.pkg2.time, data.pkg2.drogueStatus, data.pkg2.mainStatus, data.pkg2.mainStatusCOTS, data.pkg2.drogueStatusCOTS, data.pkg2.pressureBar, data.pkg2.temperature, data.pkg2.timeStamp, data.pkg2.aglAlt, data.pkg2.battery, data.pkg2.humidity, data.pkg2.temperatureHTS221);
    }
    else if(state == 2){
        memcpy(&data.pkg3, payload, BUFFER_SIZE);
        //ser->printf("Header: %x, time: %d, timeOfWeek: %d, frac: %d, gpsFix: %d\r\n", pkg3.header, pkg3.time, pkg3.timeOfWeek, pkg3.timeOfWeekFracPart, pkg3.gpsFix);
        //ser->printf("eceFx: %d, eceFy: %d, eceFz: %d, posAcc3D: %d, eceFvx: %d, eceFvy: %d, eceFvz: %d\r\n", pkg3.ecefx, pkg3.ecefy, pkg3.ecefz, pkg3.positionAcc3D, pkg3.ecefvx, pkg3.ecefvy, pkg3.ecefvz);
        //ser->printf("speedAcc: %d, numbSat: %d\r\n", pkg3.speedAcc, pkg3.numbSat);
        ser->printf("%x,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d\r\n", data.pkg3.header, data.pkg3.time, data.pkg3.timeOfWeek, data.pkg3.timeOfWeekFracPart, data.pkg3.gpsFix, data.pkg3.ecefx, data.pkg3.ecefy, data.pkg3.ecefz, data.pkg3.positionAcc3D, data.pkg3.ecefvx, data.pkg3.ecefvy, data.pkg3.ecefvz, data.pkg3.speedAcc, data.pkg3.numbSat);
    }
}

void OnTxTimeout(void *radio, void *userThisPtr, void *userData)
{
    Radio->Sleep( );
    if(DEBUG_MESSAGE)
        ser->printf("> OnTxTimeout\r\n");
}

void OnRxTimeout(void *radio, void *userThisPtr, void *userData)
{
    Radio->Sleep( );
    if (DEBUG_MESSAGE)
        ser->printf("> OnRxTimeout\r\n");
}

void OnRxError(void *radio, void *userThisPtr, void *userData)
{
    Radio->Sleep( );
    received = true;
    if (DEBUG_MESSAGE)
        ser->printf("> OnRxError\r\n");
}
