/* mbed NationZ I2C/SPI TPM 2.0 Library,
 * Copyright (c) 2015, Microsoft Coprporation Inc.
 * by Stefan Thom (LordOfDorks) StefanTh@Microsoft.com, Stefan@ThomsR.Us
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 *
 * This code was developped and tested with a STMicro Nucleo-L152RE board with
 * the NationZ (http://www.nationz.com.cn/en) I2C and SPI parts as TSSOP28
 * connected in the following manner:
 *
 *                -------------
 *             NC +  1     28 + NC
 *             NC +  2  N  27 + NC
 *             NC +  3  a  26 + NC
 *            GND +  4  t  25 + GND
 * SCL/D15<-+  3V +  5  i  24 + 3V
 * 3V<-2K2<-+-SCL +  6  o  23 + NC
 *         NC<-PP +  7  n  22 + NC
 *             NC +  8  Z  21 + NC
 *             NC +  9     20 + NC
 *             3V + 10  I  19 + 3V
 *            GND + 11  2  18 + GND
 *             NC + 12  C  17 + NC
 *             NC + 13     16 + #RST->3V
 *             NC + 14     15 + SDA-+->2k2->3V
 *                -------------     +->SDA/D14
 *
 * TestCode for I2C:
 * 
 * NTZTPM20 tpm(I2C_SDA, I2C_SCL);
 * uint8_t randBuf[32] = {0};
 * uint32_t result = 0;
 *
 * result = tpm.TPM2_Startup(NTZTPM20::TPM_SU_CLEAR);
 * printf("TPM2_Startup = 0x%08x\n\r", result);
 * result =tpm.TPM2_SelfTest(0x01);
 * printf("TPM2_SelfTest = 0x%08x\n\r", result);
 * result =tpm.TPM2_GetRandom(sizeof(randBuf), randBuf);
 * printf("TPM2_GetRandom = 0x%08x\n\r", result);
 * result =tpm.TPM2_StirRandom(sizeof(randBuf), randBuf);
 * printf("TPM2_StirRandom = 0x%08x\n\r", result);
 * result =tpm.TPM2_Shutdown(NTZTPM20::TPM_SU_CLEAR);
 * printf("TPM2_Shutdown = 0x%08x\n\r", result);
 *
 *                -------------
 *             NC +  1     28 + NC
 *             NC +  2  N  27 + NC
 * MOSI/D11<-MOSI +  3  a  26 + NC
 *            GND +  4  t  25 + GND
 *             3V +  5  i  24 + 3V
 *             NC +  6  o  23 + NC
 *         NC<-PP +  7  n  22 + SPI_CS->CS/D10
 *             NC +  8  Z  21 + SPI_CLK->SCK/D13
 * MISO/D12<-MISO +  9     20 + NC
 *             3V + 10  S  19 + 3V
 *            GND + 11  P  18 + GND
 *             NC + 12  I  17 + NC
 *             NC + 13     16 + #RST->3V
 *             NC + 14     15 + NC
 *                -------------
 *
 * TestCode for SPI:
 * 
 * NTZTPM20 tpm(SPI_MOSI, SPI_MISO, SPI_SCK, SPI_CS);
 * uint8_t randBuf[32] = {0};
 * uint32_t result = 0;
 *
 * result = tpm.TPM2_Startup(NTZTPM20::TPM_SU_CLEAR);
 * printf("TPM2_Startup = 0x%08x\n\r", result);
 * result =tpm.TPM2_SelfTest(0x01);
 * printf("TPM2_SelfTest = 0x%08x\n\r", result);
 * result =tpm.TPM2_GetRandom(sizeof(randBuf), randBuf);
 * printf("TPM2_GetRandom = 0x%08x\n\r", result);
 * result =tpm.TPM2_StirRandom(sizeof(randBuf), randBuf);
 * printf("TPM2_StirRandom = 0x%08x\n\r", result);
 * result =tpm.TPM2_Shutdown(NTZTPM20::TPM_SU_CLEAR);
 * printf("TPM2_Shutdown = 0x%08x\n\r", result);
 *
 * All TPM 2.0 commands and structures are defined in the TCG TPM 2.0 Library
 * specification that can be found at:
 * http://www.trustedcomputinggroup.org/resources/tpm_library_specification
 * The NationZ TPM does provice a simple and easy to use vendor specific CBR
 * interface and not the TCG defined TIS interface. It supports all major
 * Algorithms (RSA, ECC, AES, SHA-HMAC) as well as the family of SMx algotithms.
 *
 */

#include "mbed.h"

//#define TPM_NTZ_DEBUG_OUTPUT 1

#ifndef min
#define min(a,b) (((a) < (b)) ? (a) : (b))
#endif

#define UINT16_TO_BYTEARRAY(__dataIn, __array, __offset)\
__array[__offset] = (uint8_t)((__dataIn >> 8) & 0x00ff);\
__array[__offset + 1] = (uint8_t)(__dataIn & 0x00ff);\
 
#define UINT32_TO_BYTEARRAY(__dataIn, __array, __offset)\
__array[__offset + 0] = (uint8_t)((__dataIn >> 24) & 0x000000ff);\
__array[__offset + 1] = (uint8_t)((__dataIn >> 16) & 0x000000ff);\
__array[__offset + 2] = (uint8_t)((__dataIn >> 8) & 0x000000ff);\
__array[__offset + 3] = (uint8_t)(__dataIn & 0x000000ff);\

#define BYTEARRAY_TO_UINT16(__arrayIn, __offset)\
((((uint16_t)(__arrayIn[__offset])) << 8) | \
   (uint16_t)(__arrayIn[__offset + 1])) \

#define BYTEARRAY_TO_UINT32(__arrayIn, __offset)\
((((uint32_t)(__arrayIn[__offset])) << 24) | \
 (((uint32_t)(__arrayIn[__offset + 1])) << 16) | \
 (((uint32_t)(__arrayIn[__offset + 2])) << 8) | \
  (uint32_t)(__arrayIn[__offset + 3])) \
 
class NTZTPM20
{
public:
    // Subset of defined TCG constants
    const static uint32_t TPM_RC_SUCCESS = 0x00000000;
    const static uint32_t TPM_RC_FAILURE = 0x00000101;
    const static uint16_t TPM_ST_NO_SESSIONS = 0x8001;
    const static uint16_t TPM_ST_SESSIONS = 0x8002;
    const static uint32_t TPM_CC_GetRandom = 0x0000017B;
    const static uint32_t TPM_CC_StirRandom = 0x00000146;
    const static uint16_t TPM_SU_CLEAR = 0x0000;
    const static uint16_t TPM_SU_STATE = 0x0001;

    // Constructor for the I2C variant of the chip
    NTZTPM20(PinName sda, PinName scl);
    
    // Constructor for the SPI variant of the chip
    NTZTPM20(PinName mosi, PinName miso, PinName clk, PinName cs);
    
    // Release all held resources
    ~NTZTPM20();
    
    // Execute command on the TPM
    uint32_t Execute(uint8_t* pbCmd, uint32_t cbCmd, uint8_t* pbRsp, uint32_t cbRsp, uint32_t timeout);
    
    // Perform TPM startup
    uint32_t TPM2_Startup(uint16_t startupType);

    // Perform TPM shutdown
    uint32_t TPM2_Shutdown(uint16_t shutdownType);

    // Perform TPM selftest
    uint32_t TPM2_SelfTest(uint8_t fullTest);

    // Get random number
    uint32_t TPM2_GetRandom(uint16_t bytesRequested, uint8_t* randomBytes);

    // Stir random number generator
    uint32_t TPM2_StirRandom(uint16_t inDataLen, uint8_t* inData);

private:
    // NationZ I2C device address on the bus
    const static uint8_t m_I2CDevice_Address = (0x58 << 1);
    
    // Timeout flag
    volatile bool m_TimeoutTriggered;

    // Poor mans execution synchronization
    bool m_ExclusiveAccess;
        
    // Device hook-up
    I2C* m_I2CTpmDev;
    SPI* m_SPITpmDev;
    DigitalOut* m_SPICSTpmDev;
    
    uint32_t ParseResponseHeader(uint8_t* pbRsp, uint32_t rspLen, uint16_t* rspTag, uint32_t* rspSize, uint32_t* cursor);
    void TimeoutTrigger(void);
    uint32_t ExecuteI2C(uint8_t* pbCmd, uint32_t cbCmd, uint8_t* pbRsp, uint32_t cbRsp);
    uint32_t ExecuteSPI(uint8_t* pbCmd, uint32_t cbCmd, uint8_t* pbRsp, uint32_t cbRsp);
};