//Hydraulic Control Board Rainbow
//distributed by Sungwoo Kim
//       2022/05/31

#include "mbed.h"
#include "FastPWM.h"
#include "INIT_HW.h"
#include "function_CAN.h"
#include "SPI_EEP_ENC.h"
#include "setting.h"
#include "function_utilities.h"
#include "stm32f4xx_flash.h"
#include "FlashWriter.h"
#include <string>
#include <iostream>
#include <cmath>


// DAC ///////////////////////////////////////////
AnalogOut dac_1(PA_4); // 0.0f ~ 1.0f
AnalogOut dac_2(PA_5); // 0.0f ~ 1.0f

// ADC ///////////////////////////////////////////
ADC_HandleTypeDef hadc1;
ADC_HandleTypeDef hadc2;
ADC_HandleTypeDef hadc3;

AnalogIn adc1(PC_4); //pressure_1
AnalogIn adc2(PC_5); //pressure_2
AnalogIn adc3(PC_1); //current
AnalogIn adc4(PB_1); //V_EXI
AnalogIn adc5(PA_1); //LVDT

// PWM ///////////////////////////////////////////
float PWM_duty = 0.0f;

// SPI ///////////////////////////////////////////
SPI eeprom(PB_15, PB_14, PB_13); // EEPROM //(SPI_MOSI, SPI_MISO, SPI_SCK);
DigitalOut eeprom_cs(PB_12);
SPI enc(PC_12,PC_11,PC_10);
DigitalOut enc_cs(PD_2);

// LED ///////////////////////////////////////////
DigitalOut LED(PA_15);

// LVDT ///////////////////////////////////////////
DigitalOut LVDT_H(PB_6);
DigitalOut LVDT_L(PB_7);

// MOTOR_ENA ///////////////////////////////////////////
DigitalOut M_ENABLE(PA_2);

// CAN ///////////////////////////////////////////
CAN can(PB_8, PB_9, 1000000);
CANMessage msg;
//void onMsgReceived()
//{
//    CAN_RX_HANDLER();
//}

// Variables ///////////////////////////////////////////
State pos;
State vel;
State Vout;
State force;
State torq;         // unit : N
State torq_dot;
State pres_A;       // unit : bar
State pres_B;
State cur;          // unit : mA
State valve_pos;
State valve_pos_raw;

extern int CID_RX_CMD;
extern int CID_RX_REF_POSITION;
extern int CID_RX_REF_OPENLOOP;
extern int CID_RX_REF_PWM;

extern int CID_TX_INFO;
extern int CID_TX_POS_VEL_TORQ;
extern int CID_TX_PWM;
extern int CID_TX_CURRENT;
extern int CID_TX_VOUT;
extern int CID_TX_VALVE_POSITION;
extern int CID_TX_SOMETHING;

inline float tanh_inv(float y)
{
    if(y >= 1.0f - 0.000001f) y = 1.0f - 0.000001f;
    if(y <= -1.0f + 0.000001f) y = -1.0f + 0.000001f;
    return log(sqrt((1.0f+y)/(1.0f-y)));
}


/*******************************************************************************
 *  REFERENCE MODE
 ******************************************************************************/
enum _REFERENCE_MODE {
    MODE_REF_NO_ACT = 0,
    MODE_REF_DIRECT,
    MODE_REF_FINDHOME
};

/*******************************************************************************
 *  CONTROL MODE
 ******************************************************************************/
enum _CONTROL_MODE {
    //control mode
    MODE_NO_ACT = 0,                                    //0
    MODE_VALVE_POSITION_CONTROL,                        //1
    MODE_JOINT_CONTROL,                                 //2

    MODE_VALVE_OPEN_LOOP,                               //3
    MODE_JOINT_ADAPTIVE_BACKSTEPPING,                   //4
    MODE_RL,                                            //5

    MODE_JOINT_POSITION_PRES_CONTROL_PWM,               //6
    MODE_JOINT_POSITION_PRES_CONTROL_VALVE_POSITION,    //7
    MODE_VALVE_POSITION_PRES_CONTROL_LEARNING,          //8

    MODE_TEST_CURRENT_CONTROL,                          //9
    MODE_TEST_PWM_CONTROL,                              //10

    MODE_CURRENT_CONTROL,                               //11
    MODE_JOINT_POSITION_TORQUE_CONTROL_CURRENT,         //12
    MODE_JOINT_POSITION_PRES_CONTROL_CURRENT,           //13
    MODE_VALVE_POSITION_TORQUE_CONTROL_LEARNING,        //14

    //utility
    MODE_TORQUE_SENSOR_NULLING = 20,                    //20
    MODE_VALVE_NULLING_AND_DEADZONE_SETTING,            //21
    MODE_FIND_HOME,                                     //22
    MODE_VALVE_GAIN_SETTING,                            //23
    MODE_PRESSURE_SENSOR_NULLING,                       //24
    MODE_PRESSURE_SENSOR_CALIB,                         //25
    MODE_ROTARY_FRICTION_TUNING,                        //26

    MODE_DDV_POS_VS_PWM_ID = 30,                        //30
    MODE_DDV_DEADZONE_AND_CENTER,                       //31
    MODE_DDV_POS_VS_FLOWRATE,                           //32
    MODE_SYSTEM_ID,                                     //33
    MODE_FREQ_TEST,                                     //34
    MODE_SEND_BUFFER,                                   //35
    MODE_SEND_OVER,                                     //36
    MODE_STEP_TEST,                                     //37
};

void SystemClock_Config(void)
{
    RCC_OscInitTypeDef RCC_OscInitStruct = {0};
    RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};

    __HAL_RCC_PWR_CLK_ENABLE();
    __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE1);

    RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE;
    RCC_OscInitStruct.HSEState = RCC_HSE_ON;
//    RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_DEFAULT;
    RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
    RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSE;
    RCC_OscInitStruct.PLL.PLLM = 4;//4
    RCC_OscInitStruct.PLL.PLLN = 96; //96
    RCC_OscInitStruct.PLL.PLLP = RCC_PLLP_DIV2;
    RCC_OscInitStruct.PLL.PLLQ = 2;
    RCC_OscInitStruct.PLL.PLLR = 2;
    if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK) {
        //Error_Handler();
    }
    if (HAL_PWREx_EnableOverDrive() != HAL_OK) {
        //Error_Handler();
    }
    RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_HCLK|RCC_CLOCKTYPE_SYSCLK
                                  |RCC_CLOCKTYPE_PCLK1|RCC_CLOCKTYPE_PCLK2;
    RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
    RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
    RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV4;
    RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV2;
}



int main()
{

    HAL_Init();
    SystemClock_Config();

    LED = 0;

    // SPI INIT
    eeprom_cs = 1;
    eeprom.format(8,3);
    eeprom.frequency(5000000); //5M
    eeprom_cs = 0;
    make_delay();

    enc_cs = 1;
    enc.format(8,0);
    enc.frequency(5000000); //10M
    enc_cs = 0;
    make_delay();

    // spi _ enc
    spi_enc_set_init();
    make_delay();

    //  bno rom
    spi_eeprom_write(RID_BNO, (int16_t) 0);
    make_delay();
    ////////

    // rom
    ROM_CALL_DATA();
    make_delay();

    // ADC init
    RCC->APB2ENR |= RCC_APB2ENR_ADC1EN;                        // clock for ADC3
    RCC->APB2ENR |= RCC_APB2ENR_ADC2EN;                        // clock for ADC2
    RCC->APB2ENR |= RCC_APB2ENR_ADC3EN;                        // clock for ADC1

    RCC->AHB1ENR |= RCC_AHB1ENR_GPIOAEN;                        // Enable clock for GPIOC
    RCC->AHB1ENR |= RCC_AHB1ENR_GPIOBEN;                        // Enable clock for GPIOB
    RCC->AHB1ENR |= RCC_AHB1ENR_GPIOCEN;                        // Enable clock for GPIOB

    Init_ADC1();
    Init_ADC2();
    Init_ADC3();
    make_delay();

    // CAN
    can.attach(&CAN_RX_HANDLER);
    CAN_ID_INIT();
    make_delay();

    //can.reset();
    can.filter(msg.id, 0xFFFFF000, CANStandard);
//    can.filter(0b100000000, 0b100000010, CANStandard);  //CAN ID 100~400번대 통과하게

    // TMR1 init (PWM)
    Init_TMR1();
    TIM1->CR1 &= ~(TIM_CR1_UDIS);
    make_delay();

    // TMR2 init (Control)
    Init_TMR2();
    TIM2->CR1 &= ~(TIM_CR1_UDIS);
    make_delay();

    // TMR3 init (Sensors)
    Init_TMR3();
    TIM3->CR1 &= ~(TIM_CR1_UDIS);
    make_delay();

    // TIM4 init (LVDT)
    Init_TIM4();
    TIM4->CR1 &= ~(TIM_CR1_UDIS);
    make_delay();

    //Timer priority
    NVIC_SetPriority(TIM3_IRQn, 3);
    NVIC_SetPriority(TIM2_IRQn, 4);
    NVIC_SetPriority(TIM4_IRQn, 2);
    HAL_NVIC_SetPriority(CAN1_RX0_IRQn, 14, 0);

    //DAC init
    if (SENSING_MODE == 0) {
        dac_1 = FORCE_VREF / 3.3f;
        dac_2 = 0.0f;
    } else if (SENSING_MODE == 1) {
        dac_1 = PRES_A_VREF / 3.3f;
        dac_2 = PRES_B_VREF / 3.3f;
    }

    make_delay();

    for (int i=0; i<50; i++) {
        if(i%2==0)
            ID_index_array[i] = - i * 0.5f;
        else
            ID_index_array[i] =  (i+1) * 0.5f;
    }

    M_ENABLE = 1;

    /************************************
    ***     Program is operating!
    *************************************/
    while(1) {

//        if (LED > 0) LED = 0;
//        else LED = 1;
    }
}


// Velocity feedforward for SW valve
float DDV_JOINT_POS_FF(float REF_JOINT_VEL)
{
    int i = 0;
    float Ref_Valve_Pos_FF = 0.0f;
    for(i=0; i<VALVE_POS_NUM; i++) {
        if(REF_JOINT_VEL >= min(JOINT_VEL[i],JOINT_VEL[i+1]) && REF_JOINT_VEL <=  max(JOINT_VEL[i],JOINT_VEL[i+1])) {
            if(i==0) {
                if(JOINT_VEL[i+1] == JOINT_VEL[i]) {
                    Ref_Valve_Pos_FF = (float) VALVE_CENTER;
                } else {
                    Ref_Valve_Pos_FF = ((float) 10/(JOINT_VEL[i+1] - JOINT_VEL[i]) * (REF_JOINT_VEL - JOINT_VEL[i])) + (float) VALVE_CENTER;
                }
            } else {
                if(JOINT_VEL[i+1] == JOINT_VEL[i-1]) {
                    Ref_Valve_Pos_FF = (float) VALVE_CENTER;
                } else {
                    Ref_Valve_Pos_FF = ((float) 10*(ID_index_array[i+1] - ID_index_array[i-1])/(JOINT_VEL[i+1] - JOINT_VEL[i-1]) * (REF_JOINT_VEL - JOINT_VEL[i-1])) + (float) VALVE_CENTER + (float) (10*ID_index_array[i-1]);
                }
            }
            break;
        }
    }
    if(REF_JOINT_VEL > max(JOINT_VEL[VALVE_POS_NUM-1], JOINT_VEL[VALVE_POS_NUM-2])) {
        Ref_Valve_Pos_FF = (float) VALVE_MAX_POS;
    } else if(REF_JOINT_VEL < min(JOINT_VEL[VALVE_POS_NUM-1], JOINT_VEL[VALVE_POS_NUM-2])) {
        Ref_Valve_Pos_FF = (float) VALVE_MIN_POS;
    }

    Ref_Valve_Pos_FF = (float) VELOCITY_COMP_GAIN * 0.01f * (float) (Ref_Valve_Pos_FF - (float) VALVE_CENTER);  //VELOCITY_COMP_GAIN : 0~100
    return Ref_Valve_Pos_FF;
}

// Valve feedforward for SW valve
void VALVE_POS_CONTROL(float REF_VALVE_POS)
{
    int i = 0;

    if(REF_VALVE_POS > VALVE_MAX_POS) {
        REF_VALVE_POS = VALVE_MAX_POS;
    } else if(REF_VALVE_POS < VALVE_MIN_POS) {
        REF_VALVE_POS = VALVE_MIN_POS;
    }
    valve_pos_err = (float) (REF_VALVE_POS - valve_pos.sen);
    valve_pos_err_diff = valve_pos_err - valve_pos_err_old;
    valve_pos_err_old = valve_pos_err;
    valve_pos_err_sum += valve_pos_err;
    if (valve_pos_err_sum > 1000.0f) valve_pos_err_sum = 1000.0f;
    if (valve_pos_err_sum<-1000.0f) valve_pos_err_sum = -1000.0f;

    VALVE_PWM_RAW_FB = P_GAIN_VALVE_POSITION * valve_pos_err + I_GAIN_VALVE_POSITION * valve_pos_err_sum + D_GAIN_VALVE_POSITION * valve_pos_err_diff;

    for(i=0; i<24; i++) {
        if(REF_VALVE_POS >= min(VALVE_POS_VS_PWM[i],VALVE_POS_VS_PWM[i+1]) && (float) REF_VALVE_POS <=  max(VALVE_POS_VS_PWM[i],VALVE_POS_VS_PWM[i+1])) {
            if(i==0) {
                VALVE_PWM_RAW_FF = (float) 1000.0f / (float) (VALVE_POS_VS_PWM[i+1] - VALVE_POS_VS_PWM[i]) * ((float) REF_VALVE_POS - VALVE_POS_VS_PWM[i]);
            } else {
                VALVE_PWM_RAW_FF = (float) 1000.0f* (float) (ID_index_array[i+1] - ID_index_array[i-1])/(VALVE_POS_VS_PWM[i+1] - VALVE_POS_VS_PWM[i-1]) * ((float) REF_VALVE_POS - VALVE_POS_VS_PWM[i-1]) + 1000.0f * (float) ID_index_array[i-1];
            }
            break;
        }
    }
    Vout.ref = VALVE_PWM_RAW_FF + VALVE_PWM_RAW_FB;
//    int R_FET = 1.5f;
//    int R_Motor = 5.4f;
//    Vout.ref = Vout.ref + Vout.ref/R_Motor*R_FET*2.0f;
}

// Valve feedforward for SW valve
void VALVE_POS_CONTROL_DZ(float REF_VALVE_POS)
{
    int i = 0;
    if(REF_VALVE_POS > VALVE_MAX_POS) {
        REF_VALVE_POS = VALVE_MAX_POS;
    } else if(REF_VALVE_POS < VALVE_MIN_POS) {
        REF_VALVE_POS = VALVE_MIN_POS;
    }
    valve_pos_err = (float) (REF_VALVE_POS - valve_pos.sen);
    valve_pos_err_diff = valve_pos_err - valve_pos_err_old;
    valve_pos_err_old = valve_pos_err;
    valve_pos_err_sum += valve_pos_err;
    if (valve_pos_err_sum > 1000.0f) valve_pos_err_sum = 1000.0f;
    if (valve_pos_err_sum<-1000.0f) valve_pos_err_sum = -1000.0f;

    VALVE_PWM_RAW_FB = P_GAIN_VALVE_POSITION * valve_pos_err + I_GAIN_VALVE_POSITION * valve_pos_err_sum + D_GAIN_VALVE_POSITION * valve_pos_err_diff;

    for(i=0; i<24; i++) {
        if(REF_VALVE_POS >= min(VALVE_POS_VS_PWM[i],VALVE_POS_VS_PWM[i+1]) && (float) REF_VALVE_POS <=  max(VALVE_POS_VS_PWM[i],VALVE_POS_VS_PWM[i+1])) {
            if(i==0) {
                VALVE_PWM_RAW_FF = (float) 1000.0f / (float) (VALVE_POS_VS_PWM[i+1] - VALVE_POS_VS_PWM[i]) * ((float) REF_VALVE_POS - VALVE_POS_VS_PWM[i]);
            } else {
                VALVE_PWM_RAW_FF = (float) 1000.0f* (float) (ID_index_array[i+1] - ID_index_array[i-1])/(VALVE_POS_VS_PWM[i+1] - VALVE_POS_VS_PWM[i-1]) * ((float) REF_VALVE_POS - VALVE_POS_VS_PWM[i-1]) + 1000.0f * (float) ID_index_array[i-1];
            }
            break;
        }
    }
    Vout.ref = VALVE_PWM_RAW_FF + VALVE_PWM_RAW_FB;
//    int R_FET = 1.5f;
//    int R_Motor = 5.4f;
//    Vout.ref = Vout.ref + Vout.ref/R_Motor*R_FET*2.0f;
}

// PWM duty vs. voltage output of L6205 in STM board
#define LT_MAX_IDX  57
float LT_PWM_duty[LT_MAX_IDX] = {-100.0f, -80.0f, -60.0f, -50.0f, -40.0f, -35.0f, -30.0f, -25.0f, -20.0f,
                                 -19.0f, -18.0f, -17.0f, -16.0f, -15.0f, -14.0f, -13.0f, -12.0f, -11.0f, -10.0f,
                                 -9.0f, -8.0f, -7.0f, -6.0f, -5.0f, -4.0f, -3.0f, -2.0f, -1.0f, 0.0f,
                                 1.0f, 2.0f, 3.0f, 4.0f, 5.0f, 6.0f, 7.0f, 8.0f, 9.0f, 10.0f,
                                 11.0f, 12.0f, 13.0f, 14.0f, 15.0f, 16.0f, 17.0f, 18.0f, 19.0f, 20.0f,
                                 25.0f, 30.0f, 35.0f, 40.0f, 50.0f, 60.0f, 80.0f, 100.0f
                                };  // duty
float LT_Voltage_Output[LT_MAX_IDX] = {-230.0f, -215.0f, -192.5f, -185.0f, -177.5f, -170.0f, -164.0f, -160.0f, -150.0f,
                                       -150.0f, -145.0f, -145.0f, -145.0f, -135.0f, -135.0f, -135.0f, -127.5f, -127.5f, -115.0f,
                                       -115.0f, -115.0F, -100.0f, -100.0f, -100.0f, -60.0f, -60.0f, -10.0f, -5.0f, 0.0f,
                                       7.5f, 14.0f, 14.0f, 14.0f, 42.5f, 42.5f, 42.5f, 80.0f, 80.0f, 105.0f,
                                       105.0f, 105.0f, 120.0f, 120.0f, 120.0f, 131.0f, 131.0f, 140.0f, 140.0f, 140.0f,
                                       155.0f, 160.0f, 170.0f, 174.0f, 182.0f, 191.0f, 212.0f, 230.0f
                                      }; // mV

float PWM_duty_byLT(float Ref_V)
{
    float PWM_duty = 0.0f;
    if(Ref_V<LT_Voltage_Output[0]) {
        PWM_duty = (Ref_V-LT_Voltage_Output[0])/1.5f+LT_PWM_duty[0];
    } else if (Ref_V>=LT_Voltage_Output[LT_MAX_IDX-1]) {
        PWM_duty = (Ref_V-LT_Voltage_Output[LT_MAX_IDX-1])/1.5f+LT_PWM_duty[LT_MAX_IDX-1];
    } else {
        int idx = 0;
        for(idx=0; idx<LT_MAX_IDX-1; idx++) {
            float ini_x = LT_Voltage_Output[idx];
            float fin_x = LT_Voltage_Output[idx+1];
            float ini_y = LT_PWM_duty[idx];
            float fin_y = LT_PWM_duty[idx+1];
            if(Ref_V>=ini_x && Ref_V<fin_x) {
                PWM_duty = (fin_y-ini_y)/(fin_x-ini_x)*(Ref_V-ini_x) + ini_y;
                break;
            }
        }
    }
    return PWM_duty;
}



/*******************************************************************************
                            TIMER INTERRUPT
*******************************************************************************/

//------------------------------------------------
//     TMR3 : Sensor 20kHz
//------------------------------------------------
float FREQ_TMR3 = (float)FREQ_20k;
long  CNT_TMR3 = 0;
float DT_TMR3 = (float)DT_20k;
extern "C" void TIM3_IRQHandler(void)
{
    if (TIM3->SR & TIM_SR_UIF ) {

//        if (LED > 0) LED = 0;
//        else LED = 1;

        float PSEN1 = 0.0f;
        float PSEN2 = 0.0f;
        float CURRENT_SEN = 0.0f;

        /////////////////////////Current////////////////////////////////////////////////////////////////////////////
        HAL_ADC_Start(&hadc2);
        HAL_ADC_PollForConversion(&hadc2, 1);
        CURRENT_SEN = (float) HAL_ADC_GetValue(&hadc2);
        cur.UpdateSen(((float)CURRENT_SEN-2047.5f)/2047.5f*10.0f, FREQ_TMR3, 500.0f); // unit : mA

        /////////////////////////V_EXI////////////////////////////////////////////////////////////////////////////
        HAL_ADC_Start(&hadc2);
        HAL_ADC_PollForConversion(&hadc2, 1);
        V_EXI = (float) HAL_ADC_GetValue(&hadc2);

        /////////////////////////Encoder////////////////////////////////////////////////////////////////////////////
//        if (CNT_TMR1 % 2) == 0) {
        ENC_UPDATE();
//        }

        /////////////////////////Force or Pressure//////////////////////////////////////////////////////////////////
        if (SENSING_MODE == 0) {  // Force sensing

            HAL_ADC_Start(&hadc1);
            HAL_ADC_PollForConversion(&hadc1, 1);
            PSEN1 = (float) HAL_ADC_GetValue(&hadc1);

            HAL_ADC_Start(&hadc1);
            HAL_ADC_PollForConversion(&hadc1, 1);
            PSEN2 = (float) HAL_ADC_GetValue(&hadc1);

            force.UpdateSen((((float)PSEN1) - 2047.5f)/TORQUE_SENSOR_PULSE_PER_TORQUE, FREQ_TMR3, 100.0f); // unit : N //100Hz

        } else if (SENSING_MODE == 1) { // Pressure sensing

            HAL_ADC_Start(&hadc1);
            HAL_ADC_PollForConversion(&hadc1, 1);
            PSEN1 = (float) HAL_ADC_GetValue(&hadc1);

            HAL_ADC_Start(&hadc1);
            HAL_ADC_PollForConversion(&hadc1, 1);
            PSEN2 = (float) HAL_ADC_GetValue(&hadc1);

            float pres_A_new, pres_B_new;
            pres_A_new = (((float)PSEN1) - PRES_A_NULL_pulse)/ PRES_SENSOR_A_PULSE_PER_BAR; // unit : bar
            pres_B_new = (((float)PSEN2) - PRES_B_NULL_pulse)/ PRES_SENSOR_B_PULSE_PER_BAR;

            pres_A.UpdateSen(pres_A_new,FREQ_TMR3,200.0f);
            pres_B.UpdateSen(pres_B_new,FREQ_TMR3,200.0f);

            if ((OPERATING_MODE & 0b01) == 0) { // Rotary Actuator
                float torq_new = (PISTON_AREA_A * pres_A.sen - PISTON_AREA_B * pres_B.sen) * 0.0001f; // mm^3*bar >> Nm
                torq.UpdateSen(torq_new,FREQ_TMR3,100.0f);  // unit : Nm   //1000Hz
            } else if ((OPERATING_MODE & 0b01) == 1) { // Linear Actuator
                float force_new = (PISTON_AREA_A * pres_A.sen - PISTON_AREA_B * pres_B.sen) * 0.1f; // mm^2*bar >> N
                force.UpdateSen(force_new,FREQ_TMR3,100.0f);  // unit : N  //1000Hz
            }
        }
        CNT_TMR3++;
    }
    TIM3->SR = 0x0;  // reset the status register
}



//------------------------------------------------
//     TMR4 : LVDT 1kHz
//------------------------------------------------
float LVDT_new = 0.0f;
float LVDT_old = 0.0f;
float LVDT_f_cut = 1000.0f;
float LVDT_LPF = 0.0f;
float LVDT_sum = 0.0f;
float FREQ_TMR4 = (float)FREQ_1k;
float DT_TMR4 = (float)DT_1k;
long  CNT_TMR4 = 0;
int   TMR4_FREQ_10k = (int)FREQ_10k;
extern "C" void TIM4_IRQHandler(void)
{
    if (TIM4->SR & TIM_SR_UIF ) {

        float LVDT_OUT = 0.0f;
        LVDT_sum = 0.0f;

        LVDT_L = 0;
        LVDT_H = 1;

        for (int ij = 0; ij<120; ij++) {
            if (ij < 20) {
                continue;
            } else if (ij == 20) {
//                LED = 1;
            } else if (ij == 100) {
                LVDT_H = 0;
                LVDT_L = 0;
            }
            ADC3->CR2  |= 0x40000000;
            LVDT_new = ((float)ADC3->DR - 2047.0f);
            if(DIR_VALVE_ENC < 0) LVDT_new = 0.0f - LVDT_new;
            LVDT_sum = LVDT_sum + LVDT_new;
        }
//        LED = 0;

//        LVDT_H = 0;
//        LVDT_L = 0;

        LVDT_new = (LVDT_sum + 2047.0f * 100.0f) * 0.01f;

        float alpha_LVDT = 1.0f/(1.0f+TMR_FREQ_1k/(2.0f*PI*300.0f));
        LVDT_LPF = (1.0f-alpha_LVDT) * LVDT_LPF + alpha_LVDT * LVDT_new;
        valve_pos.sen = LVDT_LPF;


        CNT_TMR4++;
    }
    TIM4->SR = 0x0;  // reset the status register
}

//------------------------------------------------
//     TMR2 : Control 5kHz
//------------------------------------------------
float FREQ_TMR2 = (float)FREQ_5k;
float DT_TMR2 = (float)DT_5k;
int cnt_trans = 0;
int cnt_jitter = 0;
extern "C" void TIM2_IRQHandler(void)
{
    if (TIM2->SR & TIM_SR_UIF ) {
        LED = 1;
        if(MODE_POS_FT_TRANS == 1) {
            if (alpha_trans == 1.0f) MODE_POS_FT_TRANS = 2;
            alpha_trans = (float)(1.0f - cos(3.141592f * (float)cnt_trans * DT_TMR2 /3.0f))/2.0f;
            cnt_trans++;
            torq.err_int = 0.0f;
            force.err_int = 0.0f;
            if((float)cnt_trans * DT_TMR2 > 3.0f)
                MODE_POS_FT_TRANS = 2;
        } else if(MODE_POS_FT_TRANS == 3) {
            if (alpha_trans == 0.0f) MODE_POS_FT_TRANS = 0;
            alpha_trans = (float)(1.0f + cos(3.141592f * (float)cnt_trans * DT_TMR2 /3.0f))/2.0f;
            cnt_trans++;
            torq.err_int = 0.0f;
            force.err_int = 0.0f;
            if((float) cnt_trans * DT_TMR2 > 3.0f )
                MODE_POS_FT_TRANS = 0;
        } else if(MODE_POS_FT_TRANS == 2) {
            alpha_trans = 1.0f;
            cnt_trans = 0;
        } else {
            alpha_trans = 0.0f;
            cnt_trans = 0;
        }


        // Reference Update ==========================================================
        switch (REFERENCE_MODE) {
            case MODE_REF_NO_ACT: {
                break;
            }
            case MODE_REF_DIRECT: {
                pos.ref = REF_POSITION;
                vel.ref = REF_VELOCITY;
                torq.ref = REF_TORQUE;
                force.ref = REF_FORCE;
                break;
            }
            case MODE_REF_FINDHOME: {
                pos.ref = REF_POSITION_FINDHOME;
                vel.ref = 0.0f;
                torq.ref = 0.0f;
                force.ref = 0.0f;
                break;
            }
            default:
                break;
        }

        if (((OPERATING_MODE&0b110)>>1) == 0) {
            K_v = 1.03f; // Q = K_v*sqrt(deltaP)*tanh(C_d*Xv);
            C_d = 0.16f;
            mV_PER_mA = 500.0f; // 5000mV/10mA
            mV_PER_pulse = 0.5f; // 5000mV/10000pulse
            mA_PER_pulse = 0.001f; // 10mA/10000pulse
        } else if (((OPERATING_MODE&0b110)>>1) == 1) {
            K_v = 0.5f; // KNR (LPM >> mA) , 100bar
            mV_PER_mA = 166.6666f; // 5000mV/30mA
            mV_PER_pulse = 0.5f; // 5000mV/10000pulse
            mA_PER_pulse = 0.003f; // 30mA/10000pulse
        } else if (((OPERATING_MODE&0b110)>>1) == 2) {
            C_d = 0.0000845f; // Q = C_d * Valve_pos * sqrt(deltaP*alpha/(1+alpha)) : Valve_pos = 10000, deltaP = 70, alpha = 1 -> Q = 5
        }

        // =====================================================================
        // CONTROL LOOP --------------------------------------------------------
        // =====================================================================
        int UTILITY_MODE = 0;
        int CONTROL_MODE = 0;

        if (CONTROL_UTILITY_MODE >= 20 || CONTROL_UTILITY_MODE == 0) {
            UTILITY_MODE = CONTROL_UTILITY_MODE;
            CONTROL_MODE = MODE_NO_ACT;
        } else {
            CONTROL_MODE = CONTROL_UTILITY_MODE;
            UTILITY_MODE = MODE_NO_ACT;
        }
        // UTILITY MODE ------------------------------------------------------------
        switch (UTILITY_MODE) {
            case MODE_NO_ACT: {
                break;
            }

            case MODE_TORQUE_SENSOR_NULLING: {
                static float FORCE_pulse_sum = 0.0f;
                static float PresA_pulse_sum = 0.0f;
                static float PresB_pulse_sum = 0.0f;

                // DAC Voltage reference set
                float VREF_TuningGain = -0.000003f;
                if (TMR3_COUNT_TORQUE_NULL < TMR_FREQ_5k * 5) {
                    LED = 1;
                    if(SENSING_MODE == 0) { // Force Sensor (Loadcell)
                        FORCE_pulse_sum = FORCE_pulse_sum + force.sen*TORQUE_SENSOR_PULSE_PER_TORQUE;
                        if (TMR3_COUNT_TORQUE_NULL % 10 == 0) {
                            float FORCE_pluse_mean = FORCE_pulse_sum / 10.0f;
                            FORCE_pulse_sum = 0.0f;
                            FORCE_VREF += VREF_TuningGain * (0.0f - FORCE_pluse_mean);
                            if (FORCE_VREF > 3.3f) FORCE_VREF = 3.3f;
                            if (FORCE_VREF < 0.0f) FORCE_VREF = 0.0f;
                            dac_1 = FORCE_VREF / 3.3f;
                        }
                    } else if (SENSING_MODE == 1) { // Pressure Sensor
                        PresA_pulse_sum += pres_A.sen*PRES_SENSOR_A_PULSE_PER_BAR;
                        PresB_pulse_sum += pres_B.sen*PRES_SENSOR_B_PULSE_PER_BAR;
                        if (TMR3_COUNT_TORQUE_NULL % 10 == 0) {
                            float PresA_pluse_mean = PresA_pulse_sum / 10.0f;
                            float PresB_pluse_mean = PresB_pulse_sum / 10.0f;
                            PresA_pulse_sum = 0.0f;
                            PresB_pulse_sum = 0.0f;

                            PRES_A_VREF += VREF_TuningGain * (0.0f - PresA_pluse_mean);
                            if (PRES_A_VREF > 3.3f) PRES_A_VREF = 3.3f;
                            if (PRES_A_VREF < 0.0f) PRES_A_VREF = 0.0f;
                            dac_1 = PRES_A_VREF / 3.3f;
                            PRES_B_VREF += VREF_TuningGain * (0.0f - PresB_pluse_mean);
                            if (PRES_B_VREF > 3.3f) PRES_B_VREF = 3.3f;
                            if (PRES_B_VREF < 0.0f) PRES_B_VREF = 0.0f;
                            dac_2 = PRES_B_VREF / 3.3f;
                        }
                    }
                    TMR3_COUNT_TORQUE_NULL++;
                } else {
                    if(SENSING_MODE == 0 ) { // Force Sensor (Loadcell)
                        FORCE_pulse_sum = 0.0f;
                        dac_1 = FORCE_VREF / 3.3f;
                        spi_eeprom_write(RID_FORCE_SENSOR_VREF, (int16_t)(FORCE_VREF * 1000.0f));
                    } else if (SENSING_MODE == 1) {
                        PresA_pulse_sum = 0.0f;
                        PresB_pulse_sum = 0.0f;
                        dac_1 = PRES_A_VREF / 3.3f;
                        dac_2 = PRES_B_VREF / 3.3f;
                        spi_eeprom_write(RID_PRES_A_SENSOR_VREF, (int16_t)(PRES_A_VREF * 1000.0f));
                        spi_eeprom_write(RID_PRES_B_SENSOR_VREF, (int16_t)(PRES_B_VREF * 1000.0f));
                    }
                    CONTROL_UTILITY_MODE = MODE_NO_ACT;
                    TMR3_COUNT_TORQUE_NULL = 0;
                }
                break;
            }

            case MODE_FIND_HOME: {
                static int cnt_findhome = 0;
                static int cnt_terminate_findhome = 0;
                static float FINDHOME_POSITION_pulse = 0.0f;
                static float FINDHOME_POSITION_pulse_OLD = 0.0f;
                static float FINDHOME_VELOCITY_pulse = 0.0f;
                static float REF_POSITION_FINDHOME_INIT = 0.0f;

                if (FINDHOME_STAGE == FINDHOME_INIT) {
                    REFERENCE_MODE = MODE_REF_FINDHOME;
                    cnt_findhome = 0;
                    cnt_terminate_findhome = 0;
                    pos.ref = pos.sen;
                    vel.ref = 0.0f;
                    REF_POSITION_FINDHOME = pos.ref;
                    FINDHOME_STAGE = FINDHOME_GOTOLIMIT;
                } else if (FINDHOME_STAGE == FINDHOME_GOTOLIMIT) {
                    int cnt_check_enc = (TMR_FREQ_5k/20); // 5000/20 = 250tic = 50msec
                    if(cnt_findhome%cnt_check_enc == 0) {
                        FINDHOME_POSITION_pulse = pos.sen*ENC_PULSE_PER_POSITION;
                        FINDHOME_VELOCITY_pulse = FINDHOME_POSITION_pulse - FINDHOME_POSITION_pulse_OLD;
                        FINDHOME_POSITION_pulse_OLD = FINDHOME_POSITION_pulse;
                    }
                    cnt_findhome++;

                    if (fabs(FINDHOME_VELOCITY_pulse) <= 1) {
                        cnt_terminate_findhome = cnt_terminate_findhome + 1;
                    } else {
                        cnt_terminate_findhome = 0;
                    }

                    if ((cnt_terminate_findhome < 3*TMR_FREQ_5k) &&  cnt_findhome < 10*TMR_FREQ_5k) { // wait for 3sec
                        double GOTOHOME_SPEED = 10.0f; // 20mm/s or 20deg/s
                        if (HOMEPOS_OFFSET > 0) {
                            REF_POSITION_FINDHOME = REF_POSITION_FINDHOME + GOTOHOME_SPEED*DT_5k;
                        } else {
                            REF_POSITION_FINDHOME = REF_POSITION_FINDHOME - GOTOHOME_SPEED*DT_5k;
                        }
                        CONTROL_MODE = MODE_JOINT_CONTROL;
                        alpha_trans = 0.0f;
                    } else {
                        ENC_SET((long)((long)HOMEPOS_OFFSET*10));
                        REF_POSITION_FINDHOME_INIT = (float)((long)HOMEPOS_OFFSET*10);
                        FINDHOME_POSITION_pulse = 0;
                        FINDHOME_POSITION_pulse_OLD = 0;
                        FINDHOME_VELOCITY_pulse = 0;

                        cnt_findhome = 0;
                        cnt_terminate_findhome = 0;
                        pos.ref = 0.0f;
                        FINDHOME_STAGE = FINDHOME_ZEROPOSE;
                    }
                } else if (FINDHOME_STAGE == FINDHOME_ZEROPOSE) {

//                    int T_move = 2*TMR_FREQ_5k;
                    int T_move = 10000;
                    REF_POSITION_FINDHOME = ((0.0f - REF_POSITION_FINDHOME_INIT)*0.5f*(1.0f - cos(3.14159f * (float)cnt_findhome / (float)T_move)) + (float)REF_POSITION_FINDHOME_INIT)/ENC_PULSE_PER_POSITION;

                    cnt_findhome++;

                    REFERENCE_MODE = MODE_REF_FINDHOME;
                    CONTROL_MODE = MODE_JOINT_CONTROL;
                    alpha_trans = 0.0f;

                    if (cnt_findhome >= T_move) {
                        cnt_findhome = 0;
                        pos.ref = 0.0f;
                        FINDHOME_STAGE = FINDHOME_INIT;
                        CONTROL_UTILITY_MODE = MODE_JOINT_CONTROL;
                        REFERENCE_MODE = MODE_REF_DIRECT;
                    }
                }
                break;
            }

            case MODE_DDV_POS_VS_PWM_ID: {
                CONTROL_MODE = MODE_VALVE_OPEN_LOOP;
                VALVE_ID_timer = VALVE_ID_timer + 1;

                if(VALVE_ID_timer < TMR_FREQ_5k*1) {
                    Vout.ref = 3000.0f * sin(2.0f*3.14159f*VALVE_ID_timer/TMR_FREQ_5k * 100.0f);
                } else if(VALVE_ID_timer < TMR_FREQ_5k*2) {
                    Vout.ref = 1000.0f*(ID_index_array[ID_index]);
//                    int R_FET = 1.5f;
//                    int R_Motor = 5.4f;
//                    Vout.ref = Vout.ref + Vout.ref/R_Motor*R_FET*2.0f;
                } else if(VALVE_ID_timer == TMR_FREQ_5k*2) {
                    VALVE_POS_TMP = 0;
                    data_num = 0;
                } else if(VALVE_ID_timer < TMR_FREQ_5k*3) {
                    data_num = data_num + 1;
                    VALVE_POS_TMP = VALVE_POS_TMP + valve_pos.sen;
                } else if(VALVE_ID_timer == TMR_FREQ_5k*3) {
                    Vout.ref = 0.0f;
                } else {
                    VALVE_POS_AVG[ID_index] = VALVE_POS_TMP / data_num;
                    VALVE_ID_timer = 0;
                    ID_index= ID_index +1;
                }

                if(ID_index>=25) {
                    int i;
                    VALVE_POS_AVG_OLD = VALVE_POS_AVG[0];
                    for(i=0; i<25; i++) {
                        VALVE_POS_VS_PWM[i] = (int16_t) (VALVE_POS_AVG[i]);
                        if(VALVE_POS_AVG[i] > VALVE_POS_AVG_OLD) {
                            VALVE_MAX_POS = VALVE_POS_AVG[i];
                            VALVE_POS_AVG_OLD = VALVE_MAX_POS;
                        } else if(VALVE_POS_AVG[i] < VALVE_POS_AVG_OLD) {
                            VALVE_MIN_POS = VALVE_POS_AVG[i];
                            VALVE_POS_AVG_OLD = VALVE_MIN_POS;
                        }
                    }
                    VALVE_ELECTRIC_CENTER = VALVE_POS_VS_PWM[0];
                    spi_eeprom_write(RID_VALVE_ELECTRIC_CENTER, (int16_t) VALVE_ELECTRIC_CENTER);
                    spi_eeprom_write(RID_VALVE_MAX_POS, (int16_t) VALVE_MAX_POS);
                    spi_eeprom_write(RID_VALVE_MIN_POS, (int16_t) VALVE_MIN_POS);
                    for(int i=0; i<25; i++) {
                        spi_eeprom_write(RID_VALVE_POS_VS_PWM_0 + i, (int16_t) VALVE_POS_VS_PWM[i]);
                    }
                    ID_index = 0;
                    CONTROL_UTILITY_MODE = MODE_NO_ACT;
                }
                break;
            }
            case MODE_DDV_DEADZONE_AND_CENTER: {
                CONTROL_MODE = MODE_VALVE_OPEN_LOOP;
                VALVE_DZ_timer = VALVE_DZ_timer + 1;
                if(first_check == 0) {
                    if(VALVE_DZ_timer < (int) (1.0f * (float) TMR_FREQ_5k)) {
                        Vout.ref = VALVE_VOLTAGE_LIMIT * 1000.0f;
                    } else if(VALVE_DZ_timer == (int) (1.0f * (float) TMR_FREQ_5k)) {
                        Vout.ref = VALVE_VOLTAGE_LIMIT * 1000.0f;
                        pos_plus_end = pos.sen;
                    } else if(VALVE_DZ_timer < (int) (2.0f * (float) TMR_FREQ_5k)) {
                        Vout.ref = -VALVE_VOLTAGE_LIMIT * 1000.0f;
                    } else if(VALVE_DZ_timer == (int) (2.0f * (float) TMR_FREQ_5k)) {
                        Vout.ref = -VALVE_VOLTAGE_LIMIT * 1000.0f;
                        pos_minus_end = pos.sen;
                    } else if(VALVE_DZ_timer < (int) (3.0f * (float) TMR_FREQ_5k)) {
                        Vout.ref = (float) P_GAIN_JOINT_POSITION * (0.5f * (float) pos_plus_end + 0.5f * (float) pos_minus_end - (float) pos.sen)*(float) 50.0f;
                    } else if(VALVE_DZ_timer < (int) (4.0f * (float) TMR_FREQ_5k)) {
                        Vout.ref = (float) P_GAIN_JOINT_POSITION * (0.5f * (float) pos_plus_end + 0.5f * (float) pos_minus_end - (float) pos.sen)*(float) 50.0f;
                        data_num = data_num + 1;
                        VALVE_POS_TMP = VALVE_POS_TMP + valve_pos.sen;
                    } else if(VALVE_DZ_timer == (int) (4.0f * (float) TMR_FREQ_5k)) {
                        Vout.ref = (float) P_GAIN_JOINT_POSITION * (0.5f * (float) pos_plus_end + 0.5f * (float) pos_minus_end - (float) pos.sen)*(float) 50.0f;
                        DDV_POS_AVG = VALVE_POS_TMP / data_num;
                        START_POS = pos.sen;
                        VALVE_POS_TMP = 0;
                        data_num = 0;

                    } else if(VALVE_DZ_timer < (int) (5.0f * (float) TMR_FREQ_5k)) {
                        valve_pos_raw.ref = DDV_POS_AVG;
                        VALVE_POS_CONTROL(valve_pos_raw.ref);

                    } else if(VALVE_DZ_timer < (int) (6.0f * (float) TMR_FREQ_5k)) {
                        valve_pos_raw.ref = DDV_POS_AVG;
                        VALVE_POS_CONTROL(valve_pos_raw.ref);

                    } else if(VALVE_DZ_timer == (int) (6.0f * (float) TMR_FREQ_5k)) {
                        valve_pos_raw.ref = DDV_POS_AVG;
                        VALVE_POS_CONTROL(valve_pos_raw.ref);
                        FINAL_POS = pos.sen;

                        if((FINAL_POS - START_POS)>1) {
                            DZ_case = 1;
                        } else if((FINAL_POS - START_POS)<-1) {
                            DZ_case = -1;
                        } else {
                            DZ_case = 0;
                        }

                        first_check = 1;
                        DZ_DIRECTION = 1;
                        VALVE_DZ_timer = 0;
                        Ref_Valve_Pos_Old = DDV_POS_AVG;
                        DZ_NUM = 1;
                        DZ_index = 1;

                    }
                } else {
                    if((DZ_case == -1 && DZ_NUM == 1) | (DZ_case == 1 && DZ_NUM == 1)) {
                        if(VALVE_DZ_timer < (int) (1.0f * (float) TMR_FREQ_5k)) {
                            Vout.ref = (float) P_GAIN_JOINT_POSITION * (0.5f * (float) pos_plus_end + 0.5f * (float) pos_minus_end - (float) pos.sen)*(float) 50.0f;
                        } else if(VALVE_DZ_timer == (int) (1.0f * (float) TMR_FREQ_5k)) {
                            START_POS = pos.sen;
                        } else if(VALVE_DZ_timer < (int) (2.0f * (float) TMR_FREQ_5k)) {
//                            valve_pos_raw.ref = Ref_Valve_Pos_Old  - DZ_case * DZ_DIRECTION * 64 / DZ_index;
                            valve_pos_raw.ref = Ref_Valve_Pos_Old  - DZ_case * DZ_DIRECTION * 2048 / DZ_index;
                            if(valve_pos_raw.ref <= VALVE_MIN_POS) {
                                valve_pos_raw.ref = VALVE_MIN_POS;
                            } else if(valve_pos_raw.ref >= VALVE_MAX_POS) {
                                valve_pos_raw.ref = VALVE_MAX_POS;
                            }
                            VALVE_POS_CONTROL(valve_pos_raw.ref);

                        } else if(VALVE_DZ_timer == (int) (2.0f * (float) TMR_FREQ_5k)) {
                            Ref_Valve_Pos_Old = valve_pos_raw.ref;
                            FINAL_POS = pos.sen;

                            if((FINAL_POS - START_POS)>1) {
                                DZ_DIRECTION = 1 * DZ_case;
                            } else if((FINAL_POS - START_POS)<-1) {
                                DZ_DIRECTION = -1 * DZ_case;
                            } else {
                                DZ_DIRECTION = 1 * DZ_case;
                            }

                            VALVE_DZ_timer = 0;
                            DZ_index= DZ_index *2;
//                            if(DZ_index >= 128) {
                            if(DZ_index >= 4096) {
                                FIRST_DZ = valve_pos_raw.ref;
                                DZ_NUM = 2;
                                Ref_Valve_Pos_Old = FIRST_DZ;
                                DZ_index = 1;
                                DZ_DIRECTION = 1;
                            }
                        }
                    } else if((DZ_case == -1 && DZ_NUM == 2) | (DZ_case == 1 && DZ_NUM == 2)) {
                        if(VALVE_DZ_timer < (int) (1.0f * (float) TMR_FREQ_5k)) {
                            Vout.ref = (float) P_GAIN_JOINT_POSITION * (0.5f * (float) pos_plus_end + 0.5f * (float) pos_minus_end - (float) pos.sen)*(float) 50.0f;
                        } else if(VALVE_DZ_timer == (int) (1.0f * (float) TMR_FREQ_5k)) {
                            START_POS = pos.sen;
                        } else if(VALVE_DZ_timer < (int) (2.0f * (float) TMR_FREQ_5k)) {
//                            valve_pos_raw.ref = Ref_Valve_Pos_Old  - DZ_case * DZ_DIRECTION * 64 / DZ_index;
                            valve_pos_raw.ref = Ref_Valve_Pos_Old  - DZ_case * DZ_DIRECTION * 2048 / DZ_index;
                            if(valve_pos_raw.ref <= VALVE_MIN_POS) {
                                valve_pos_raw.ref = VALVE_MIN_POS;
                            } else if(valve_pos_raw.ref >= VALVE_MAX_POS) {
                                valve_pos_raw.ref = VALVE_MAX_POS;
                            }
                            VALVE_POS_CONTROL(valve_pos_raw.ref);

                        } else if(VALVE_DZ_timer == (int) (2.0f * (float) TMR_FREQ_5k)) {
                            Vout.ref = 0.0f;
                        } else if(VALVE_DZ_timer > (int) (2.0f * (float) TMR_FREQ_5k)) {
                            Ref_Valve_Pos_Old = valve_pos_raw.ref;
                            FINAL_POS = pos.sen;

                            if((FINAL_POS - START_POS)>1) {
                                DZ_DIRECTION = 1 * DZ_case;
                            } else if((FINAL_POS - START_POS)<-1) {
                                DZ_DIRECTION = -1 * DZ_case;
                            } else {
                                DZ_DIRECTION = -1 * DZ_case;
                            }

                            VALVE_DZ_timer = 0;
                            DZ_index= DZ_index * 2;
//                            if(DZ_index >= 128) {
                            if(DZ_index >= 4096) {
                                SECOND_DZ = valve_pos_raw.ref;
                                VALVE_CENTER = (int) (0.5f * (float) (FIRST_DZ) + 0.5f * (float) (SECOND_DZ));
                                first_check = 0;
                                VALVE_DEADZONE_MINUS = (float) FIRST_DZ;
                                VALVE_DEADZONE_PLUS = (float) SECOND_DZ;

                                spi_eeprom_write(RID_VALVE_CNETER, (int16_t) VALVE_CENTER);
                                spi_eeprom_write(RID_VALVE_DEADZONE_PLUS, (int16_t) VALVE_DEADZONE_PLUS);
                                spi_eeprom_write(RID_VALVE_DEADZONE_MINUS, (int16_t) VALVE_DEADZONE_MINUS);

                                CONTROL_UTILITY_MODE = MODE_NO_ACT;
                                DZ_index = 1;
                            }
                        }
                    } else if(DZ_case == 0 && DZ_NUM ==1) {
                        if(VALVE_DZ_timer < (int) (1.0f * (float) TMR_FREQ_5k)) {
                            Vout.ref = (float) P_GAIN_JOINT_POSITION * (0.5f * (float) pos_plus_end + 0.5f * (float) pos_minus_end - (float) pos.sen)*(float) 50.0f;
                        } else if(VALVE_DZ_timer == (int) (1.0f * (float) TMR_FREQ_5k)) {
                            START_POS = pos.sen;
                        } else if(VALVE_DZ_timer < (int) (2.0f * (float) TMR_FREQ_5k)) {
//                            valve_pos_raw.ref = Ref_Valve_Pos_Old  - DZ_DIRECTION * 64 / DZ_index;
                            valve_pos_raw.ref = Ref_Valve_Pos_Old  - DZ_DIRECTION * 2048 / DZ_index;
                            if(valve_pos_raw.ref <= VALVE_MIN_POS) {
                                valve_pos_raw.ref = VALVE_MIN_POS;
                            } else if(valve_pos_raw.ref >= VALVE_MAX_POS) {
                                valve_pos_raw.ref = VALVE_MAX_POS;
                            }
                            VALVE_POS_CONTROL(valve_pos_raw.ref);

                        } else if(VALVE_DZ_timer == (int) (2.0f * (float) TMR_FREQ_5k)) {
                            Ref_Valve_Pos_Old = valve_pos_raw.ref;
                            FINAL_POS = pos.sen;

                            if((FINAL_POS - START_POS)>1) {
                                DZ_DIRECTION = 1;
                            } else if((FINAL_POS - START_POS)<-1) {
                                DZ_DIRECTION = -1;
                            } else {
                                DZ_DIRECTION = 1;
                            }
                            VALVE_DZ_timer = 0;
                            DZ_index= DZ_index *2;
//                            if(DZ_index >= 128) {
                            if(DZ_index >= 4096) {
                                FIRST_DZ = valve_pos_raw.ref;
                                DZ_NUM = 2;
                                Ref_Valve_Pos_Old = FIRST_DZ;
                                DZ_index = 1;
                                DZ_DIRECTION = 1;
                            }
                        }
                    } else {
                        if(VALVE_DZ_timer < (int) (1.0f * (float) TMR_FREQ_5k)) {
                            Vout.ref = (float) P_GAIN_JOINT_POSITION * (0.5f * (float) pos_plus_end + 0.5f * (float) pos_minus_end - (float) pos.sen)*(float) 50.0f;
                        } else if(VALVE_DZ_timer == (int) (1.0f * (float) TMR_FREQ_5k)) {
                            START_POS = pos.sen;
                        } else if(VALVE_DZ_timer < (int) (2.0f * (float) TMR_FREQ_5k)) {
//                            valve_pos_raw.ref = Ref_Valve_Pos_Old  + DZ_DIRECTION * 64 / DZ_index;
                            valve_pos_raw.ref = Ref_Valve_Pos_Old  + DZ_DIRECTION * 2048 / DZ_index;
                            if(valve_pos_raw.ref <= VALVE_MIN_POS) {
                                valve_pos_raw.ref = VALVE_MIN_POS;
                            } else if(valve_pos_raw.ref > VALVE_MAX_POS) {
                                valve_pos_raw.ref = VALVE_MAX_POS;
                            }
                            VALVE_POS_CONTROL(valve_pos_raw.ref);

                        } else if(VALVE_DZ_timer == (int) (2.0f * (float) TMR_FREQ_5k)) {
                            Vout.ref = 0.0f;
                        } else if(VALVE_DZ_timer > (int) (2.0f * (float) TMR_FREQ_5k)) {
                            Ref_Valve_Pos_Old = valve_pos_raw.ref;
                            FINAL_POS = pos.sen;

                            if((FINAL_POS - START_POS)>1) {
                                DZ_DIRECTION = -1;
                            } else if((FINAL_POS - START_POS)<-1) {
                                DZ_DIRECTION = 1;
                            } else {
                                DZ_DIRECTION = 1;
                            }

                            VALVE_DZ_timer = 0;
                            DZ_index= DZ_index *2;
//                            if(DZ_index >= 128) {
                            if(DZ_index >= 4096) {
                                SECOND_DZ = valve_pos_raw.ref;
                                VALVE_CENTER = (int) (0.5f * (float) (FIRST_DZ) + 0.5f * (float) (SECOND_DZ));
                                first_check = 0;
                                VALVE_DEADZONE_MINUS = (float) FIRST_DZ;
                                VALVE_DEADZONE_PLUS = (float) SECOND_DZ;

                                spi_eeprom_write(RID_VALVE_CNETER, (int16_t) VALVE_CENTER);
                                spi_eeprom_write(RID_VALVE_DEADZONE_PLUS, (int16_t) VALVE_DEADZONE_PLUS);
                                spi_eeprom_write(RID_VALVE_DEADZONE_MINUS, (int16_t) VALVE_DEADZONE_MINUS);

                                CONTROL_UTILITY_MODE = MODE_NO_ACT;
                                DZ_index = 1;
                            }
                        }
                    }
                }
                break;
            }

            case MODE_STEP_TEST: {
                float valve_pos_ref = 0.0f;
                if (cnt_step_test < (int) (1.0f * (float) TMR_FREQ_5k)) {
                    valve_pos_ref = 0.0f;
                } else {
                    valve_pos_ref = 10000.0f;
                }
                if(valve_pos_ref >= 0) {
                    valve_pos_raw.ref = (float)VALVE_ELECTRIC_CENTER + (float)valve_pos_ref * ((float)VALVE_MAX_POS-(float)VALVE_ELECTRIC_CENTER)/10000.0f;
                } else {
                    valve_pos_raw.ref = (float)VALVE_ELECTRIC_CENTER - (float)valve_pos_ref * ((float)VALVE_MIN_POS-(float)VALVE_ELECTRIC_CENTER)/10000.0f;
                }

                VALVE_POS_CONTROL_DZ(valve_pos_raw.ref);

                ref_array[cnt_step_test] = valve_pos_ref;
                if(valve_pos.sen >= (float) VALVE_ELECTRIC_CENTER) {
                    pos_array[cnt_step_test] = 10000.0f*((float)valve_pos.sen - (float)VALVE_ELECTRIC_CENTER)/((float)VALVE_MAX_POS - (float)VALVE_ELECTRIC_CENTER);
                } else {
                    pos_array[cnt_step_test] = -10000.0f*((float)valve_pos.sen - (float)VALVE_ELECTRIC_CENTER)/((float)VALVE_MIN_POS - (float)VALVE_ELECTRIC_CENTER);
                }

                CONTROL_MODE = MODE_VALVE_OPEN_LOOP;
                cnt_step_test++;
                if (cnt_step_test > (int) (2.0f * (float) TMR_FREQ_5k)) {
                    buffer_data_size = cnt_step_test;
                    cnt_step_test = 0;
                    cnt_send_buffer = 0;
                    CONTROL_UTILITY_MODE = MODE_SEND_OVER;
                    CONTROL_MODE = MODE_NO_ACT;
                }

                break;
            }
            case MODE_SEND_OVER: {
                CONTROL_UTILITY_MODE = MODE_NO_ACT;
                CONTROL_MODE = MODE_NO_ACT;
                break;
            }

            case MODE_FREQ_TEST: {
                float valve_pos_ref = 2500.0f * sin(2.0f * 3.141592f * freq_test_valve_ref * (float) cnt_freq_test * DT_TMR2);
                if(valve_pos_ref >= 0) {
                    valve_pos_raw.ref = (float)VALVE_ELECTRIC_CENTER + (float)valve_pos_ref * ((float)VALVE_MAX_POS-(float)VALVE_ELECTRIC_CENTER)/10000.0f;
                } else {
                    valve_pos_raw.ref = (double)VALVE_ELECTRIC_CENTER - (float)valve_pos_ref * ((float)VALVE_MIN_POS-(float)VALVE_ELECTRIC_CENTER)/10000.0f;
                }

                VALVE_POS_CONTROL_DZ(valve_pos_raw.ref);

                ref_array[cnt_freq_test] = valve_pos_ref;
//                if(value>=(float) VALVE_ELECTRIC_CENTER) {
                if(valve_pos.sen>=(float) VALVE_ELECTRIC_CENTER) {
//                    pos_array[cnt_freq_test] = 10000.0f*((float)value - (float)VALVE_ELECTRIC_CENTER)/((float)VALVE_MAX_POS - (float)VALVE_ELECTRIC_CENTER);
                    pos_array[cnt_freq_test] = 10000.0f*((float)valve_pos.sen - (float)VALVE_ELECTRIC_CENTER)/((float)VALVE_MAX_POS - (float)VALVE_ELECTRIC_CENTER);
                } else {
//                    pos_array[cnt_freq_test] = -10000.0f*((float)value - (float)VALVE_ELECTRIC_CENTER)/((float)VALVE_MIN_POS - (float)VALVE_ELECTRIC_CENTER);
                    pos_array[cnt_freq_test] = -10000.0f*((float)valve_pos.sen - (float)VALVE_ELECTRIC_CENTER)/((float)VALVE_MIN_POS - (float)VALVE_ELECTRIC_CENTER);
                }

                CONTROL_MODE = MODE_VALVE_OPEN_LOOP;
                cnt_freq_test++;
                if (freq_test_valve_ref * (float) cnt_freq_test * DT_TMR2 > 2) {
                    buffer_data_size = cnt_freq_test;
                    cnt_freq_test = 0;
                    cnt_send_buffer = 0;
                    freq_test_valve_ref = freq_test_valve_ref * 1.05f;
                    if (freq_test_valve_ref >= 400) {
                        CONTROL_UTILITY_MODE = MODE_NO_ACT;
                        CONTROL_MODE = MODE_NO_ACT;
                    }
                    CONTROL_MODE = MODE_NO_ACT;
                    CONTROL_UTILITY_MODE = MODE_SEND_OVER;

                }
                break;
            }

            default:
                break;
        }

        // CONTROL MODE ------------------------------------------------------------
        switch (CONTROL_MODE) {
            case MODE_NO_ACT: {
                V_out = 0.0f;
                break;
            }

            case MODE_VALVE_POSITION_CONTROL: {
                if (OPERATING_MODE == 5) { //SW Valve
                    VALVE_POS_CONTROL_DZ(valve_pos.ref);
                    V_out = Vout.ref;
                } else if (CURRENT_CONTROL_MODE == 0) { //PWM
                    I_REF = valve_pos.ref;
                } else {
                    I_REF = valve_pos.ref * 0.001f; // Unit : pulse >> mA
                    float I_MAX = 10.0f; // Max : 10mA
                    if (I_REF > I_MAX) {
                        I_REF = I_MAX;
                    } else if (I_REF < -I_MAX) {
                        I_REF = -I_MAX;
                    }
                }
                break;
            }

            case MODE_JOINT_CONTROL: {

                float temp_vel_pos = 0.0f; // desired velocity for position control
                float temp_vel_FT = 0.0f; // desired velocity for force/torque control
                float temp_vel_ff = 0.0f; // desired velocity for feedforward control
                float temp_vel = 0.0f;

                float wn_Pos = 2.0f * PI * 5.0f; // f_cut : 5Hz Position Control

                pos.err = pos.ref - pos.sen; // Unit : mm or deg
                vel.err = vel.ref - vel.sen; // Unit : mm/s or deg/s

                // position control command ===============================================================================================================================================
                if ((OPERATING_MODE & 0b01) == 0) { // Rotary Mode
                    temp_vel_pos = 0.1f * (P_GAIN_JOINT_POSITION * wn_Pos * pos.err) * PI / 180.0f; // rad/s
                    //                            L when P-gain = 100, f_cut = 10Hz
                } else {
                    temp_vel_pos = 0.1f * (P_GAIN_JOINT_POSITION * wn_Pos * pos.err); // mm/s
                    //                            L when P-gain = 100, f_cut = 10Hz
                }

                // torque control command ===============================================================================================================================================
                float alpha_SpringDamper = 1.0f/(1.0f+TMR_FREQ_5k/(2.0f*PI*30.0f));
                K_LPF = (1.0f-alpha_SpringDamper) * K_LPF + alpha_SpringDamper * K_SPRING;
                D_LPF = (1.0f-alpha_SpringDamper) * D_LPF + alpha_SpringDamper * D_DAMPER;

                if ((OPERATING_MODE & 0b01) == 0) { // Rotary Mode
                    float torq_ref_act = torq.ref + K_LPF * pos.err + D_LPF * vel.err; // unit : Nm
                    torq.err = torq_ref_act - torq.sen;
                    if (torq.err > 10.0f || torq.err < -10.0f) {
                        torq.err_int += torq.err/((float)TMR_FREQ_5k);
                    }
                    temp_vel_FT = 0.01f * (P_GAIN_JOINT_TORQUE * torq.err + I_GAIN_JOINT_TORQUE * torq.err_int); // Nm >> rad/s
                } else {
                    float force_ref_act = force.ref + K_LPF * pos.err + D_LPF * vel.err; // unit : N
                    //////////////////////////////////////////////////force_reference_filter////////////////////////////////////////////////////////////////////
//                    float alpha_torque_ref = 1.0f/(1.0f+TMR_FREQ_5k/(2.0f*PI*1.0f));
//                    force_ref_filter = (1.0f-alpha_torque_ref) * force_ref_filter + alpha_torque_ref * force_ref_act;
                    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                    force.err = force_ref_act - force.sen;
                    if (force.err > 10.0f || force.err < -10.0f) {
                        force.err_int += force.err/((float)TMR_FREQ_5k);
                    }
                    temp_vel_FT = 0.01f * (P_GAIN_JOINT_TORQUE * force.err + I_GAIN_JOINT_TORQUE * force.err_int); // N >> mm/s
                }


                // velocity feedforward command ========================================================================================================================================
                if ((OPERATING_MODE & 0b01) == 0) { // Rotary Mode
                    temp_vel_ff = 0.01f * (float)VELOCITY_COMP_GAIN * vel.ref * PI / 180.0f; // rad/s
                } else {
                    temp_vel_ff = 0.01f * (float)VELOCITY_COMP_GAIN * vel.ref; // mm/s
                }

                // command integration =================================================================================================================================================
                temp_vel = (1.0f - alpha_trans) * temp_vel_pos  + alpha_trans * temp_vel_FT + temp_vel_ff; // Position Control + Torque Control + Velocity Feedforward

                float Qact = 0.0f; // required flow rate
                float valve_pos_pulse = 0.0f;
                if( temp_vel > 0.0f ) {
                    Qact = temp_vel * ((float)PISTON_AREA_A * 0.00006f); // mm^3/sec >> LPM
                    if (((OPERATING_MODE&0b110)>>1) == 0 || ((OPERATING_MODE&0b110)>>1) == 1) { //Moog Valve or KNR Valve
                        I_REF = tanh_inv(Qact/(K_v * sqrt(PRES_SUPPLY * alpha3 / (alpha3 + 1.0f))))/C_d;
//                        I_REF = Qact*2.0f;
                    } else {    // SW valve
                        valve_pos_pulse = Qact / (C_d * sqrt(PRES_SUPPLY * alpha3 / (alpha3 + 1.0f)));
                    }
                } else {
                    Qact = temp_vel * ((float)PISTON_AREA_B * 0.00006f); // mm^3/sec >> LPM
                    if (((OPERATING_MODE&0b110)>>1) == 0 || ((OPERATING_MODE&0b110)>>1) == 1) { //Moog Valve or KNR Valve
                        I_REF = tanh_inv(Qact/(K_v * sqrt(PRES_SUPPLY / (alpha3 + 1.0f))))/C_d;
//                        I_REF = Qact*2.0f;
                    } else {    // SW valve
                        valve_pos_pulse = Qact / (C_d * sqrt(PRES_SUPPLY / (alpha3 + 1.0f)));
                    }
                }

                if (((OPERATING_MODE&0b110)>>1) == 0 || ((OPERATING_MODE&0b110)>>1) == 1) { //Moog Valve or KNR Valve

                    float I_MAX = 10.0f; // Maximum Current : 10mA
                    // Anti-windup for FT
                    if (I_GAIN_JOINT_TORQUE > 0.001f) {
                        float Ka = 2.0f;
                        if (I_REF > I_MAX) {
                            float I_rem = I_REF - I_MAX;
                            I_REF = I_MAX;
                            float temp_vel_rem = K_v * sqrt(PRES_SUPPLY * alpha3 / (alpha3 + 1.0f)) * tanh(C_d*I_rem) / ((double) PISTON_AREA_A * 0.00006f); // Unit : mm/s [linear] / rad/s [rotary]
                            torq.err_int = torq.err_int - Ka * temp_vel_rem * (10000.0f/I_GAIN_JOINT_TORQUE);
                        } else if (I_REF < -I_MAX) {
                            double I_rem = I_REF - (-I_MAX);
                            I_REF = -I_MAX;
                            float temp_vel_rem = K_v * sqrt(PRES_SUPPLY / (alpha3 + 1.0f)) * tanh(C_d*I_rem) / ((double) PISTON_AREA_B * 0.00006f); // Unit : mm/s [linear] / rad/s [rotary]
                            torq.err_int = torq.err_int - Ka * temp_vel_rem * (10000.0f/I_GAIN_JOINT_TORQUE);
                        }
                    } else {
                        if(I_REF > I_MAX) {
                            I_REF = I_MAX;
                        } else if (I_REF < -I_MAX) {
                            I_REF = -I_MAX;
                        }
                    }
                } else {    //SW valve
                    float Valve_pos_MAX = 10000.0f; // Maximum Valve Pos : 10000
                    // Anti-windup for FT
                    if (I_GAIN_JOINT_TORQUE > 0.001f) {
                        float Ka = 2.0f;
                        if (valve_pos_pulse > Valve_pos_MAX) {
                            float valve_pos_rem = valve_pos_pulse - Valve_pos_MAX;
                            valve_pos_pulse = Valve_pos_MAX;
                            float temp_vel_rem = C_d * valve_pos_rem * sqrt(PRES_SUPPLY * alpha3 / (alpha3 + 1.0f)) / ((double) PISTON_AREA_A * 0.00006f); // Unit : mm/s [linear] / rad/s [rotary]
                            torq.err_int = torq.err_int - Ka * temp_vel_rem * (10000.0f/I_GAIN_JOINT_TORQUE);
                        } else if (valve_pos_pulse < -Valve_pos_MAX) {
                            double valve_pos_rem = valve_pos_pulse - (-Valve_pos_MAX);
                            valve_pos_pulse = -Valve_pos_MAX;
                            float temp_vel_rem = C_d * valve_pos_rem * sqrt(PRES_SUPPLY / (alpha3 + 1.0f)) / ((double) PISTON_AREA_B * 0.00006f); // Unit : mm/s [linear] / rad/s [rotary]
                            torq.err_int = torq.err_int - Ka * temp_vel_rem * (10000.0f/I_GAIN_JOINT_TORQUE);
                        }
                    } else {
                        if(valve_pos_pulse > 10000.0f) {
                            valve_pos_pulse = 10000.0f;
                        } else if(valve_pos_pulse < -10000.0f) {
                            valve_pos_pulse = -10000.0f;
                        }
                    }
//                    float valve_control_deadzone = 5.0f;
//                    if(valve_pos_pulse < valve_control_deadzone && valve_pos_pulse > -valve_control_deadzone) {
//                        valve_pos.ref = ((float) VALVE_DEADZONE_PLUS + (float) VALVE_DEADZONE_MINUS) * 0.5f;
//                    } else if (valve_pos_pulse >= valve_control_deadzone) {
//                        valve_pos.ref = valve_pos_pulse/10000.0f * ((float) VALVE_MAX_POS - (float) VALVE_DEADZONE_PLUS) + (float) VALVE_DEADZONE_PLUS;
//                    } else {
//                        valve_pos.ref = -valve_pos_pulse/10000.0f * ((float) VALVE_MIN_POS - (float) VALVE_DEADZONE_MINUS) + (float) VALVE_DEADZONE_MINUS;
//                    }
                    
                    if (valve_pos_pulse >= 0) {
                        valve_pos.ref = valve_pos_pulse/10000.0f * ((float) VALVE_MAX_POS - (float) VALVE_DEADZONE_PLUS) + (float) VALVE_DEADZONE_PLUS;
                    } else {
                        valve_pos.ref = -valve_pos_pulse/10000.0f * ((float) VALVE_MIN_POS - (float) VALVE_DEADZONE_MINUS) + (float) VALVE_DEADZONE_MINUS;
                    } 

                    VALVE_POS_CONTROL_DZ(valve_pos.ref);
                    V_out = Vout.ref;
                }
                break;
            }

            case MODE_VALVE_OPEN_LOOP: {
                V_out = (float) Vout.ref;
                break;
            }

            default:
                break;
        }


        if (((OPERATING_MODE&0b110)>>1) == 0 || ((OPERATING_MODE&0b110)>>1) == 1) { //Moog Valve or KNR Valve

            ////////////////////////////////////////////////////////////////////////////
            ////////////////////////////  CURRENT CONTROL //////////////////////////////
            ////////////////////////////////////////////////////////////////////////////
            if (CURRENT_CONTROL_MODE) {
                double alpha_update_Iref = 1.0f / (1.0f + 5000.0f / (2.0f * 3.14f * 300.0f)); // f_cutoff : 500Hz
                I_REF_fil = (1.0f - alpha_update_Iref) * I_REF_fil + alpha_update_Iref*I_REF;

                if (I_REF_fil > 0.0f) I_REF_fil_DZ = I_REF_fil + (double)VALVE_DEADZONE_PLUS*mA_PER_pulse; // unit: mA
                else if (I_REF_fil < 0.0f) I_REF_fil_DZ = I_REF_fil + (double)VALVE_DEADZONE_MINUS*mA_PER_pulse; // unit: mA
                else I_REF_fil_DZ = I_REF_fil + (double)(VALVE_DEADZONE_PLUS+VALVE_DEADZONE_MINUS)/2.0f*mA_PER_pulse; // unit: mA

                I_ERR = I_REF_fil_DZ - (double)cur.sen;
                I_ERR_INT = I_ERR_INT + (I_ERR) * 0.0002f;

                // Moog Valve Current Control Gain
                double R_model = 500.0f; // ohm
                double L_model = 1.2f;
                double w0 = 2.0f * 3.14f * 100.0f;
                double KP_I = 0.1f * L_model*w0;
                double KI_I = 0.1f * R_model*w0;

                // KNR Valve Current Control Gain
                if (((OPERATING_MODE & 0b110)>>1) == 1) { // KNR Valve
                    R_model = 163.0f; // ohm
                    L_model = 1.0f;
                    w0 = 2.0f * 3.14f * 80.0f;
                    KP_I = 1.0f * L_model*w0;
                    KI_I = 0.08f * R_model*w0;
                }

                double FF_gain = 1.0f;
                VALVE_PWM_RAW = KP_I * 2.0f * I_ERR + KI_I * 2.0f* I_ERR_INT;
                I_REF_fil_diff = I_REF_fil_DZ - I_REF_fil_old;
                I_REF_fil_old = I_REF_fil_DZ;
//                VALVE_PWM_RAW = VALVE_PWM_RAW + FF_gain * (R_model * I_REF_fil + L_model * I_REF_fil_diff * 5000.0f); // Unit : mV
                VALVE_PWM_RAW = VALVE_PWM_RAW + FF_gain * (R_model * I_REF_fil_DZ); // Unit : mV
                double V_MAX = VALVE_VOLTAGE_LIMIT*1000.0f; // Maximum Voltage : 12V = 12000mV

                double Ka = 3.0f / KP_I;
                if (VALVE_PWM_RAW > V_MAX) {
                    V_rem = VALVE_PWM_RAW - V_MAX;
                    V_rem = Ka*V_rem;
                    VALVE_PWM_RAW = V_MAX;
//                    I_ERR_INT = I_ERR_INT - V_rem * 0.0002f;
                    I_ERR_INT = I_ERR_INT - V_rem;
                } else if (VALVE_PWM_RAW < -V_MAX) {
                    V_rem = VALVE_PWM_RAW - (-V_MAX);
                    V_rem = Ka*V_rem;
                    VALVE_PWM_RAW = -V_MAX;
//                    I_ERR_INT = I_ERR_INT - V_rem * 0.0002f;
                    I_ERR_INT = I_ERR_INT - V_rem;
                }
            } else {
                VALVE_PWM_RAW = I_REF * mV_PER_mA;
            }

            ////////////////////////////////////////////////////////////////////////////
            /////////////////  Dead Zone Cancellation & Linearization //////////////////
            ////////////////////////////////////////////////////////////////////////////

            // Output Voltage Linearization
            double CUR_PWM_nonlin = (double)VALVE_PWM_RAW; // Unit : mV
            double CUR_PWM_lin = PWM_duty_byLT(CUR_PWM_nonlin);  // -8000~8000

            // Dead Zone Cancellation (Electrical dead-zone)
            if (CUR_PWM_lin > 0) V_out = (float) (CUR_PWM_lin + 169.0f);
            else if (CUR_PWM_lin < 0) V_out = (float) (CUR_PWM_lin - 174.0f);
            else V_out = (float) (CUR_PWM_lin);

        } else {            //////////////////////////sw valve
            // Output Voltage Linearization & Dead Zone Cancellation (Electrical dead-zone) by SW

            /////////////Jitter/////////////////////
            V_out = V_out + 3000.0f*sin(2.0f*3.14159f*500.0f*((float) cnt_jitter)*0.0002f);     //4000, 500Hz
            cnt_jitter++;
            if(cnt_jitter > 50000000) cnt_jitter = 0;

            //L6205D
            if (V_out > 0 ) V_out = 800.0f + V_out*1.1275f;
            else if (V_out < 0) V_out = -800.0f + V_out*1.1275f;
            else V_out = 0.0f;
        }

        ////////////////////////////////////////////////////////////////////
        ///////////////////  PWM Command ///////////////////////////////////
        ////////////////////////////////////////////////////////////////////

        if(DIR_VALVE<0) {
            V_out = -V_out;
        }

        if (V_out >= VALVE_VOLTAGE_LIMIT*1000.0f) {
            V_out = VALVE_VOLTAGE_LIMIT*1000.0f;
        } else if(V_out<=-VALVE_VOLTAGE_LIMIT*1000.0f) {
            V_out = -VALVE_VOLTAGE_LIMIT*1000.0f;
        }

        PWM_out= V_out/(SUPPLY_VOLTAGE*1000.0f);

        // Saturation of output voltage
        if(PWM_out > 1.0f) PWM_out=1.0f;
        else if (PWM_out < -1.0f) PWM_out=-1.0f;

        if (PWM_out>0.0f) {
            TIM1->CCR1 = (TMR1_COUNT)*(PWM_out);
            TIM1->CCR2 = (TMR1_COUNT)*(0.0f);
        } else {
            TIM1->CCR1 = (TMR1_COUNT)*(0.0f);
            TIM1->CCR2 = 0.0f -(TMR1_COUNT)*(PWM_out);
        }

        ////////////////////////////////////////////////////////////////////////////
        //////////////////////  Data transmission through CAN //////////////////////
        ////////////////////////////////////////////////////////////////////////////

//        if (TMR2_COUNT_CAN_TX % (int) ((int) TMR_FREQ_5k/CAN_FREQ) == 0) {
        if (TMR2_COUNT_CAN_TX % (int) ((int) TMR_FREQ_5k/500) == 0) {
            // Position, Velocity, and Torque (ID:1200)

            if (flag_data_request[0] == HIGH) {
                if ((OPERATING_MODE & 0b01) == 0) { // Rotary Actuator
                    CAN_TX_POSITION_FT((int16_t) (pos.sen*200.0f), (int16_t) (vel.sen*20.0f), (int16_t) (torq.sen*TORQUE_SENSOR_PULSE_PER_TORQUE*10.0f));
                } else if ((OPERATING_MODE & 0b01) == 1) { // Linear Actuator
                    CAN_TX_POSITION_FT((int16_t) (pos.sen*200.0f), (int16_t) (vel.sen*20.0f), (int16_t) (force.sen*TORQUE_SENSOR_PULSE_PER_TORQUE*10.0f));
                }
            }

            // Valve Position (ID:1300)
            if (flag_data_request[1] == HIGH) {
                if (((OPERATING_MODE&0b110)>>1) == 0 || ((OPERATING_MODE&0b110)>>1) == 1) { //Moog Valve or KNR Valve
                    CAN_TX_PWM((int16_t)(cur.sen/mA_PER_pulse));
                } else {
                    float valve_pos_can = 0.0f;
                    if(valve_pos.sen >= VALVE_ELECTRIC_CENTER) {
                        valve_pos_can = 10000.0f*((float)valve_pos.sen-(float)VALVE_ELECTRIC_CENTER)/((float)VALVE_MAX_POS-(float)VALVE_ELECTRIC_CENTER);
                    } else {
                        valve_pos_can = -10000.0f*((float)valve_pos.sen -(float)VALVE_ELECTRIC_CENTER)/((float)VALVE_MIN_POS-(float)VALVE_ELECTRIC_CENTER);
                    }
                    CAN_TX_PWM((int16_t)(valve_pos_can));
                }
            }

            // Others : SW (ID:1400)
            if (flag_data_request[2] == HIGH) {
                CAN_TX_CURRENT((int16_t) valve_pos.sen, (int16_t) valve_pos.ref);
            }

            TMR2_COUNT_CAN_TX = 0;
        }
        TMR2_COUNT_CAN_TX++;
        LED = 0;
    }
    TIM2->SR = 0x0;  // reset the status register
}
