/* mbed Si7005 relative humidity sensor Library
 *
 * Copyright (c) 2013, Lieven Hollevoet (http://likatronix.be)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef SI7005_H
#define SI7005_H

#include "mbed.h"

/*
 * register names
 */

#define SI7005_REG_STATUS  0
#define SI7005_REG_DATAH   1
#define SI7005_REG_DATAL   2
#define SI7005_REG_CONFIG  3
#define SI7005_REG_ID     17

/*
 * measurement types
 */

#define SI7005_RH 0
#define SI7005_T  1

/** Class to interface to a Silicon Labs Si7005 relative humidity sensor.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "Si7005.h"
 *
 * Si7005 rh(p28, p27);         // sda, scl
 * DigitalOut rh_cs_n(p26);     // Chip select signal
 * Serial uart(USBTX, USBRX);   // tx, rx
 
 * int main() {
 *
 *     // Enable RH sensor
 *     rh_cs_n = 0;
 *     // Let it startup!
 *     wait(0.05);
 *
 *     char id;
 *     id = rh.readID();
 *     uart.printf("Sensor type: %02X\n", id);
 
 *     // Relative humidity measurement
 *     rh.startMeasurement(SI7005_RH);
 *     while (!rh.conversionDone()) {
 *             wait(0.01);
 *     }
 *     int measurement = rh.readResult(SI7005_RH);
 *     uart.printf("RH = %i procent\n", measurement);
 *
 *       // Start temperature measurement
 *       rh.startMeasurement(SI7005_T);
 *       while (!rh.conversionDone()){
 *         wait (0.01);
 *       }
 *       measurement = rh.readResult(SI7005_T);
 *       uart.printf("Temp = %i degrees C\n", measurement);
 *
 *     // Disable the sensor
 *     rh_cs_n = 1;
 *
 * }
 * @endcode
 */
class Si7005 {

public:

    /** Creates the sensor interface instance
     * @param sda I2C data pin
     * @param scl I2C clock pin
     */
    Si7005(PinName sda, PinName scl);

    /** Read the chip ID
     * @returns The chip ID register contents. Should be in the format 0x5y for a Si7005 sensor where 'y' is the chip revision.
     */
    int readID (void);

    /** Start a measurement for relative humidity or temperature
     * Initiate a measurement. The type can be either SI7005_RH for a relative humidity measurement or SI7005_T for a temperature measurement.
     * This function starts the conversion in the sensor. Before reading out the
     * result, ensure the conversion is done by calling conversionDone().
     * @param type Either SI7005_RH for humidity measurement or SI7005_T for temperature
     */
    void startMeasurement (char type);

    /** Checks if the measurement is done.
     * Reports true if the conversion is finished. This is detected by reading the DONE bit in the STATUS register of the sensor.
     * @returns true if the conversion is done
     */
    bool conversionDone (void);

    /** Reads the measurement result.
     * @param type Either SI7005_RH for humidity measurement or SI7005_T for temperature
     * @returns The measurement result as specified in the datasheet, a 12-bit value for the relative humidity measurement and a 14-bit value for a temperature measurement.
     */
    int readResult (char type);

protected:

    void _write(int reg, int data);
    int _read(int reg);
    int _read_int(int reg);
    int _addr;
    I2C _i2c;

};


#endif

