/* mbed Si7005 relative humidity sensor Library
 *
 * Copyright (c) 2013, Lieven Hollevoet (http://likatronix.be)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


#include "Si7005.h"
#include "mbed.h"

/*
 Constructor, pin names for I2C of the device
 */
Si7005::Si7005(PinName scl, PinName sda)
        : _i2c(scl, sda) {

    _i2c.frequency(50000);
      _addr = 0x80;
}


int Si7005::readID(void) {
    int data;

    data = _read(SI7005_REG_ID);

    return data;

}

void Si7005::startMeasurement(char type) {
    char command;

    switch (type) {
    case SI7005_RH:
        command = 0x01; // Start relative humidity measurement
        break;
    case SI7005_T:
        command = 0x11; // Start temperature measurement
        break;
    default:
        return;
    }

    _write(SI7005_REG_CONFIG, command);

    return;
}

bool Si7005::conversionDone(void) {
    int result;
    result = _read(SI7005_REG_STATUS);

    if (result == 0) {
        return true;
    } else {
        return false;
    }

}

int Si7005::readResult(char type) {
    int result;
    float f_res;
    float f_lin;

    result = _read_int(SI7005_REG_DATAH);

    switch (type) {
    case SI7005_RH:
        // Keep upper 12 bits
        result = result >> 4;
        // Calculate %
        f_res = ((float)result / 16) - 24;
        // Linearize
        f_lin = f_res - ((f_res * f_res) * - 0.00393 + f_res * 0.4008 - 4.7844 );
        return (int)f_lin;
    case SI7005_T:
        // Keep upper 14 bits
        result = result >> 2;
        // Calculate %
        result = (result / 32) - 50;
        return result;
    default:
        return -1;
    }
}
// private functions for low level IO
// The Si7005 needs a restart bit when doing read accesses

void Si7005::_write(int reg, int data) {
    char args[2];
    args[0] = reg;
    args[1] = data;
    _i2c.write(_addr, args,2);
}

int Si7005::_read(int reg) {
    char args[2];
    args[0] = reg;
    _i2c.write(_addr, args, 1, true); // No stop condition here
    _i2c.read(_addr, args, 1, true);  // And a restart condition here
    _i2c.stop();
    return(args[0]);
}

int Si7005::_read_int(int reg) {
    char data[2];
    data[0] = reg;
    _i2c.write(_addr, data, 1, true); // No stop condition here
    _i2c.read(_addr, data, 2, true);  // And a restart condition here
    _i2c.stop();

    // Data is read out MSB first

    return ((data[0] << 8) + data[1]);

}

