//remove repetition
#ifndef MBED_JOYSTICK_H
#define MBED_JOYSTICK_H

//required to use mbed functions
#include "mbed.h"
#include "FunctionPointer.h"

#define JOYSTICK_NONE            0x00
#define JOYSTICK_UP              0x01
#define JOYSTICK_DOWN            0x02
#define JOYSTICK_LEFT            0x04
#define JOYSTICK_RIGHT           0x08
#define JOYSTICK_PRESS           0x10

/** Digital Joystick class
 *
 * Used for an digital joystick.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "Joystick.h"
 *
 * Joystick joystick(P2_3, P0_15, P2_4, P0_16, P0_17);
 *
 * int main() {
 *     joystick.getStatus();
 * }
 *
 * @endcode
 */

class Joystick {
    public:
        typedef enum {
            none = JOYSTICK_NONE,
            up = JOYSTICK_UP,
            down = JOYSTICK_DOWN,
            left = JOYSTICK_LEFT,
            right = JOYSTICK_RIGHT,
            press = JOYSTICK_PRESS
        } Status;

        /** Create a Joystick HID for using regular mbed pins
         *
         * @param up    Joystick Up
         * @param down  Joystick Down
         * @param left  Joystick Left
         * @param right Joystick Right
         * @param press Joystick Press
         */
        Joystick(PinName up, PinName down, PinName left, PinName right, PinName press);

        void functions(FunctionPointer *functionNone, FunctionPointer *functionUp, FunctionPointer *functionDown,
                FunctionPointer *functionLeft, FunctionPointer *functionRight, FunctionPointer *functionPress);

        /** Get status
         * Read the joystick status
         *
         * @code
         * switch (joystick.getStatus()) {
         *     case Joystick::none:
         *         break;
         *     case Joystick::up:
         *         break;
         *     case Joystick::down:
         *         break;
         *     case Joystick::left:
         *         break;
         *     case Joystick::right:
         *         break;
         *     case Joystick::press:
         *         break;
         *     default:
         *         break;
         * }
         * @endcode
         *
         * @returns A uint8_t values representing the bits
         */
        Status getStatus();

        void poll();

    private:

        /** Regular mbed pins bus
         */
        DigitalIn _up, _down, _left, _right, _press;

        FunctionPointer *_functionNone, *_functionUp, *_functionDown, *_functionLeft, *_functionRight, *_functionPress;
};

#endif
