#ifndef NR_NETWORK_H
#define NR_NETWORK_H

#include "arrival_board.h"
#include "departure_board.h"
#include "EthernetInterface.h"
#include <string>

/*
 * This project is distributed by Leigh Brooks under the MIT license.
 * 
 * MIT License
 * 
 * Copyright (c) 2016 Leigh Brooks
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

/** NR_Network_Conn
*   Used for establishing a connection to the Huxley NR SOAP proxy and to generate departure/arrivals data
*/
class NR_Network_Conn
{
private:
    void init(const char* address, const char* mask, const char* gateway);
    EthernetInterface conn;
    TCPSocketConnection socket;
    char _address[17];
    char _sub_mask[17];
    char _gateway[17];

public:
    /** Make a connection. The empty constructor creates a connection using DHCP.
    */
    NR_Network_Conn();
    /** Make a connection. This constructor creates a connection with the specified IP address, subnet and gateway
    * @param address the IP address of this device (in the format "192.168.1.101"
    * @param mask the subnet mask of this device (in the format "255.255.255.0"
    * @param gateway the gatway of this device (in the format "192.168.1.1"
    */
    NR_Network_Conn(const char* address, const char* mask, const char* gateway);
    /** Get the socket (an object of type TCPSocketConnection)
    */
    TCPSocketConnection GetSocket();
    /** Get the IP address of the current connection
    */
    char* GetIP();
    /** Establish a connection to the API
    */
    int Connect(const char* url);
    /** Disconnect from the API
    */
    void Disconnect();
    /** Request a list of arrivals from the API and format them into an Arrival_Board object.
    * @param code_stn the station code to get arrivals from
    * @param number the number of arrivals to retrieve
    * @return an Arrival_Board object containing all the arrivals to this station
    */
    Arrival_Board GetArrivals(const std::string& code_stn, const std::string& number);
    /** Request a list of departures from the API and format them into a Departure_Board object.
    * @param code_stn the station code to get departures from
    * @param number the number of departures to retrieve
    * @return a Departure_Board object containing all the departures from this station
    */
    Departure_Board GetDepartures(const std::string& code_stn, const std::string& number);
};

#endif