#include "departure_board.h"
#include "arrival_board.h"
#include "nr_network.h"
#include "mbed.h"
#include <string>
#include <vector>

#ifndef nullptr
#define nullptr 0 // nullptr is undefined by default...
#endif

using std::string;
using std::vector;

/*
 * Initialise the network connection with the required attributes. 
 */
void NR_Network_Conn::init(const char* address, const char* mask, const char* gateway)
{
    conn = EthernetInterface();
    
    if(address == nullptr) _address[0] = '\0'; else strcpy(_address, address);
    if(mask == nullptr) _sub_mask[0] = '\0'; else strcpy(_sub_mask, mask);
    if(gateway == nullptr) _gateway[0] = '\0'; else strcpy(_gateway, gateway);
}    

NR_Network_Conn::NR_Network_Conn()
{
    init(nullptr, nullptr, nullptr);    
}    

TCPSocketConnection NR_Network_Conn::GetSocket()
{
    return socket;
}    

Departure_Board NR_Network_Conn::GetDepartures(const string& code_stn, const string& number)
{
    const string command = "GET /departures/" + code_stn + "/" + number + "/ HTTP/1.1";
    
    vector<char> command_cstr (command.begin(), command.end());
    command_cstr.push_back('\0');
    
    socket.send_all(&command_cstr[0], sizeof(&command_cstr[0]) - 1); // Ignore the null-terminating character
    
    char buf[1000];
    string json;
    
    int ret;
    
    while(true) {
        ret = socket.receive(buf, sizeof(buf) - 1);
        if(ret <= 0){
            json.append(buf);
            break;
        }    
        buf[ret] = '\0';
        json.append(buf);
        memset(buf, '\0', 1000);
    }
    
    Departure_Board dep = depBoardFromJson(json);
    
    return dep;
}    

Arrival_Board NR_Network_Conn::GetArrivals(const string& code_stn, const string& number)
{
    const string command = "GET /arrivals/" + code_stn + "/" + number + "/ HTTP/1.1";
    
    vector<char> command_cstr (command.begin(), command.end());
    command_cstr.push_back('\0');
    
    socket.send_all(&command_cstr[0], sizeof(&command_cstr[0]) - 1); // Ignore the null-terminating character
    
    char buf[1000];
    string json;
    
    int ret;
    
    while(true) {
        ret = socket.receive(buf, sizeof(buf) - 1);
        if(ret <= 0){
            json.append(buf);
            break;
        }    
        buf[ret] = '\0';
        json.append(buf);
        memset(buf, '\0', 1000);
    }
    
    Arrival_Board arr = arrBoardFromJson(json);
    
    return arr;
}    

char* NR_Network_Conn::GetIP()
{
    return conn.getIPAddress();
}    

NR_Network_Conn::NR_Network_Conn(const char* address, const char* mask, const char* gateway) 
{
    init(address, mask, gateway);
}

/*
 * Connect to the Huxley (NR) API.
 * @param url URL of the Huxley implementation to be used
 */
int NR_Network_Conn::Connect(const char* url)
{
    Serial pc(USBTX, USBRX);
    pc.baud(115200);
    pc.printf("Initiating connection (2A)...\n");
    
    if((_address[0] == '\0') || (_sub_mask[0] == '\0') || (_gateway == '\0')) {
        pc.printf("IP Address is being set by DHCP.\n");
        if(conn.init() < 0) return -1;
    } else {
        pc.printf("IP Address is being set manually.\n");
        if(conn.init(_address, _sub_mask, _gateway) < 0) return -1;
    }
    
    pc.printf("Connecting (2B)...\n");
    
    if(int err = conn.connect() < 0) return err;
    
    pc.printf("Creating TCPSocketConnection (2C)...\n");
    
    socket = TCPSocketConnection();
    
    socket.connect(url, 80);
    
    return 0;
}

void NR_Network_Conn::Disconnect()
{
    socket.close();
    conn.disconnect();
}