/* Driver for the 2.7" EM027BS013 E-Paper display produced by Pervasive Displays.
 * Based off of the 2.7" EM027AS182 display driver produced by Peter Drescher, but with key differences to use the newer Pervasive Displays driver.
 * Copyright notice is below for original driver; this is distributed under the same license.
 * 
 * mbed library for 264*176 pixel 2.7 INCH E-PAPER DISPLAY from Pervasive Displays
 * Copyright (c) 2013 Peter Drescher - DC2PD
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef EAEPAPER_H
#define EAEPAPER_H

/**
 * Includes
 */
#include "GraphicsDisplay.h"
#include "EM027BS013.h"
#include "mbed.h"

// we have to double buffer the display  
#define EA_IMG_BUF_SZ (5808) // 264 x 176 / 8 = 5808

/** Draw mode
  * NORMAl
  * XOR set pixel by xor of the screen
  */
enum {NORMAL,XOR};

/** Bitmap
 */
struct Bitmap{
    int xSize;
    int ySize;
    int Byte_in_Line;
    char* data;
    };


/* color definitions   */
#define Black           0x0
#define White           0x1   

/**
* Main driver class to drive the EA027BS023 E-Paper display.
*/
class EaEpaper : public GraphicsDisplay
{
    public:
        /** EaEpaper constructor.
         * @param sec03_SpiSCK the SPI SCK pin
         * @param sec04_SpiMOSI the SPI MOSI pin
         * @param sec05_SpiMISO the SPI MISO pin
         * @param sec06_EpdCS the SPI chip select pin connected to the display
         * @param sec07_EpdBusy the pin connected to the busy pin on the display
         * @param sec08_EpdBorder the pin connected to the border pin on the display
         * @param sec09_I2cSCL the I2C SCL pin
         * @param sec10_I2cSDA the I2C SDA pin
         * @param sec11_FlashCS the Flash Chip Select pin connected to the display
         * @param sec12_EpdReset the reset pin connected to the display
         * @param sec13_EpdPanelOn the power pin connected to the display
         * @param sec14_EpdDischarge the discharge pin connected to the display
         */
        EaEpaper(PinName sec03_SpiSCK,
               PinName sec04_SpiMOSI,
               PinName sec05_SpiMISO,
               PinName sec06_EpdCS,
               PinName sec07_EpdBusy,
               PinName sec08_EpdBorder,
               PinName sec09_I2cSCL,
               PinName sec10_I2cSDA,
               PinName sec11_FlashCS,
               PinName sec12_EpdReset,
               PinName sec13_EpdPanelOn,
               PinName sec14_EpdDischarge,
               const char* name = "EPD");
               
    /** Get the width of the screen in pixel
      *
      * @param
      * @returns width of screen in pixel
      *
      */
    virtual int width();

    /** Get the height of the screen in pixel
     *
     * @returns height of screen in pixel
     *
     */
    virtual int height();
    
    
    /**
     * Clear the display
     */    
    void clear();
    
    /**
     * Write image buffer to display
     */    
    void write_disp(void);
    
    /**
     * set or reset a single pixel
     *
     * @param x horizontal position
     * @param y vertical position
     * @param color : 0 white, 1 black
     */ 
    virtual void pixel(int x, int y, int color);
    
    /** Fill the screen with white
     *
     */ 
    virtual void cls(void);
    
    /** draw a 1 pixel line
     *
     * @param x0,y0 start point
     * @param x1,y1 stop point
     * @param color  : 0 white, 1 black
     */  
    void line(int x0, int y0, int x1, int y1, int color);
    
    /** draw a rect
     *
     * @param x0,y0 top left corner
     * @param x1,y1 down right corner
     * @param color : 0 white, 1 black
     */    
    void rect(int x0, int y0, int x1, int y1, int color);
    
    /** draw a filled rect
     *
     * @param x0,y0 top left corner
     * @param x1,y1 down right corner
     * @param color : 0 white, 1 black
     */  
    void fillrect(int x0, int y0, int x1, int y1, int color);
    
    /** draw a circle
     *
     * @param x0,y0 center
     * @param r radius
     * @param color : 0 white, 1 black                                                          *
     */    
    void circle(int x0, int y0, int r, int color);
    
    /** draw a filled circle
     *
     * @param x0,y0 center
     * @param r radius
     * @param color : 0 white, 1 black                                                                 *
     */  
    void fillcircle(int x0, int y0, int r, int color);
    
    /** set drawing mode
     *  @param NORMAL : paint normal color, XOR : paint XOR of current pixels
     */
    void setmode(int mode);
    
    /** setup cursor position
     *
     * @param x x-position (top left)
     * @param y y-position 
     */   
    virtual void locate(int x, int y);
    
    /** calculate the max number of char in a line
     *
     * @returns max columns
     * depends on actual font size
     */ 
    virtual int columns();
    
    /** calculate the max number of columns
     *
     * @returns max column
     * depends on actual font size
     */   
    virtual int rows();
    
    /** put a char on the screen
     *
     * @param value char to print
     * @returns printed char
     */
    virtual int _putc(int value);
    
    /** paint a character on given position out of the active font to the screen buffer
     *
     * @param x x-position (top left)
     * @param y y-position 
     * @param c char code
     */
    virtual void character(int x, int y, int c);
    
    /** select the font to use
     *
     * @param f pointer to font array 
     *                                                                              
     *   font array can created with GLCD Font Creator from http://www.mikroe.com
     *   you have to add 4 parameter at the beginning of the font array to use: 
     *   - the number of byte / char
     *   - the vertial size in pixel
     *   - the horizontal size in pixel
     *   - the number of byte per vertical line
     *   you also have to change the array to char[]
     */  
    void set_font(unsigned char* f);
    
    /** print bitmap to buffer
      *
      * @param bm struct Bitmap in flash
      * @param x  x start
      * @param y  y start 
      *
      */
    void print_bm(Bitmap bm, int x, int y);
    
    /** write raw characters to the display
     * @param img array of raw image data
     */
    void drawImage(uint8_t* img);
    
    unsigned char* font;
    unsigned int draw_mode;
    
 private:
    
    EM027BS013 _epd;
    unsigned int char_x;
    unsigned int char_y;
    
};


#endif