#include "HL1606Stripe.h"
#include <iostream>

/*Constructeur*/
HL1606Stripe::HL1606Stripe(PinName mosi, PinName miso, PinName sclk, PinName latch, uint8_t numLEDs, uint8_t numColors) : m_numLEDs(numLEDs), m_numColors(numColors), m_SPI(mosi, miso, sclk), m_latchPin(latch)
{   
    m_SPI.format(8,3);          // 8 bits per frame, SPI mode 3
    m_SPI.frequency(450000);    // 450kHz freq
         
    /*Allocation en fonction du nombre de LEDs sur le bandeau*/
    m_redPWM = new uint8_t[m_numLEDs];
    m_greenPWM = new uint8_t[m_numLEDs];
    m_bluePWM = new uint8_t[m_numLEDs];
                
    /*Eteindre le bandeau de LEDs*/
    SwitchOffRGB();
    
    /*Mettre à jour les LEDs du bandeau toutes les 45ms*/
    m_tickerUpdate.attach(this, &HL1606Stripe::Update, 0.0045);
}

/*Destructeur*/
HL1606Stripe::~HL1606Stripe()
{
    m_tickerUpdate.detach();
    
    delete [] m_redPWM;
    delete [] m_greenPWM;
    delete [] m_bluePWM;
}

/*Mettre à jour le bandeau de LEDs*/
void HL1606Stripe::Update(void)
{    
    uint8_t i, data; 
    static int m_pwmCounter;

    /*Ecrire les données sur le bus SPI*/
    for (i = 0; i < m_numLEDs; i++) {
        
        data = 0x80;

        /*Calculer la couleur de la LED suivante*/
        if (m_pwmCounter < m_redPWM[i]){
            data |= 0x04;
        }

        if (m_pwmCounter < m_bluePWM[i]){
            data |= 0x10;
        }

        if (m_pwmCounter < m_greenPWM[i]){
            data |= 0x01;
        }

        m_SPI.write(data);
    }

    /*Incrémenter le compteur de PWM (soft)*/
    m_pwmCounter += 1;

    if (m_pwmCounter > 3){
        m_pwmCounter = 0;
    }

    /*Latch*/
    m_latchPin = 1;
    wait_us(1);
    m_latchPin = 0;   
}

/*Eteindre le bandeau de LEDs*/
void HL1606Stripe::SwitchOffRGB()
{
    FillRGB(OFF,OFF,OFF);
}

/*Set d'une LED sur le bandeau*/
void HL1606Stripe::setLED(uint8_t Color, uint8_t LEDx)
{  
    if(LEDx>m_numLEDs)
    {
        std::cerr << "Erreur lors du set d'une LED du bandeau : index error" << std::endl;   
        return;
    }
    if(Color>NUM_COLORS)
    {
        std::cerr << "Erreur lors du set d'une LED du bandeau : Num color error" << std::endl;   
        return;
    }
    switch(Color)
        {
            case BLUE:
                m_redPWM[LEDx] = OFF;
                m_greenPWM[LEDx] = OFF;
                m_bluePWM[LEDx] = LOW;
            break;
            
            case RED:
                m_redPWM[LEDx] = LOW;
                m_greenPWM[LEDx] = OFF;
                m_bluePWM[LEDx] = OFF;
            break;
            
            case GREEN:
                m_redPWM[LEDx] = OFF;
                m_greenPWM[LEDx] = LOW;
                m_bluePWM[LEDx] = OFF;
            break;
            
             case YELLOW:
                m_redPWM[LEDx] = LOW;
                m_greenPWM[LEDx] = LOW;
                m_bluePWM[LEDx] = OFF;
            break;
                        
             case CYAN:
                m_redPWM[LEDx] = OFF;
                m_greenPWM[LEDx] = MEDIUM;
                m_bluePWM[LEDx] = LOW;
            break;
                         
            /*OFF*/
            default:
                m_redPWM[LEDx] = OFF;
                m_greenPWM[LEDx] = OFF;
                m_bluePWM[LEDx] = OFF;
        }
}

/*Remplir aléatoirement le bandeau de LEDs*/
void HL1606Stripe::FillRandomlyRGB(uint8_t* randomColors)
{              
    for(int LEDx = 0; LEDx < m_numLEDs; LEDx++)
    {
        randomColors[LEDx] = rand()%(m_numColors);                
        setLED(randomColors[LEDx],LEDx);
    }
}

/*Remplir le bandeau de LEDs avec une couleur définie dans Colour_t*/
void HL1606Stripe::FillRGB(uint8_t color)
{
    if(color>NUM_COLORS)
    {
        std::cerr << "Erreur lors du set d'une LED du bandeau : Num color error" << std::endl;   
        return;
    }
    for (int LEDx = 0; LEDx < m_numLEDs; LEDx++) 
    {
        setLED(color,LEDx);
    }
}

/*Remplir le bandeau de LEDs avec des composantes RGB définie par l'utilisateur*/
void HL1606Stripe::FillRGB(uint8_t red, uint8_t green, uint8_t blue)
{
    for (int i = 0; i < m_numLEDs; i++) {
        m_redPWM[i] = red;
        m_greenPWM[i] = green;
        m_bluePWM[i] = blue;
    }
}

/*Remplir le bandeau de LEDs avec des couleurs définies par l'utilisateur*/
void HL1606Stripe::FillRGB(uint8_t* colors)
{
    for(int LEDx = 0; LEDx < m_numLEDs; LEDx++) {
        setLED(colors[LEDx],LEDx);
    }
}
