#include "MatchColors.h"
#define TIME_TO_WAIT 10

/*Constructeur*/
MatchColors::MatchColors()
{  
    uint8_t nbColors = 0;
    configSPI spi;
    configI2C i2c;
    PinName latchPin;
            
    /*Lecture des paramètres de jeu et de configuration des pins*/
    ConfigFile("/local/Config.txt", m_difficulty, m_nbleds, spi, i2c, latchPin);

    /*Instanciation du capteur de couleur*/
    m_colourSensor = new GroveColourSensor(i2c.sda, i2c.scl);
    m_colourSensor->powerUp();
        
    /*Définition du nombre de couleur max affichées en fonction de la difficulté sélectionnée*/
    switch(m_difficulty){
        case 0 :
            nbColors = 3 ;
            break;
        case 1 :
            nbColors = 4 ;
            break;
        case 2 : 
            nbColors = 5 ;
            break;
        default : 
            nbColors = 4 ;
            break;
    }
    
    /*Instanciation du bandeau de LEDs RGB*/
    m_stripe = new HL1606Stripe(spi.mosi, spi.miso, spi.sck, latchPin, m_nbleds, nbColors);
    
    /*Instanciation de la gestion de couleurs*/
    m_colormManager = new ColorManager();
    
    m_randomColors = new uint8_t[m_nbleds];
}

/*Destructeur*/
MatchColors::~MatchColors()
{
    m_timeoutInit.detach();
    m_timeoutPlay.detach();
    m_tickerResultat.detach();
    
    if(m_colourSensor != NULL)
        delete m_colourSensor;
    
    if(m_stripe != NULL)
        delete m_stripe;
}

/*Initialisation d'une partie*/
void MatchColors::InitGame()
{
    static bool stripeIsON = false;
    
    m_timeoutInit.detach();

    if(stripeIsON) {
        /*Masquer les couleurs au joueur*/
        stripeIsON = false;
        m_stripe->SwitchOffRGB();
        
        /*Lancer la série de match*/
        m_timeoutPlay.attach(this, &MatchColors::Play, 3);
    } else {
        /*Remplir le bandeau de leds avec des couleurs aléatoires*/
        m_stripe->FillRandomlyRGB(m_randomColors);
        stripeIsON = true;
        m_result = false;
        
        /*Lancer la seconde phase de l'initialisation pour masquer les couleurs*/
        m_timeoutInit.attach(this, &MatchColors::InitGame, TIME_TO_WAIT-2*m_difficulty);
    }
}

void MatchColors::Play()
{
    HSV hsv;
    RGB rgb;
    static uint8_t tour = 0;
    Colour_t color;

    /*Lire la couleur sélectionnée par le joueur*/
    m_colourSensor->readColours(rgb);
    m_colormManager->RGBtoHSV(rgb, hsv);
    color=m_colormManager->getColorFromHue(hsv.hue);

    /*Matcher la couleur sélectionnée avec la solution*/
    if(color == m_randomColors[tour]) {
        /*La couleur sélectionnée est bonne*/
        m_timeoutPlay.detach();
        m_stripe->setLED(color,tour);
        tour++;

        /*Est ce la fin de la manche? (tour = 8)*/
        if(tour == m_nbleds) {
            m_result = true;
            tour = 0;
            
            /*Afficher animation de victoire*/
            m_tickerResultat.attach(this, &MatchColors::ShowResult, 0.125);
        } else {          
            /*Tour suivant*/
            m_timeoutPlay.attach(this, &MatchColors::Play, 4);
        }
    } else {
        /*La couleur sélectionnée est fausse*/
        m_result = false;
        tour = 0;
        
        /*Afficher animation de défaite*/
        m_tickerResultat.attach(this, &MatchColors::ShowResult, 0.125);
    }
}

void MatchColors::ShowResult()
{
    static uint8_t counter = 0;
    static bool status = false;

    if(counter < 20) {
        if(status) {
            /*Animation de victoire ou de défaite*/
            if(m_result) {
                m_stripe->FillRGB(GREEN);
            } else {
                m_stripe->FillRGB(m_randomColors);
            }
        } else {
            m_stripe->SwitchOffRGB();
        }

        counter++;
        status = !status;
        
    } else {
        m_tickerResultat.detach();
        counter = 0;
        status = false;
        m_stripe->SwitchOffRGB();
        
        /*Lancer l'initialisation d'une nouvelle partie*/
        m_timeoutInit.attach(this, &MatchColors::InitGame, 2);
    }
}
