/*
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#ifndef __GROVE_COLOUR_SENSOR_HPP__
#define __GROVE_COLOUR_SENSOR_HPP__
 
#include "mbed.h"
 
/**
 * This module is based on the color sensor TCS3414CS with digital output over I2C.
 * http://www.seeedstudio.com/wiki/index.php?title=Twig_-_I2C_Color_Sensor_v0.9b
 */
#ifndef COLOR_H
#define COLOR_H
#include "Color.h"
#endif

class GroveColourSensor {
public:
 
    // Constructeur qui initialise la connexion I2C
    GroveColourSensor(PinName sda, PinName scl) : m_i2c(sda, scl) {
        /* empty*/
    }
 
    // Méthode pour allumer le capteur
    void powerUp(void) {
        static const char powerUpCommand[] = {0x80, 0x03};
        if (m_i2c.write((SEVEN_BIT_ADDRESS << 1), powerUpCommand, sizeof(powerUpCommand)) != 0) {
            std::cerr << "failed to power up the sensor" << std::endl;
        }
    }
    
    // Méthode pour éteindre le capteur
    void powerDown(void) {
        static const char powerDownCommand[] = {0x80, 0x00};
        /* turn down the color sensor */
        if (m_i2c.write((SEVEN_BIT_ADDRESS << 1), powerDownCommand, sizeof(powerDownCommand)) != 0) {
            std::cerr << "failed to power down the sensor" << std::endl;
        }
    }

    // Méthode permettant de récupérer la valeur d'une couleur en donnant en paramètre une couleur de l'enum Colour_t 
    uint16_t readColour(Colour_t colour) {
        char readColorRegistersCommand = 0xb0 + (2 * static_cast<unsigned>(colour));
        m_i2c.write((SEVEN_BIT_ADDRESS << 1), &readColorRegistersCommand, 1 /* size */);
 
        uint16_t colourValue;
        m_i2c.read((SEVEN_BIT_ADDRESS << 1), reinterpret_cast<char *>(&colourValue), sizeof(uint16_t));
        return colourValue;
    }
 
    // Méthode permettant de récupérer la valeur d'une couleur en donnant en paramètre le numéro de la couleur dans l'énum 
    uint16_t readColour(unsigned colour) {
        if (colour >= NUM_COLORS) {
            return 0;
        } 
        return readColour(static_cast<Colour_t>(colour));
    }

    // Méthode permettant de récupérer les trois couleurs en donnant en paramètre une structure RGB (par référence)
    void readColours(RGB &rgb) {
        rgb.red= readColour(RED);
        rgb.blue= readColour(BLUE);
        rgb.green= readColour(GREEN);
    }
 
private:
    static const uint8_t SEVEN_BIT_ADDRESS = 0x39;
    I2C m_i2c;
};
#endif /* #ifndef __GROVE_COLOUR_SENSOR_HPP__ */
