/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed.h"
#include "cc3000.h"
#include "main.h"
#include "xively.h"
#include "xi_err.h"

#define XI_FEED_ID 123 // set Xively Feed ID (numerical, no quoutes)
#define XI_API_KEY "apikey" // set Xively API key (double-quoted string) 

using namespace mbed_cc3000;

DigitalOut myled(LED1);
Ticker flipper_seconds;
Ticker flipper_led;
uint32_t time_seconds;

/* cc3000 module declaration specific for user's board. Check also init() */
#if (MY_BOARD == WIGO)
cc3000 wifi(PTA16, PTA13, PTD0, SPI(PTD2, PTD3, PTC5), "ssid", "key", WPA2, false);
Serial pc(USBTX, USBRX);
#elif (MY_BOARD == WIFI_DIPCORTEX)
cc3000 wifi(p28, p27, p30, SPI(p21, p14, p37), "ssid", "key", WPA2, false);
Serial pc(UART_TX, UART_RX);
#elif (MY_BOARD == MBED_BOARD_EXAMPLE)
cc3000 wifi(p9, p10, p8, SPI(p5, p6, p7), "ssid", "key", WPA2, false);
Serial pc(USBTX, USBRX);
#else

#endif

/**
 *  \brief Count seconds
 *  \param none
 *  \return none
 */
void update_timer() {
    time_seconds++;
}

/**
 *  \brief xively demo
 *  \param  none
 *  \return int
 */
int main() {
    init(); /* board dependent init */
    pc.baud(115200);

    printf("cc3000 xively demo. \r\n");
    wifi.init();
    if (wifi.connect() == -1) {
        printf("Failed to connect. Please verify connection details and try again. \r\n");
    } else {
        printf("IP address: %s \r\n", wifi.getIPAddress());
    }
    
    xi_feed_t feed;
    printf("Size of xi_feed_t: %d \r\n", sizeof(xi_feed_t)); //with default settings this can get over 11kB
    std::memset(&feed, 0, sizeof( xi_feed_t ) );
   
    feed.feed_id = XI_FEED_ID;
    feed.datastream_count = 2;
    
    feed.datastreams[0].datapoint_count = 1;
    xi_datastream_t* status_datastream = &feed.datastreams[0];
    strcpy(status_datastream->datastream_id, "LED1");
    xi_datapoint_t* led_status = &status_datastream->datapoints[0];

    feed.datastreams[1].datapoint_count = 1;
    xi_datastream_t* counter_datastream = &feed.datastreams[1];
    strcpy(counter_datastream->datastream_id, "Uptime");
    xi_datapoint_t* counter = &counter_datastream->datapoints[0];
    
    xi_context_t* xi_context = xi_create_context(XI_HTTP, XI_API_KEY, feed.feed_id);

    if (xi_context == NULL)
    {
        printf("Context failed to initialized. \r\n");
        return -1;
    }
        
    flipper_seconds.attach(&update_timer, 1.0);

    while(1) {  
        xi_set_value_f32(led_status, myled);  
        xi_set_value_f32(counter, time_seconds);
      
        printf( "update...\r\n");
        xi_feed_update(xi_context, &feed);
        printf( "done...\r\n");
        wait(10);
        myled = myled ^ 0x1;
    }

}

