/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed.h"
#include "NVIC_set_all_priorities.h"
#include "cc3000.h"
#include "main.h"

#include "TCPSocketConnection.h"
#include "TCPSocketServer.h"

using namespace mbed_cc3000;

#if (WIGO_BOARD)
// Wi-Go battery charger control
DigitalOut PWR_EN1(PTB2);
DigitalOut PWR_EN2(PTB3);
#endif

Serial pc(USBTX, USBRX);
cc3000 wigo(PTA16, PTA13, PTD0, SPI(PTD2, PTD3, PTC5), PORTA_IRQn);
tUserFS user_info;


#ifndef CC3000_UNENCRYPTED_SMART_CONFIG
  const uint8_t smartconfigkey[] = {0x73,0x6d,0x61,0x72,0x74,0x63,0x6f,0x6e,0x66,0x69,0x67,0x41,0x45,0x53,0x31,0x36};
#else
  const uint8_t smartconfigkey = 0;
#endif

/** Print cc3000 information
 *  \param  none
 *  \return none
 */
void print_cc3000_info() {
    uint8_t myMAC[8];

    printf("MAC address + cc3000 info\n");
    wigo.get_user_file_info((uint8_t *)&user_info, sizeof(user_info));
    wigo.get_mac_address(myMAC);
    printf(" MAC address %02x:%02x:%02x:%02x:%02x:%02x\n\n", myMAC[0], myMAC[1], myMAC[2], myMAC[3], myMAC[4], myMAC[5]);

    printf(" FTC        %i\n",user_info.FTC);
    printf(" PP_version %i.%i\n",user_info.PP_version[0], user_info.PP_version[1]);
    printf(" SERV_PACK  %i.%i\n",user_info.SERV_PACK[0], user_info.SERV_PACK[1]);
    printf(" DRV_VER    %i.%i.%i\n",user_info.DRV_VER[0], user_info.DRV_VER[1], user_info.DRV_VER[2]);
    printf(" FW_VER     %i.%i.%i\n",user_info.FW_VER[0], user_info.FW_VER[1], user_info.FW_VER[2]);
}

/** Connect to SSID with a timeout
 *  \param  ssid     Name of SSID
 *  \param  key      Password
 *  \param  sec_mode Security mode
 *  \return none
 */
void connect_to_ssid(char *ssid, char *key, unsigned char sec_mode) {
    printf("Connecting to SSID: %s. Timeout is 10s.\n",ssid);
    if (wigo.connect_to_AP((uint8_t *)ssid, (uint8_t *)key, sec_mode) == true) {
        printf(" Connected\n");
    } else {
        printf(" Connection timed-out (error). Please restart.\n");
        while(1);
  }
}

/** Connect to SSID without security
 *  \param  ssid Name of SSID
 *  \return none
 */
void connect_to_ssid(uint8_t *ssid) {
    wigo.connect_open((uint8_t *)ssid);
}

void wigo_init(void) {
    /* KL25 specific code */
    // Wi-Go set current to 500mA since we're turning on the Wi-Fi
    SET_PWR_EN1;
    SET_PWR_EN2;

    NVIC_set_all_irq_priorities(3);
    NVIC_SetPriority(SPI0_IRQn, 0x0);     // Wi-Fi SPI interrupt must be higher priority than SysTick
    NVIC_SetPriority(PORTA_IRQn, 0x1);
    NVIC_SetPriority(SysTick_IRQn, 0x2);  // SysTick set to lower priority than Wi-Fi SPI bus interrupt
    CLEAR_PCR_INTERRUPT;
    CLEAN_PORT_INTERRUPT;
}

/** First time configuration
 *  \param  none
 *  \return none
 */
void do_FTC(void) {
    printf("Running First Time Configuration\n");
    wigo.start_smart_config(smartconfigkey);
    while (wigo.is_dhcp_configured() == false) {
         wait_ms(500);
         printf("Waiting for dhcp to be set.\n");
    }
    user_info.FTC = 1;
    wigo.set_user_file_info((uint8_t *)&user_info, sizeof(user_info));
    wigo._wlan.stop();
    printf("FTC finished.\n");
}

/** Ping a site
 *  \param  none
 *  \return int
 */
int main() {
    wigo_init();
    pc.baud(115200);

    wigo.start(0);
    printf("CC3000 tcp client demo.\n");
    print_cc3000_info();

    printf("Attempting SSID Connection\n");
#if (USE_SMART_CONFIG == 1)
    if (user_info.FTC == 1) {
        wigo._wlan.ioctl_set_connection_policy(0, 1, 1);
    } else {
        printf("Smart config is not set, starting configuration\n");
        do_FTC();
        printf("Smart config is set. Please restart your board.\n");
        while(1);
    }
#else
    wigo._wlan.ioctl_set_connection_policy(0, 0, 0);
#ifndef CC3000_TINY_DRIVER
#ifdef AP_KEY
    connect_to_ssid(SSID,AP_KEY,AP_SECURITY);
#else
    connect_to_ssid((uint8_t *)SSID);
#endif
#else
    connect_to_ssid((uint8_t *)SSID);
#endif
#endif
    printf("DHCP request\n");
    while (wigo.is_dhcp_configured() == false) {
         wait_ms(500);
         printf("  Waiting for dhcp to be set.\n");
    }

    tNetappIpconfigRetArgs ipinfo2;
    wigo.get_ip_config(&ipinfo2);                  // data is returned in the ipinfo2 structure
    printf("\n*** Wi-Go board DHCP assigned IP Address = %d.%d.%d.%d\n", ipinfo2.aucIP[3], ipinfo2.aucIP[2], ipinfo2.aucIP[1], ipinfo2.aucIP[0]);

    const char* ECHO_SERVER_ADDRESS = "192.168.1.10";
    const int ECHO_SERVER_PORT = 1895;
    
     TCPSocketConnection socket;
     while (socket.connect(ECHO_SERVER_ADDRESS, ECHO_SERVER_PORT) < 0) {
         printf("Unable to connect to (%s) on port (%d)\n", ECHO_SERVER_ADDRESS, ECHO_SERVER_PORT);
         wait(1);
     }
     
     char hello[] = "Hello World\n";
     socket.send_all(hello, sizeof(hello) - 1);
     
     char buf[256];
     int n = socket.receive(buf, 256);
     buf[n] = '\0';
     printf("%s", buf);
     
     socket.close();
     printf("Completed.\n");

}
