/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed.h"
#include "cc3000.h"
#include "main.h"
#include "TCPSocketConnection.h"
#include "TCPSocketServer.h"

#define STRINGIFY(x) #x
#define TO_STRING(x) STRINGIFY(x)

using namespace mbed_cc3000;

tUserFS user_info;

/* cc3000 module declaration specific for user's board. Check also init() */
#if (MY_BOARD == WIGO)
cc3000 wifi(PTA16, PTA13, PTD0, SPI(PTD2, PTD3, PTC5), PORTA_IRQn);
Serial pc(USBTX, USBRX);
#elif (MY_BOARD == WIFI_DIPCORTEX)
cc3000 wifi(p28, p27, p30, SPI(p21, p14, p37), PIN_INT0_IRQn);
Serial pc(UART_TX, UART_RX);
#else

#endif

#ifndef CC3000_UNENCRYPTED_SMART_CONFIG
  const uint8_t smartconfigkey[] = {0x73,0x6d,0x61,0x72,0x74,0x63,0x6f,0x6e,0x66,0x69,0x67,0x41,0x45,0x53,0x31,0x36};
#else
  const uint8_t smartconfigkey = 0;
#endif

/**
 *  \brief Print cc3000 information
 *  \param none
 *  \return none
 */
void print_cc3000_info() {
    uint8_t myMAC[8];

    printf("MAC address + cc3000 info \r\n");
    wifi.get_user_file_info((uint8_t *)&user_info, sizeof(user_info));
    wifi.get_mac_address(myMAC);
    printf(" MAC address %02x:%02x:%02x:%02x:%02x:%02x \r\n \r\n", myMAC[0], myMAC[1], myMAC[2], myMAC[3], myMAC[4], myMAC[5]);

    printf(" FTC        %i \r\n",user_info.FTC);
    printf(" PP_version %i.%i \r\n",user_info.PP_version[0], user_info.PP_version[1]);
    printf(" SERV_PACK  %i.%i \r\n",user_info.SERV_PACK[0], user_info.SERV_PACK[1]);
    printf(" DRV_VER    %i.%i.%i \r\n",user_info.DRV_VER[0], user_info.DRV_VER[1], user_info.DRV_VER[2]);
    printf(" FW_VER     %i.%i.%i \r\n",user_info.FW_VER[0], user_info.FW_VER[1], user_info.FW_VER[2]);
}

/**
 *  \brief Connect to SSID with a timeout
 *  \param ssid     Name of SSID
 *  \param key      Password
 *  \param sec_mode Security mode
 *  \return none
 */
void connect_to_ssid(char *ssid, char *key, unsigned char sec_mode) {
    printf("Connecting to SSID: %s. Timeout is 10s. \r\n",ssid);
    if (wifi.connect_to_AP((uint8_t *)ssid, (uint8_t *)key, sec_mode) == true) {
        printf(" Connected. \r\n");
    } else {
        printf(" Connection timed-out (error). Please restart. \r\n");
        while(1);
  }
}

/**
 *  \brief Connect to SSID without security
 *  \param ssid Name of SSID
 *  \return none
 */
void connect_to_ssid(char *ssid) {
    wifi.connect_open((uint8_t *)ssid);
}

/**
 *  \brief First time configuration
 *  \param none
 *  \return none
 */
void do_FTC(void) {
    printf("Running First Time Configuration \r\n");
    wifi.start_smart_config(smartconfigkey);
    while (wifi.is_dhcp_configured() == false) {
         wait_ms(500);
         printf("Waiting for dhcp to be set. \r\n");
    }
    user_info.FTC = 1;
    wifi.set_user_file_info((uint8_t *)&user_info, sizeof(user_info));
    wifi._wlan.stop();
    printf("FTC finished. \r\n");
}

/**
 *  \brief Start smart config
 *  \param none
 *  \return none
 */
void start_smart_config() {
    wifi.start_smart_config(smartconfigkey);
}

/**
 *  \brief Simple socket demo
 *  \param none
 *  \return int
 */
int main() {
    init(); /* board dependent init */
    pc.baud(115200);

    wifi.start(0);
    printf("cc3000 simple socket demo. \r\n");
    print_cc3000_info();

    printf("User's AP setup: SSID: %s, Password: %s, Security: %s \r\n", TO_STRING(SSID), TO_STRING(AP_KEY), TO_STRING(AP_SECURITY));

    printf("\n<0> Normal run. SmartConfig will \r\n    start if no valid connection exists. \r\n");
    printf("<1> Connect using fixed SSID without AP_KEY: %s \r\n", SSID);
    printf("<2> Connect using fixed SSID with AP_KEY: %s \r\n", SSID);
    printf("<8> Erase all stored profiles.\r\n");
    printf("<9> SmartConfig. \r\n");

    signed char c = getchar();
    switch (c)
    {
        case '0':
            if (!user_info.FTC) {
                do_FTC();
                wifi._wlan.stop();
            } else {
                wifi._wlan.ioctl_set_connection_policy(0, 1, 1);
            }
            break;
        case '1':
            printf("Attempting SSID Connection. \r\n");

            wifi._wlan.ioctl_set_connection_policy(0, 0, 0);
            connect_to_ssid(SSID);
            break;
        case '2':
            printf("Attempting SSID Connection. \r\n");

            wifi._wlan.ioctl_set_connection_policy(0, 0, 0);
#ifndef CC3000_TINY_DRIVER
            connect_to_ssid(SSID, AP_KEY, AP_SECURITY);
#else
            connect_to_ssid(SSID);
#endif
            break;
        case '8':
            printf("Erasing all wireless profiles. \r\n");
            wifi.delete_profiles();
            wifi.stop();
            printf("Done - press the reset button on your board... \r\n");
            while(1);

        case '9':
            printf("Starting Smart Config configuration. \r\n");
            start_smart_config();
            while (wifi.is_dhcp_configured() == false)
            {
                wait_ms(500);
                printf("Waiting for dhcp to be set. \r\n");
            }

            printf("Press the reset button on your board and select 0... \r\n");
            while(1);
        default:
            printf("Wrong selection. \r\n");
            printf("Reset the board and try again. \r\n");
            break;
    }

    printf("DHCP request \r\n");
    while (wifi.is_dhcp_configured() == false) {
         wait_ms(500);
         printf("  Waiting for dhcp to be set. \r\n");
    }

    tNetappIpconfigRetArgs ipinfo2;
    wifi.get_ip_config(&ipinfo2); // data is returned in the ipinfo2 structure
    printf("DHCP assigned IP Address = %d.%d.%d.%d \r\n", ipinfo2.aucIP[3], ipinfo2.aucIP[2], ipinfo2.aucIP[1], ipinfo2.aucIP[0]);

    char python_msg[] = "Hello Python\n";
    TCPSocketServer server;
    TCPSocketConnection client;

    server.bind(15000);
    server.listen();

    while (1) {
        int32_t status = server.accept(client);
        if (status >= 0) {
            client.set_blocking(false, 1500); // Timeout after (1.5)s
            printf("Connection from: %s \r\n", client.get_address());
            char buffer[256];
            int stat = client.receive(buffer, sizeof(buffer));
            if (stat >=0) {
                printf("Received: %s \r\n",buffer);
                printf("Sending the message to the server. \r\n");
                client.send_all(python_msg, sizeof(python_msg));
                client.close();
            }
        }
    }
}
