/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed.h"
#include "cc3000.h"
#include "main.h"

#include "HTTPClient.h"

using namespace mbed_cc3000;

/* cc3000 module declaration specific for user's board. Check also init() */
#if (MY_BOARD == WIGO)
cc3000 wifi(PTA16, PTA13, PTD0, SPI(PTD2, PTD3, PTC5), PORTA_IRQn);
Serial pc(USBTX, USBRX);
#elif (MY_BOARD == WIFI_DIPCORTEX)
cc3000 wifi(p28, p27, p30, SPI(p21, p14, p37), PIN_INT0_IRQn);
Serial pc(UART_TX, UART_RX);
#else

#endif

tUserFS user_info;
HTTPClient http;
char str[512];

#ifndef CC3000_UNENCRYPTED_SMART_CONFIG
  const uint8_t smartconfigkey[] = {0x73,0x6d,0x61,0x72,0x74,0x63,0x6f,0x6e,0x66,0x69,0x67,0x41,0x45,0x53,0x31,0x36};
#else
  const uint8_t smartconfigkey = 0;
#endif

/**
 *  \brief Print cc3000 information
 *  \param none
 *  \return none
 */
void print_cc3000_info() {
    uint8_t myMAC[8];

    printf("MAC address + cc3000 info \r\n");
    wifi.get_user_file_info((uint8_t *)&user_info, sizeof(user_info));
    wifi.get_mac_address(myMAC);
    printf(" MAC address %02x:%02x:%02x:%02x:%02x:%02x \r\n \r\n", myMAC[0], myMAC[1], myMAC[2], myMAC[3], myMAC[4], myMAC[5]);

    printf(" FTC        %i \r\n",user_info.FTC);
    printf(" PP_version %i.%i \r\n",user_info.PP_version[0], user_info.PP_version[1]);
    printf(" SERV_PACK  %i.%i \r\n",user_info.SERV_PACK[0], user_info.SERV_PACK[1]);
    printf(" DRV_VER    %i.%i.%i \r\n",user_info.DRV_VER[0], user_info.DRV_VER[1], user_info.DRV_VER[2]);
    printf(" FW_VER     %i.%i.%i \r\n",user_info.FW_VER[0], user_info.FW_VER[1], user_info.FW_VER[2]);
}

/**
 *  \brief Connect to SSID with a timeout
 *  \param ssid     Name of SSID
 *  \param key      Password
 *  \param sec_mode Security mode
 *  \return none
 */
void connect_to_ssid(char *ssid, char *key, unsigned char sec_mode) {
    printf("Connecting to SSID: %s. Timeout is 10s. \r\n",ssid);
    if (wifi.connect_to_AP((uint8_t *)ssid, (uint8_t *)key, sec_mode) == true) {
        printf(" Connected. \r\n");
    } else {
        printf(" Connection timed-out (error). Please restart. \r\n");
        while(1);
  }
}

/**
 *  \brief Connect to SSID without security
 *  \param ssid Name of SSID
 *  \return none
 */
void connect_to_ssid(char *ssid) {
    wifi.connect_open((uint8_t *)ssid);
}

/**
 *  \brief First time configuration
 *  \param none
 *  \return none
 */
void do_FTC(void) {
    printf("Running First Time Configuration \r\n");
    wifi.start_smart_config(smartconfigkey);
    while (wifi.is_dhcp_configured() == false) {
         wait_ms(500);
         printf("Waiting for dhcp to be set. \r\n");
    }
    user_info.FTC = 1;
    wifi.set_user_file_info((uint8_t *)&user_info, sizeof(user_info));
    wifi._wlan.stop();
    printf("FTC finished.\n");
}

/**
 *  \brief HTTP client demo
 *  \param  none
 *  \return int
 */
int main() {
    init(); /* board dependent init */
    pc.baud(115200);

    wifi.start(0);
    printf("cc3000 HTTP client demo. \r\n");
    print_cc3000_info();

    printf("Attempting SSID Connection. \r\n");
#if (USE_SMART_CONFIG == 1)
    if (user_info.FTC == 1) {
        wifi._wlan.ioctl_set_connection_policy(0, 1, 1);
    } else {
        printf("Smart config is not set, starting configuration. \r\n");
        do_FTC();
        printf("Smart config is set. Please restart your board. \r\n");
        while(1);
    }
#else
    wifi._wlan.ioctl_set_connection_policy(0, 0, 0);
#ifndef CC3000_TINY_DRIVER
#ifdef AP_KEY
    connect_to_ssid(SSID, AP_KEY, AP_SECURITY);
#else
    connect_to_ssid(SSID);
#endif
#else
    connect_to_ssid(SSID);
#endif
#endif
    printf("DHCP request \r\n");
    while (wifi.is_dhcp_configured() == false) {
         wait_ms(500);
         printf("  Waiting for dhcp to be set. \r\n");
    }

    tNetappIpconfigRetArgs ipinfo2;
    wifi.get_ip_config(&ipinfo2); // data is returned in the ipinfo2 structure
    printf("DHCP assigned IP Address = %d.%d.%d.%d \r\n", ipinfo2.aucIP[3], ipinfo2.aucIP[2], ipinfo2.aucIP[1], ipinfo2.aucIP[0]);

    //GET data
    printf("\r\nTrying to fetch page... \r\n");
    int ret = http.get("http://mbed.org/media/uploads/donatien/hello.txt", str, 128);
    if (!ret)
    {
      printf("Page fetched successfully - read %d characters \r\n", strlen(str));
      printf("Result: %s \r\n", str);
    }
    else
    {
      printf("Error - ret = %d - HTTP return code = %d \r\n", ret, http.getHTTPResponseCode());
    }

    //POST data
    HTTPMap map;
    HTTPText inText(str, 512);
    map.put("Hello", "World");
    map.put("test", "1234");
    printf(" \r\nTrying to post data... \r\n");
    ret = http.post("http://httpbin.org/post", map, &inText);
    if (!ret)
    {
      printf("Executed POST successfully - read %d characters \r\n", strlen(str));
      printf("Result: %s \r\n", str);
    }
    else
    {
      printf("Error - ret = %d - HTTP return code = %d \r\n", ret, http.getHTTPResponseCode());
    }

    //PUT data
    strcpy(str, "This is a PUT test!");
    HTTPText outText(str);
    //HTTPText inText(str, 512);
    printf(" \r\nTrying to put resource... \r\n");
    ret = http.put("http://httpbin.org/put", outText, &inText);
    if (!ret)
    {
      printf("Executed PUT successfully - read %d characters \r\n", strlen(str));
      printf("Result: %s \r\n", str);
    }
    else
    {
      printf("Error - ret = %d - HTTP return code = %d \r\n", ret, http.getHTTPResponseCode());
    }

    //DELETE data
    //HTTPText inText(str, 512);
    printf(" \r\nTrying to delete resource... \r\n");
    ret = http.del("http://httpbin.org/delete", &inText);
    if (!ret)
    {
      printf("Executed DELETE successfully - read %d characters \r\n", strlen(str));
      printf("Result: %s \r\n", str);
    }
    else
    {
      printf("Error - ret = %d - HTTP return code = %d \r\n", ret, http.getHTTPResponseCode());
    }

    printf("Demo completed. \r\n");
    wifi.disconnect();
}
