/*****************************************************************************
*
*  C++ interface/implementation created by Martin Kojtal (0xc0170). Thanks to
*  Jim Carver and Frank Vannieuwkerke for their inital cc3000 mbed port and
*  provided help.
*
*  This version of "host driver" uses CC3000 Host Driver Implementation. Thus
*  read the following copyright:
*
*  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*****************************************************************************/
#include "cc3000.h"
#include "cc3000_event.h"

namespace mbed_cc3000 {

/* TODO this prefix remove? verify */
static uint8_t cc3000_prefix[] = {'T', 'T', 'T'};
cc3000 *cc3000::_inst;

cc3000::cc3000(PinName cc3000_irq, PinName cc3000_en, PinName cc3000_cs, SPI cc3000_spi, IRQn_Type irq_port)
            :  _event(_simple_link, _hci, _spi, *this), _socket(_simple_link, _hci, _event), _spi(cc3000_irq, cc3000_en, cc3000_cs, cc3000_spi, irq_port, _event, _simple_link), _hci(_spi),
            _nvmem(_hci, _event, _simple_link), _netapp(_simple_link, _nvmem, _hci, _event), _wlan(_simple_link, _event, _spi, _hci) {
    /* TODO - pIRQ riorities ?? */

    _simple_link.set_tx_complete_signal(1);
    _status.dhcp = 0;
    _status.connected = 0;
    _status.socket = 0;
    _status.dhcp_configured = 0;
    _status.smart_config_complete = 0;
    _status.stop_smart_config = 0;
    _status.ok_to_shut_down = 0;

    _inst = this;
}

cc3000::~cc3000() {

}

void cc3000::usync_callback(int32_t event_type, uint8_t * data, uint8_t length) {
    if (event_type == HCI_EVNT_WLAN_ASYNC_SIMPLE_CONFIG_DONE)
    {
        _status.smart_config_complete = 1;
        _status.stop_smart_config = 1;
    }

    if (event_type == HCI_EVNT_WLAN_UNSOL_CONNECT)
    {
        _status.connected = 1;
    }

    if (event_type == HCI_EVNT_WLAN_UNSOL_DISCONNECT)
    {
        _status.connected = 0;
        _status.dhcp      = 0;
        _status.dhcp_configured = 0;
    }

    if (event_type == HCI_EVNT_WLAN_UNSOL_DHCP)
    {
      if ( *(data + NETAPP_IPCONFIG_MAC_OFFSET) == 0) {
          _status.dhcp = 1;
      } else {
          _status.dhcp = 0;
      }
    }

    if (event_type == HCI_EVENT_CC3000_CAN_SHUT_DOWN)
    {
        _status.ok_to_shut_down = 1;
    }

    if (event_type == HCI_EVNT_WLAN_ASYNC_PING_REPORT)
    {
        memcpy(&_ping_report, data, length);
    }

    if (event_type == HCI_EVNT_BSD_TCP_CLOSE_WAIT) {
        uint8_t socketnum;
        socketnum = data[0];
        if (socketnum < MAX_SOCKETS) {
            _closed_sockets[socketnum] = true; /* clients socket is closed */
        }
    }
}

void cc3000::start_smart_config(const uint8_t *smart_config_key) {
    // Reset all the previous configuration
    _wlan.ioctl_set_connection_policy(0, 0, 0);
    _wlan.ioctl_del_profile(255);

    //Wait until CC3000 is disconected
    while (_status.connected == 1)
    {
        wait_us(5);
        _event.hci_unsolicited_event_handler();
    }

    // Trigger the Smart Config process

    _wlan.smart_config_set_prefix(cc3000_prefix);
    // Start the Smart Config process with AES disabled
    _wlan.smart_config_start(0);

    DBG_CC("Waiting for smartconfig to be completed");

    // Wait for Smart config finished
    while (_status.smart_config_complete == 0)
    {
        wait_ms(100);

    }
    
    DBG_CC("Smartconfig finished");
    
#ifndef CC3000_UNENCRYPTED_SMART_CONFIG
    // create new entry for AES encryption key
    _nvmem.create_entry(NVMEM_AES128_KEY_FILEID, 16);
    // write AES key to NVMEM
    _security.aes_write_key((uint8_t *)(&smart_config_key[0]));
    // Decrypt configuration information and add profile
    _wlan.smart_config_process();
#endif

    // Configure to connect automatically to the AP retrieved in the
    // Smart config process
    _wlan.ioctl_set_connection_policy(0, 1, 1);

    // reset the CC3000
    _wlan.stop();
    wait(2);
    _wlan.start(0);
    wait(2);

    // Mask out all non-required events
    _wlan.set_event_mask(HCI_EVNT_WLAN_KEEPALIVE|HCI_EVNT_WLAN_UNSOL_INIT|HCI_EVNT_WLAN_ASYNC_PING_REPORT);
}

bool cc3000::connect_secure(const uint8_t *ssid, const uint8_t *key, int32_t security_mode) {
    uint32_t ret;

    _wlan.disconnect();
    wait_ms(3);
    ret = _wlan.connect(security_mode, ssid, strlen((const char *)ssid), 0, (uint8_t *)key, strlen((const char *)key));
    if (ret == 0) { /* TODO static internal cc3000 state 0 to TRUE */
      ret = true;
    } else {
      ret = false;
    }
    return ret;
}

bool cc3000::connect_to_AP(const uint8_t *ssid, const uint8_t *key, int32_t security_mode) {
    Timer t;  /* TODO static? */
    bool ret = true;

    t.start();
    while (is_connected() == false) {
        if (key == 0) {
            if (connect_open(ssid)) {
                break;
            }
        } else {
#ifndef CC3000_TINY_DRIVER
            if (connect_secure(ssid,key,security_mode)) {
                break;
            }
#else
            return false; /* secure connection not supported with TINY_DRIVER */
#endif
        }

        /* timeout 10 seconds */
        if (t.read_ms() > 10000){
            ret = false;
            
            DBG_CC("Connection to AP failed");
            
            break;
        }
    }

    return ret;
}

void cc3000::start(uint8_t patch) {
    _wlan.start(patch);
    _wlan.set_event_mask(HCI_EVNT_WLAN_UNSOL_INIT | HCI_EVNT_WLAN_KEEPALIVE);
}

void cc3000::stop(void) {
    _wlan.stop();
}

void cc3000::restart(uint8_t patch) {
    _wlan.stop();
    wait_ms(500);
    _wlan.start(patch);
}

bool cc3000::connect_open(const uint8_t *ssid) {
    uint32_t ret;

    _wlan.disconnect();
    wait_ms(3);
#ifndef CC3000_TINY_DRIVER
    ret = _wlan.connect(0,ssid, strlen((const char *)ssid), 0, 0, 0);
#else
    ret = _wlan.connect(ssid, strlen((const char *)ssid));
#endif
    if (ret == 0) {
        ret = true;
    } else {
        ret = false;
    }
    return ret;
}

bool cc3000::is_connected() {
    return _status.connected;
}

bool cc3000::is_dhcp_configured() {
    return _status.dhcp;
}

bool cc3000::is_smart_confing_completed() {
    return _status.smart_config_complete;
}

void cc3000::get_mac_address(uint8_t address[6]) {
    _nvmem.get_mac_address(address);
}

void cc3000::set_mac_address(uint8_t address[6]) {
    _nvmem.set_mac_address(address);
}

void cc3000::get_user_file_info(uint8_t *info_file, size_t size) {
    _nvmem.read( NVMEM_USER_FILE_1_FILEID, size, 0, info_file);
}

#ifndef CC3000_TINY_DRIVER
bool cc3000::get_ip_config(tNetappIpconfigRetArgs *ip_config) {
    if ((_status.dhcp == false) || (_status.connected == false)) {
        return false;
    }

    _netapp.ipconfig(ip_config);
    return true;
}
#endif

cc3000_client cc3000::create_tcp_client(uint32_t ip_address, uint16_t port) {
    sockaddr socket_address = {0};
    int32_t tcp_socket;

    tcp_socket = _socket.socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (tcp_socket == -1) {
        DBG_CC("Failed to create new socket (tcp)");
        return cc3000_client(*this);
    }

    socket_address.family = AF_INET;
    socket_address.data[0] = (port & 0xFF00) >> 8;
    socket_address.data[1] = (port & 0x00FF);
    socket_address.data[2] = ip_address >> 24;
    socket_address.data[3] = ip_address >> 16;
    socket_address.data[4] = ip_address >> 8;
    socket_address.data[5] = ip_address;

    if (_socket.connect(tcp_socket, &socket_address, sizeof(socket_address)) == -1) {
        DBG_CC("Failed to connect (tcp)");
        _socket.closesocket(tcp_socket);
        return cc3000_client(*this);
    }

    return cc3000_client(*this, tcp_socket);
}
cc3000_client cc3000::create_udp_client(uint32_t ip_address, uint16_t port) {
    sockaddr socket_address = {0};
    int32_t udp_socket;

    udp_socket = _socket.socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (udp_socket == -1) {
        DBG_CC("Failed to create new socket (udp)");
        return cc3000_client(*this);
    }

    socket_address.family = AF_INET;
    socket_address.data[0] = (port & 0xFF00) >> 8;
    socket_address.data[1] = (port & 0x00FF);
    socket_address.data[2] = ip_address >> 24;
    socket_address.data[3] = ip_address >> 16;
    socket_address.data[4] = ip_address >> 8;
    socket_address.data[5] = ip_address;

    if (_socket.connect(udp_socket, &socket_address, sizeof(socket_address)) == -1) {
        DBG_CC("Failed to connect (udp)");
        _socket.closesocket(udp_socket);
        return cc3000_client(*this);
    }

    return cc3000_client(*this, udp_socket);
}

cc3000_server cc3000::create_tcp_server(uint32_t ip_address, uint16_t port) {
    sockaddr socket_address = {0};
    int32_t tcp_socket;

    tcp_socket = _socket.socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (tcp_socket == -1) {
        DBG_CC("Failed to create new socket.");
        return cc3000_server(*this, socket_address);
    }

    socket_address.family = AF_INET;
    socket_address.data[0] = (port & 0xFF00) >> 8;
    socket_address.data[1] = (port & 0x00FF);
    socket_address.data[2] = ip_address >> 24;
    socket_address.data[3] = ip_address >> 16;
    socket_address.data[4] = ip_address >> 8;
    socket_address.data[5] = ip_address;

    if (_socket.bind(tcp_socket, &socket_address, sizeof(socket_address)) != 0) {
        DBG_CC("Failed to bind the new socket");
        return cc3000_server(*this, socket_address);
    }
    if (_socket.listen(tcp_socket, 1) != 0) { /* 1 client */
        DBG_CC("Failed to listen on the new socket");
        return cc3000_server(*this, socket_address);
    }

    return cc3000_server(*this, socket_address, tcp_socket);
}

void cc3000::delete_profiles(void) {
    tUserFS user_info;

    _wlan.ioctl_set_connection_policy(0, 0, 0);
    _wlan.ioctl_del_profile(255);

    get_user_file_info((uint8_t *)&user_info, sizeof(user_info));
    user_info.FTC = 0;
    set_user_file_info((uint8_t *)&user_info, sizeof(user_info));
}

void cc3000::set_user_file_info(uint8_t *info_file, size_t size) {
    _nvmem.write( NVMEM_USER_FILE_1_FILEID, size, 0, info_file);
}

bool cc3000::disconnect(void){
    if (_wlan.disconnect()) {
        return false;
    } else {
        return true;
    }
}

uint32_t cc3000::ping(uint32_t ip, uint8_t attempts, uint16_t timeout, uint8_t size) {
    uint32_t reversed_ip = (ip >> 24) | (ip >> 8) & 0xFF00 | (ip << 8) & 0xFF0000 | (ip << 24);

    _ping_report.packets_received = 0;
    if (_netapp.ping_send(&reversed_ip, attempts, size, timeout) == -1) {
        DBG_CC("Failed to send ping");
        return 0;
    }
    wait_ms(timeout*attempts*2);

    /* known issue of cc3000 - sent number is send + received */
    // TODO : Remove the Sent/recv'd counts until ti fix the firmware issue?
    DBG_CC("Sent: %d",_ping_report.packets_sent);
    DBG_CC("Received: %d",_ping_report.packets_received);
    DBG_CC("Min time: %d",_ping_report.min_round_time);
    DBG_CC("Max time: %d",_ping_report.max_round_time);
    DBG_CC("Avg time: %d",_ping_report.avg_round_time);

    return _ping_report.packets_received;
}

/* Conversion between uint types and C strings */
uint8_t* UINT32_TO_STREAM_f (uint8_t *p, uint32_t u32)
{
    *(p)++ = (uint8_t)(u32);
    *(p)++ = (uint8_t)((u32) >> 8);
    *(p)++ = (uint8_t)((u32) >> 16);
    *(p)++ = (uint8_t)((u32) >> 24);
    return p;
}


uint8_t* UINT16_TO_STREAM_f (uint8_t *p, uint16_t u16)
{
    *(p)++ = (uint8_t)(u16);
    *(p)++ = (uint8_t)((u16) >> 8);
    return p;
}


uint16_t STREAM_TO_UINT16_f(uint8_t *p, uint16_t offset)
{
    return (uint16_t)((uint16_t)((uint16_t)
           (*(p + offset + 1)) << 8) + (uint16_t)(*(p + offset)));
}


uint32_t STREAM_TO_UINT32_f(uint8_t *p, uint16_t offset)
{
    return (uint32_t)((uint32_t)((uint32_t)
           (*(p + offset + 3)) << 24) + (uint32_t)((uint32_t)
           (*(p + offset + 2)) << 16) + (uint32_t)((uint32_t)
           (*(p + offset + 1)) << 8) + (uint32_t)(*(p + offset)));
}

} /* end of mbed_cc3000 namespace */

