#ifndef OPTIONS_H
#define OPTIONS_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"

/** External variables used inside and out of the class. */
extern float opt_brightness;
extern float opt_contrast;
extern int opt_volume;

/** Options Class
*@brief This class is for the options menu screen.
*@author Kern Fowler
*@version 1.0
*@date May 2019
*/

class Options {

public:
/** Options Constructor 
@brief Builds my default Options contructor.
@details This does not have any setup. 
*/
Options();
/** Options Destructor 
@brief Builds my default Options destructor.
@details This does not have any setup. 
*/
~Options();
// Mutators

/** 
*@brief Runs and controls all the options functions.
*@param pad The Gamepad class is used.
*@param lcd The N5110 class is used.
*@return None.
*@details Prints the options menu, and controls various functions within the menu.
*@code
void Options::options_run(Gamepad &pad, N5110 &lcd) {
    wait_ms(250);
    while (pad.check_event(Gamepad::BACK_PRESSED) == false) { // Continues to show this screen until BACK button pressed.
        //printf("Options State");
        lcd.clear();
        lcd.printString("Options",21,0);
        options_brightness(pad, lcd); // Calls brightness function within class.
        options_contrast(pad, lcd); // Calls contrast function within class.
        options_volume(pad, lcd); // Calls volume function within class.
        lcd.refresh(); // Reloads screen on every cycle, controlled by fps. Default set to 24.
        wait_ms(1.0f/24);
    }
}
@endcode
*/
void options_run(Gamepad &pad, N5110 &lcd);
/** 
*@brief Controls the state of LCD backlight.
*@param pad The Gamepad class is used.
*@param lcd The N5110 class is used.
*@return None.
*@details Toggles the backlight on LCD screen, ON/OFF based on button inputs. 
*@code
void Options::options_brightness(Gamepad &pad, N5110 &lcd) {
    if (pad.check_event(Gamepad::B_PRESSED) == true) { // If B is pressed, will turn off backlight.
        opt_brightness = 0;
    }
    if (pad.check_event(Gamepad::A_PRESSED) == true) { // If A is pressed, will turn on backlight.
        opt_brightness = 0.5;
    }
    lcd.setBrightness(opt_brightness);
    lcd.printString("A/B = BackLite",0,2);
}
@endcode
*/
void options_brightness(Gamepad &pad, N5110 &lcd);
/** 
*@brief Controls the LCD contrast.
*@param pad The Gamepad class is used.
*@param lcd The N5110 class is used.
*@return None.
*@details Controls the LCD screen contrast based on gamepad potentiometer value.
*@code
void Options::options_contrast(Gamepad &pad, N5110 &lcd) {
    opt_contrast = pad.read_pot(); // Gets potentiometer value from gamepad and assigns it to a value.
    lcd.setContrast(opt_contrast); // Uses set value to change actual LCD contrast.
    lcd.printString("Pot = Contrast",0,3);
    printf("Contrast = %f", opt_contrast);
}
@endcode
*/
void options_contrast(Gamepad &pad, N5110 &lcd);
/** 
*@brief Controls the state of game volume.
*@param pad The Gamepad class is used.
*@param lcd The N5110 class is used.
*@return None.
*@details Toggles the volume for the game, ON/OFF based on button inputs.
*@code
void Options::options_volume(Gamepad &pad, N5110 &lcd) {
    if (pad.check_event(Gamepad::Y_PRESSED) == true) { // If Y is pressed, will disable volume.
        opt_volume = 0;
    }
    if (pad.check_event(Gamepad::X_PRESSED) == true) { // If X is press, will enable volume.
        opt_volume = 1;
        pad.tone(2400, 0.2); // Plays a set of beeps to confirm the volume is on. 
        wait_ms(200);
        pad.tone(2400, 0.2);
        wait_ms(200);
        pad.tone(2400, 0.2);
    }
    lcd.printString("X/Y = Volume",0,4);
    if (opt_volume == 0) { // Shows on screen the current volume state.
        lcd.printString("Off",36,5);
    } else {
        lcd.printString("On",36,5);
    }
}
@endcode
*/
void options_volume(Gamepad &pad, N5110 &lcd);
};

#endif