#ifndef GAMEENGINE_H
#define GAMEENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Donkey.h"
#include "Barrel.h"
#include "Banana.h"
#include "HighScores.h"
#include "Options.h"

/** GameEngine Class
*@brief This class is running and contrling the main game functions.
*@author Kern Fowler
*@version 1.0
*@date May 2019
*/

class GameEngine {

public: 
/** GameEngine Constructor 
@brief Builds my default GameEngine constructor.
@details This does not have any setup. 
*/
GameEngine();
/** GameEngine Destructor 
@brief Builds my default GameEngine destructor.
@details This does not have any setup. 
*/
~GameEngine();
// Mutators

/** 
*@brief Controls the main game.
*@param pad The Gamepad class is used.
*@param lcd The N5110 class is used.
*@param barrel The Barrel class is used.
*@param banana The Banana class is used.
*@param dky The Donkey class is used.
*@param opt The Options class is used.
*@return None.
*@details Runs the main functions of the game in correct order.
*@code
void GameEngine::gameengine_run(Gamepad &pad, N5110 &lcd, Barrel &barrel, Banana &banana, Donkey &dky, Options &opt) {
    wait_ms(250);
    // Sets key variables back to default value when game first ran.
    barrel_x = 0; 
    barrel_y = 0;
    banana_x = 0;
    banana_y = 0;
    running = 1;
    banana_time = 0;
    barrel_time = 0;
    score = 0;
    while (running == 1) { // Main game loop, continues until game over occurs.
        //printf("Game State");
        lcd.clear();
        dky.donkeykong_movement(pad, lcd); // Calls Donkey Kong model section of game.
        barrel.barrel_drop(pad, lcd, dky); // Calls Barrel model section of game.
        banana.banana_drop(pad, lcd, barrel, dky, opt); // Calls Banana model section of game.
        //printf("state %d", running);
        lcd.refresh(); // Reloads screen on every cycle, controlled by fps. Default set to 24.
        wait_ms(1.0f/24);
    }
}
@endcode
*/
void gameengine_run(Gamepad &pad, N5110 &lcd, Barrel &barrel, Banana &banana, Donkey &dky, Options &opt);
/** 
*@brief Shows game over screen.
*@param pad The Gamepad class is used.
*@param lcd The N5110 class is used.
*@param banana The Banana class is used.
*@param high The HighScores class is used.
*@return None.
*@details Prints the gameover screen. Prints various text, including total player score.
*@code
void GameEngine::gameengine_score(Gamepad &pad, N5110 &lcd, Banana &banana, HighScores &high) {
    lcd.clear();
    lcd.printString("Game Over!",14,0);
    lcd.printString("Score:",0,2);
    char buffer[14]; // Shows final score on screen.
    sprintf(buffer,"%i",score);
    lcd.printString(buffer,40,2);
    high.highscores_new(pad, lcd, banana); // Checks to see if new high score is obtained.
    lcd.refresh();
    wait(5);
}
@endcode
*/
void gameengine_score(Gamepad &pad, N5110 &lcd, Banana &banana, HighScores &high);
};

#endif