#ifndef TX_POWER_H
#define TX_POWER_H

#include "BLEDevice.h"

/**
* @class Tx power service
* @brief Tx power Service. This service exposes a device’s current transmit power level when in a connection (-100dbm to +20).
* Service: https://developer.bluetooth.org/gatt/services/Pages/ServiceViewer.aspx?u=org.bluetooth.service.tx_power.xml
* Tx power Char: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.tx_power_level.xml
*/
class TxPowerService
{

public:
    /**
    * @param[ref] _ble
    *             BLEDevice object for the underlying controller.
    * @param[in] TxValue
    *             signed 8bit Transmit power.
    */
    TxPowerService(BLEDevice &_ble, int8_t TxValue):
        ble(_ble),
        TxPower(TxValue),
        TxPowerChar(GattCharacteristic::UUID_TX_POWER_LEVEL_CHAR, (uint8_t*)&TxPower, sizeof(TxPower), sizeof(TxPower),
                    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ) {

        static bool serviceAdded = false; /* We should only ever add one Tx power service. */
        if (serviceAdded) {
            return;
        }

        GattCharacteristic *charTable[] = {&TxPowerChar};
        GattService         TxPowerService(GattService::UUID_TX_POWER_SERVICE, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));
        ble.addService(TxPowerService);
        serviceAdded = true;

    }

    /**
    * @brief Update the Tx power with a new value. with a valid range from -100dbm to +20dbm according to the service Char.
    *
    * @param newTxPower
    *              update the Tx power with the new value.
    */
    void updateTxPower(int8_t newTxPower) {

        if(newTxPower >= -100 && newTxPower <= 20) {

            TxPower = newTxPower;
            ble.updateCharacteristicValue (TxPowerChar.getValueAttribute().getHandle(), (uint8_t*)&TxPower, 1);

        }
    }

private:
    BLEDevice &ble;
    int8_t TxPower;
    GattCharacteristic  TxPowerChar;
};

#endif