#ifndef SteeringServo_H
#define SteeringServo_H
#include "mbed.h"

/** NAME: SteeringServo
 *  AUTHOR: J.BATCHELAR
 *
 * @section LICENSE
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Used with Tower Pro 9g Servo Motors (http://www.micropik.com/PDF/SG90Servo.pdf) to create a steering mechanism. 
 * 
 * Example:
 * @code
 
// -- Code to move steering wheel between extreme limits
#include "mbed.h"
#include "SteeringServo.h"

DigitalIn bttn(PC_13);
DigitalOut led(LED1);

Serial pc(USBTX, USBRX);
SteeringServo steerMech(PA_10, 1650, 1000, 2300);

int main() {
    led = 1;
    pc.printf("\nServo Example\n");
   
   while(1){
        steerMech.goStraight();         //-- Don't call constantly(as you are writing into register need to allow time for value to set)!
        //-- Wait for button to be pressed
        while(bttn){};
        steerMech.goToAngle(1000); //-- Go Left
        wait(5);
        steerMech.goToAngle(2300); //-- Go Left
        wait(5);
       }
    }

 * @endcode
 */

class SteeringServo{
    public:
        /** Create a Clamp Servo Object
        *
        * @param pin The PWM Pin that the Servo Is Connected To
        * @param centrePW (us) - Pulse Width to move to drive straight
        * @param minPW (us) - Minimum pulse width that servo can move to (due to mechanical constraints)
        * @param maxPW (us) - Maximum pulse width that servo can move to (due to mechanical constraints)
        */
        SteeringServo(PinName pin, int centrePW, int minPW, int maxPW); 
        
        /** Check if the Clamp is Open
        * @param pulseWidth (us) - Pulse Width to move rotate to (is internally constrained by limits)
        */
        void goToAngle(int pulseWidth); 
        
 
        /** Goto the centre position. (Drive Straight)
        *
        */                     
        void goStraight(void);
        
        /** Get the currentPulseWidth
        *
        */
        int getPulseWidth(void);    
     
    private:
        PwmOut servo_;               // -- PWM Object  
        int minimumPulseWidth_;     // -- (us) Minimum pulse width that servo can move to (due to mechanical constraints) 
        int maximumPulseWidth_;     // -- (us) Maximum pulse width that servo can move to (due to mechanical constraints)
        int centrePulseWidth_;      // -- (us) Pulse Width to move to drive straight
        int currentPulseWidth_;     // -- (us) Current Pulse Width of the servo
        };

#endif
