/*
 * mbed library for the JDI color memory LCD LPM013M126A
 * derived from C12832_lcd
 * Copyright (c) 2016  Tadayuki Okamoto 
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef MBED_COLOR_MEM_LCD_H
#define MBED_COLOR_MEM_LCD_H

#include "mbed.h"
#include "GraphicsDisplay.h"


/** @def
 * Device define
 */
#define LCD_DEVICE_WIDTH        (176)
#define LCD_DEVICE_HEIGHT       (176)

/** @def
 * window system define
 */
#define LCD_DISP_WIDTH          (176)
#define LCD_DISP_HEIGHT         (176)
#define LCD_DISP_HEIGHT_MAX_BUF (44)


/** @def
 * some RGB color definitions
 */
/*                                        R, G, B     */
#define LCD_COLOR_BLACK     (0x00)    /*  0  0  0  0  */
#define LCD_COLOR_BLUE      (0x02)    /*  0  0  1  0  */
#define LCD_COLOR_GREEN     (0x04)    /*  0  1  0  0  */
#define LCD_COLOR_CYAN      (0x06)    /*  0  1  1  0  */
#define LCD_COLOR_RED       (0x08)    /*  1  0  0  0  */
#define LCD_COLOR_MAGENTA   (0x0a)    /*  1  0  1  0  */
#define LCD_COLOR_YELLOW    (0x0c)    /*  1  1  0  0  */
#define LCD_COLOR_WHITE     (0x0e)    /*  1  1  1  0  */

/** @def
 * ID for setTransMode
 */
#define LCD_TRANSMODE_OPAQUE        (0x00)  //!< BackGroud is Opaque
#define LCD_TRANSMODE_TRANSPARENT   (0x01)  //!< BackGroud is Transparent
#define LCD_TRANSMODE_TRANSLUCENT   (0x02)  //!< BackGroud is Translucent

/** @def
 *ID for setBlinkMode
 */
#define LCD_BLINKMODE_NONE      (0x00)  //!< Blinking None
#define LCD_BLINKMODE_WHITE     (0x01)  //!< Blinking White
#define LCD_BLINKMODE_BLACK     (0x02)  //!< Blinking Black
#define LCD_BLINKMODE_INVERSE   (0x03)  //!< Inversion Mode

/** A class for Color Memory LCD Library
*/
class ColorMemLCD : public GraphicsDisplay {
    public:

    /** Create a SPI_LCD object connected to SPI
     *
     * @param mosi pin connected to SDO of display
     * @param miso pin connected to SDI of display
     * @param sclk pin connected to RS of display 
     * @param cs pin connected to CS of display
     * @param disp pin connected to DISP of display
     * @param power pin connected to POWER of display
     */
    ColorMemLCD( PinName mosi, PinName miso, PinName sclk, PinName cs, PinName disp, PinName power, const char* name ="TFT" );

    /** Get the width of the screen in pixel
     *
     * @return width of screen in pixel
     *
     */
    virtual int width();

    /** Get the height of the screen in pixel
     *
     * @return height of screen in pixel 
     *
     */
    virtual int height();

    /** Set window region
     *
     * @param x horizontal position
     * @param y vertical position
     * @param w window width in pixel
     * @param h window height in pixels
     */
    virtual void window( int x, int y, int w, int h );

    /** Set a pixel on the window memory
     *  
     * @param x horizontal position
     * @param y vertical position
     * @param color 4 bit pixel color
     */
    virtual void pixel(int x, int y,int colour);

    /** Set a pixel on the window memory (alpha blend)
     *  
     * @param x horizontal position
     * @param y vertical position
     * @param color 4 bit pixel color
     */
    virtual void pixel_alpha(int x, int y,int colour);

    /** Fill the window memory with background color
     *
     */
    virtual void cls (void);

    /** draw a circle
     *
     * @param x0,y0 center
     * @param r radius
     * @param color 4 bit color
     *
     */
    void circle(int x, int y, int r, int colour); 

    /** draw a filled circle
     *
     * @param x0,y0 center
     * @param r radius
     * @param color 4 bit color
     */
    void fillcircle(int x, int y, int r, int colour); 

    /** draw a horizontal line
     *
     * @param x0 horizontal start
     * @param x1 horizontal stop
     * @param y vertical position
     * @param color 4 bit color
     *
     */
    void hline(int x0, int x1, int y, int colour);

    /** draw a vertical line
     *
     * @param x horizontal position
     * @param y0 vertical start 
     * @param y1 vertical stop
     * @param color 4 bit color
     */
    void vline(int y0, int y1, int x, int colour);

    /** draw a 1 pixel line
     *
     * @param x0,y0 start point
     * @param x1,y1 stop point
     * @param color 4 bit color
     *
     */
    void line(int x0, int y0, int x1, int y1, int colour);

    /** draw a rect
     *
     * @param x0,y0 top left corner
     * @param x1,y1 down right corner
     * @param color 4 bit color
     *
     */
    void rect(int x0, int y0, int x1, int y1, int colour);

    /** draw a filled rect
     *
     * @param x0,y0 top left corner
     * @param x1,y1 down right corner
     * @param color 4 bit color
     *
     */
    void fillrect(int x0, int y0, int x1, int y1, int colour);

    /** setup cursor position
     *
     * @param x x-position (top left)
     * @param y y-position 
     */
    virtual void locate(int x, int y);

    /** calculate the max number of char in a line
     *
     * @return max columns depends on actual font size
     *
     */
    virtual int columns(void);

    /** calculate the max number of columns
     *
     * @return max columndepends on actual font size
     *
     */
    virtual int rows(void);

    /** put a char on the screen
     *
     * @param value char to print
     * @return printed char
     *
     */
    virtual int _putc(int value);

    /** draw a character of selected font
     *
     * @param x x-position of char (top left) 
     * @param y y-position
     * @param c char to print
     *
     */
    virtual void character(int x, int y, int c);
    
    /** select the font
     *
     * @param f pointer to font array 
     *
     */
    void set_font(unsigned char* f);

    /** set transpalent effect
     *
     * @param mode trans mode
     *
     */
    void setTransMode( char mode );

    /** set a bitmap on the window buffer
     *
     * @param x,y upper left corner 
     * @param w width of bitmap
     * @param h high of bitmap
     * @param *bitmap pointer to the bitmap data
     *
     * @remarks bitmap format 4 bit R1 G1 B1
     * 
     */
    void Bitmap4bit( int x, int y, int w, int h, unsigned char *bitmap);

    /** set a bitmap on the window buffer
     *
     * @param x,y upper left corner 
     * @param w width of draw rect
     * @param h high of draw rect
     * @param *bitmap pointer to the bitmap data
     * @param bmp_x,bmp_y upper left corner at BMP files
     * @param bmp_w width of bitmap
     * @param bmp_h high of bitmap
     *
     * @remarks bitmap format 4 bit R1 G1 B1
     * 
     */
    void Bitmap4bit( int x, int y, int w, int h, unsigned char *bitmap, int bmp_x, int bmp_y, int bmp_w, int bmp_h );

    /** set a bitmap on the window buffer
     *
     * @param x,y upper left corner 
     * @param w width of bitmap
     * @param h high of bitmap
     * @param *bitmap pointer to the bitmap data
     *
     * @remarks bitmap format 1 bit
     * 
     */
    void Bitmap1bit( int x, int y, int w, int h, unsigned char *bitmap);

    /** set a bitmap on the window buffer
     *
     * @param x,y upper left corner 
     * @param w width of draw rect
     * @param h high of draw rect
     * @param *bitmap pointer to the bitmap data
     * @param bmp_x,bmp_y upper left corner at BMP files
     * @param bmp_w width of bitmap
     * @param bmp_h high of bitmap
     *
     * @remarks bitmap format 1 bit
     * 
     */
    void Bitmap1bit( int x, int y, int w, int h, unsigned char *bitmap, int bmp_x, int bmp_y, int bmp_w, int bmp_h );

    /** Transfer to the LCD from diaply buffer
     * 
     */
    void update();

    /** Transfer a 24 bit BMP to the LCD from filesytem
     *
     * @param x,y position of upper left corner 
     * @param *filenameBMP name of the BMP file
     *
     * @retval 1 succes
     * @retval 0 error
     *
     * @remarks bitmap format 24 bit R8 G8 B8
     * 
     */
    int BMP_24( int x,  int y, const char *filenameBMP );

    /** Command for setting the state of LCD
     *
     */
    void command_SetState();

    /** Command to clear whole the LCD
     *
     */
    void command_AllClear();

    /** Command to blink
     * @param mode Blinking Mode or Inversion Mode
     *
     */
    void setBlinkMode( char mode );

    /** Toggle the LCD polarity 
     *
     */
    void polling();

    void symbol(unsigned int x, unsigned int y, unsigned char *symbol);

protected:

    /* Instance of inteface */
    SPI         _spi;
    DigitalOut _cs; 
    DigitalOut _disp;
    DigitalOut _power;

    /* polarity variable */
    char        polarity;
    char        blink_cmd;

    /* trans mode variable */
    char        trans_mode;

    /* data for character variable */
    unsigned char* font;
    int char_x;
    int char_y;

    /* window  variable */
    int window_x;
    int window_y;
    int window_w;
    int window_h;

    /* temporary buffer */
    char    cmd_buf[LCD_DISP_WIDTH/2];                              /* for sending command */
    char    disp_buf[(LCD_DISP_WIDTH/2)*LCD_DISP_HEIGHT_MAX_BUF];   /* display buffer */
    char    file_buf[118];                                          /* for reading files */

    /** send command
     *
     * @param line_cmd cmannd data
     * @param line line of display
     *
     */
    void sendLineCommand( char* line_cmd, int line );
};

#endif /* MBED_COLOR_MEM_LCD_H */
