#ifndef SENSOR_SERVICE_H
#define SENSOR_SERVICE_H

#include "BLE.h"
#include "UUID.h"
//#include "Utils.h"

const UUID::LongUUIDBytes_t SENS_SERVICE_UUID = {0x00,0x00,0x00,0x00,0x00,0x01,0x11,0xe1,0x9a,0xb4,0x00,0x02,0xa5,0xd5,0xc5,0x1b}; //Primary service
const UUID::LongUUIDBytes_t VOLT_CHAR_UUID = {0x40,0x00,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b}; //Voltage characteristic
const UUID::LongUUIDBytes_t CURR_CHAR_UUID = {0x08,0x00,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b}; //Current characteristic

#define VOLT_DATA_LEN 2+4 //TimeStamp(uint16_t) + VoltageData(uint32_t)
#define CURR_DATA_LEN 2+4 //TimeStamp(uint16_t) + VoltageData(uint32_t)

class SensorService
{
    public:
       SensorService(BLE &_ble) :
            ble(_ble),
            voltage(), //six byte array
            voltageCharacteristic(VOLT_CHAR_UUID, voltage, VOLT_DATA_LEN, VOLT_DATA_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
            currentCharacteristic(CURR_CHAR_UUID, current, CURR_DATA_LEN, CURR_DATA_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY)
            {
                static bool serviceAdded = false; //To be sure to define just once
                if(serviceAdded)
                    { return; }
                GattCharacteristic *charTable[] = {&voltageCharacteristic, &currentCharacteristic};
                GattService sensorService(SENS_SERVICE_UUID, charTable, sizeof(charTable)/sizeof (GattCharacteristic *));

                ble.addService(sensorService);
                memset(voltage, 0, VOLT_DATA_LEN);
                memset(current, 0, CURR_DATA_LEN);
                serviceAdded = true;
            }
            
        //Voltage GATT server functions
            void sendVoltage (uint32_t volt_value, uint16_t TimeStamp)
            {
                memcpy(&voltage, &TimeStamp, 2);
                memcpy(&voltage+2, &volt_value, 4);
                uint16_t volt_handle = voltageCharacteristic.getValueAttribute().getHandle();
                printf("Send voltage handle: %d\n", volt_handle);
                memcpy(pastVoltage, voltage, VOLT_DATA_LEN);
                ble.gattServer().write(volt_handle, voltage, VOLT_DATA_LEN, 0);
                printf("Update\n");
            }
            
            void updateVoltage(uint32_t volt_value, uint16_t TimeStamp)
            {
                if(ble.getGapState().connected && isEnabledVoltageNotify)
                {
                    if(memcmp(&pastVoltage[2], &volt_value, 4) != 0)
                    {
                        sendVoltage(volt_value, TimeStamp);
                    }
                }
            }
            
          //Current GATT server functions
            void sendCurrent (uint32_t curr_value, uint16_t TimeStamp)
            {
                memcpy(&current, &TimeStamp, 2);
                memcpy(&current+2, &curr_value, 4);
                uint16_t curr_handle = currentCharacteristic.getValueAttribute().getHandle();
                printf("Send current handle: %d\n", curr_handle);
                memcpy(pastCurrent, current, CURR_DATA_LEN);
                ble.gattServer().write(curr_handle, current, CURR_DATA_LEN, 0);
                printf("Update\n");
            }
            
            void updateCurrent(uint32_t curr_value, uint16_t TimeStamp)
            {
                if(ble.getGapState().connected && isEnabledCurrentNotify)
                {
                    if(memcmp(&pastCurrent[2], &curr_value, 4) != 0)
                    {
                        sendCurrent(curr_value, TimeStamp);
                    }
                }
            }  
            
            
            void enNotify (Gap::Handle_t handle)
            {
 //               printf("\n\r\n\renNotify, Handle: 0x%02X\n", handle);
                if (isVoltageHandle(handle))
                {
                    isEnabledVoltageNotify = true;
                    memset(voltage,0,VOLT_DATA_LEN);
                    memset(pastVoltage, 0, VOLT_DATA_LEN);
                    return;
                }
                
                if (isCurrentHandle(handle))
                {
                    isEnabledCurrentNotify = true;
                    memset(current,0,CURR_DATA_LEN);
                    memset(pastCurrent, 0, CURR_DATA_LEN);
                    return;
                }                    
            }

    void disNotify (Gap::Handle_t handle)
            {
 //               printf("\n\r\n\rdisNotify, Handle: 0x%02X\n", handle);
                if (isVoltageHandle(handle)) { isEnabledVoltageNotify = false; memset(voltage,0,VOLT_DATA_LEN); return; }
                if (isCurrentHandle(handle)) { isEnabledCurrentNotify = false; memset(current,0,CURR_DATA_LEN); return; }          
            }
    
//----    
      
    bool isVoltageNotificationEn (void)
    {
            return isEnabledVoltageNotify;
    }  
 
            
    bool isVoltageHandle(Gap::Handle_t handle)
            {
                if(handle == voltageCharacteristic.getValueAttribute().getHandle())
                    { return true; }
                return false;
            }
            
     bool isCurrentNotificationEn (void)
    {
            return isEnabledCurrentNotify;
    }  
 
            
    bool isCurrentHandle(Gap::Handle_t handle)
            {
                if(handle == currentCharacteristic.getValueAttribute().getHandle())
                    { return true; }
                return false;
            }
    
    protected:
        BLE &ble;
        uint8_t voltage[VOLT_DATA_LEN];
        uint8_t pastVoltage[VOLT_DATA_LEN];
        uint8_t current[CURR_DATA_LEN];
        uint8_t pastCurrent[CURR_DATA_LEN];
        
        GattCharacteristic voltageCharacteristic;
        GattCharacteristic currentCharacteristic;
        
        bool                 isEnabledVoltageNotify;
        bool                 isEnabledCurrentNotify;
};

#endif