#ifndef __ACCELERATION_SERVICE_H__
#define __ACCELERATION_SERVICE_H__

#include "ble/BLE.h"

#define UUID_ACCELERATION_X_CHAR  "5D34E2F0-3DD2-07AC-38F2-BBAD120EF853"
#define UUID_ACCELERATION_Y_CHAR  "5D34E2F0-3DD2-07AC-38F3-BBAD120EF853"
#define UUID_ACCELERATION_Z_CHAR  "5D34E2F0-3DD2-07AC-38F4-BBAD120EF853"
#define UUID_ACCELERATION_SERVICE "5D34E2F0-3DD2-07AC-38F1-BBAD120EF853"
      
/**
* @class AccelerationService
* @brief BLE Acceleration Service. This service displays the acceleration
*/
class AccelerationService {
public:
    /**
     * @param[in] _ble
     *               BLE object for the underlying controller.
     */
    AccelerationService(BLE &_ble) :
        ble(_ble),
        accelerationIndex(0),
        accelerationXCharacteristic(UUID(UUID_ACCELERATION_X_CHAR), &temp, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        accelerationYCharacteristic(UUID(UUID_ACCELERATION_Y_CHAR), &temp, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        accelerationZCharacteristic(UUID(UUID_ACCELERATION_Z_CHAR), &temp, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY) {

        GattCharacteristic *charTable[] = {&accelerationXCharacteristic, &accelerationYCharacteristic, &accelerationZCharacteristic};
        GattService         AccelerationService(UUID(UUID_ACCELERATION_SERVICE), charTable, sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.addService(AccelerationService);
    }

    /**
     * @brief Update the temperature with a new value. Valid values lie between -2732 and above.
     *
     * @param newTemp
     *              Update to temperature.
     */
    void addAcceleration(int16_t *newAccel) {
        temp = newAccel[0];
        ble.gattServer().write(accelerationXCharacteristic.getValueHandle(), (const uint8_t*)&temp, sizeof(temp));
        temp = newAccel[1];
        ble.gattServer().write(accelerationYCharacteristic.getValueHandle(), (const uint8_t*)&temp, sizeof(temp));
        temp = newAccel[2];
        ble.gattServer().write(accelerationZCharacteristic.getValueHandle(), (const uint8_t*)&temp, sizeof(temp));
    }


protected:
    /**
     * A reference to the underlying BLE instance that this object is attached to.
     * The services and characteristics will be registered in this BLE instance.
     */
    BLE &ble;

    /**
     * The current temperature represented as 0.1 degrees Celsius.
     */
    //int16_t    acceleration[30][3];
    int16_t    temp;
    
    uint16_t   accelerationIndex;
    /**
     * A ReadOnlyGattCharacteristic that allows access to the peer device to the
     * temperature value through BLE.
     */
    ReadOnlyGattCharacteristic<int16_t> accelerationXCharacteristic;
    ReadOnlyGattCharacteristic<int16_t> accelerationYCharacteristic;
    ReadOnlyGattCharacteristic<int16_t> accelerationZCharacteristic;
};

#endif /* #ifndef __ACCELERATION_SERVICE_H__*/
