#ifndef __TEMPERATURE_SERVICE_H__
#define __TEMPERATURE_SERVICE_H__

#include "ble/BLE.h"

#define UUID_TEMPERATURE_CELSIUS_CHAR 0x2A1F
#define UUID_TEMPERATURE_SERVICE "5D34E2F0-3DD2-07AC-38F1-BBAD120EF852"
      

/**
* @class TemperatureService
* @brief BLE Temperature Service. This service displays the temperature, represented as an 16bit number.
*/
class TemperatureService {
public:
    /**
     * @param[in] _ble
     *               BLE object for the underlying controller.
     * @param[in] temperature
     *               16bit temperature in 0.1 degrees Celsius
     */
    TemperatureService(BLE &_ble, int16_t temperature = 200) :
        ble(_ble),
        temperature(temperature),
        temperatureCharacteristic(UUID_TEMPERATURE_CELSIUS_CHAR, &temperature, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY) {

        GattCharacteristic *charTable[] = {&temperatureCharacteristic};
        GattService         TemperatureService(UUID(UUID_TEMPERATURE_SERVICE), charTable, sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.addService(TemperatureService);
    }

    /**
     * @brief Update the temperature with a new value. Valid values lie between -2732 and above.
     *
     * @param newTemp
     *              Update to temperature.
     */
    void updateTemperature(int16_t newTemp) {
        temperature = newTemp;
        uint16_t temp = (uint16_t)temperature;
        ble.gattServer().write(temperatureCharacteristic.getValueHandle(), (const uint8_t*)&temp, sizeof(temperature));
    }

protected:
    /**
     * A reference to the underlying BLE instance that this object is attached to.
     * The services and characteristics will be registered in this BLE instance.
     */
    BLE &ble;

    /**
     * The current temperature represented as 0.1 degrees Celsius.
     */
    int16_t    temperature;
    /**
     * A ReadOnlyGattCharacteristic that allows access to the peer device to the
     * temperature value through BLE.
     */
    ReadOnlyGattCharacteristic<int16_t> temperatureCharacteristic;
};

#endif /* #ifndef __TEMPERATURE_SERVICE_H__*/