#include "mbed.h"
#include "seeed_can.h"

SEEED_CAN can;                                                          // No parameters needed when Seeed Studios' CAN-BUS Shield is plugged into a FRDM-KL25Z mbed
                                                                        // or an LPC1768b and using pins p9, p10, p11, p12 and p13.
Serial pc(USBTX, USBRX);                                                // USB serial port TTYACM0 (or COMn in a Windows environment - use device manager to find 'n')

Ticker minute;
Timer timestamp;

bool msgRxFlag = 0;

void canInterrupt()
{
    can.attach(NULL);                                                   // Disable the interrupt - the application must re-anable the interrupt after it has acted upon it
    msgRxFlag = 1;                                                      // Set a 'Flag' to say that there is an interrupt that needs to be processed
}

void resetTimestamp()                                                   // Resets the 'timestamp' Timer (attached to the 'minute' Ticker)
{
    timestamp.reset();
}

void displayMessage()                                                   // Display a CAN message if there is one in a receive buffer
{
    SEEED_CANMessage canMsg;

    if (can.read(canMsg)) {
        printf("*** T%05dI%03xL%dD", timestamp.read_ms(), canMsg.id, canMsg.len);   // Show a Timestamp in ms, message Id and message Length
        for (uint32_t i = 0; i < canMsg.len; i++)
            printf("%02x", canMsg.data[i]);                             // Show the message's data bytes in Hex representation
        printf(" ***\r");
    }
}

int main()
{
    SEEED_CANMessage canMsg;
    char VIN[18] = {NULL};                                              // VIN code is 17 characters long + 1 for string terminator character (NULL or \0)
    char reqvin[] = {0x02,0x1A,0x90,0x00,0x00,0x00,0x00,0x00};          // CAN message to request VIN code
    char floctl[] = {0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00};          // CAN message to request all remaining frames in a multi-frame message

    printf("Seeed Studios CAN-BUS Shield 'Hello World' program :-)\r\n");
    timestamp.start();
    minute.attach(&resetTimestamp, 60.0);                               // Reset the 'Timestamp' timer every minute
    can.open(500000, SEEED_CAN::Config);                                // Initialise Seeed Studios' CAN-BUS shield with a baudrate of 500 kbps (P-bus)
    can.mask(0, 0x7FF);                                                 // Configure Mask 0 to check all bits of a Standard CAN message Id
    can.mask(1, 0x7FF, CANStandard);                                    // Configure Mask 1 to check all bits of a Standard CAN message Id
    can.filter(0, 0x7E8);                                               // Configure Filter 0 - 0x7E8 is the id used by ECUs on GMLAN
// Read and Display the VIN code stored in a GMLAN ECU
//
// ***!!! NOTE: Using while(...) as I am here is not a good idea because  !!!***
// ***!!! this 'Hello World' will get stuck if the message never arrives  !!!***
// ***!!! I should really perform checking and include a timeout.         !!!***
// ***!!! It's sort of OK for a quick demo (Just don't show Nigel Jones)  !!!***
//
    can.write(SEEED_CANMessage(0x7E0, reqvin));                         // Request VIN using ReadDataByIdentifier method (GMLAN_DID)
    while (!(can.read(canMsg) && (canMsg.id == 0x7E8)));                // Wait for the response
    memcpy(VIN+0, canMsg.data+4, 4);                                    // 1st 4 Bytes are part of message protocol, last 4 bytes are 1st 4 characters of VIN
    can.write(SEEED_CANMessage(0x7E0, floctl));                         // Send Trionic8 a "Flow Control Message to get the rest of the VIN
    while (!(can.read(canMsg) && (canMsg.id == 0x7E8)));                // Wait for the 1st continuation message
    memcpy(VIN+4, canMsg.data+1, 7);                                    // 1st Byte is message continuation sequence number, last 7 bytes are next 7 characters of VIN
    while (!(can.read(canMsg) && (canMsg.id == 0x7E8)));                // Wait for the last message
    memcpy(VIN+11, canMsg.data+1, 6);                                   // 1st Byte is message continuation sequence number, last 6 bytes are remaining characters of VIN
    printf("VIN code: %s\r\n",VIN);
// Display all messages on the CAN-BUS
    can.monitor(1);                                                     // Select Moniter mode to listen only (do not ack messages on the CAN bus)
    can.mask(0, NULL);                                                  // Clear acceptance mask 0 (i.e. accept all meassages)
    can.attach(&canInterrupt, SEEED_CAN::RxAny);
    while (1) {
        if (msgRxFlag) {
            displayMessage();
            if (can.interrupts(SEEED_CAN::RxAny)) {
                msgRxFlag = 0;
                can.attach(&canInterrupt, SEEED_CAN::RxAny);
            }
        }
    }
}
