/*

Copyright (c) 2012-2014 RedBearLab

Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
and associated documentation files (the "Software"), to deal in the Software without restriction, 
including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, 
and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, 
subject to the following conditions:
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE 
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

/*
 *    The application works with the BlueJelly.js
 *
 *    http://jellyware.jp/ 
 *
 */
 
//======================================================================
//Grobal
//====================================================================== 
//------------------------------------------------------------
//Include Header Files
//------------------------------------------------------------ 
#include "mbed.h"
#include "ble/BLE.h"
#include "ADXL362.h"
 

//------------------------------------------------------------
//Definition
//------------------------------------------------------------ 
#define TXRX_BUF_LEN 20                     //max 20[byte]
#define DEVICE_LOCAL_NAME "ADXL362"         //Change Name
#define ADVERTISING_INTERVAL 160            //16 * 0.625[ms] = 100[ms]
#define TICKER_TIME 200000                  //200000[us] = 200[ms]
#define DIGITAL_OUT_PIN P0_9
//#define ANALOG_IN_PIN   P0_4

//Set SPI Pin
#define CS P0_10
#define MOSI P0_9
#define MISO P0_11
#define SCK P0_8


//------------------------------------------------------------
//Object generation
//------------------------------------------------------------ 
BLE ble;
DigitalOut      LED_SET(DIGITAL_OUT_PIN);
//AnalogIn        ANALOG(ANALOG_IN_PIN);

//SPI pin setting and change header file.(see ADXL362.h line185)
ADXL362 adxl362(CS, MOSI, MISO, SCK);

//------------------------------------------------------------
//Service & Characteristic Setting
//------------------------------------------------------------ 
//Service UUID
static const uint8_t base_uuid[] = { 0x71, 0x3D, 0x00, 0x00, 0x50, 0x3E, 0x4C, 0x75, 0xBA, 0x94, 0x31, 0x48, 0xF1, 0x8D, 0x94, 0x1E } ;

//Characteristic UUID
static const uint8_t tx_uuid[]   = { 0x71, 0x3D, 0x00, 0x03, 0x50, 0x3E, 0x4C, 0x75, 0xBA, 0x94, 0x31, 0x48, 0xF1, 0x8D, 0x94, 0x1E } ;
static const uint8_t rx_uuid[]   = { 0x71, 0x3D, 0x00, 0x02, 0x50, 0x3E, 0x4C, 0x75, 0xBA, 0x94, 0x31, 0x48, 0xF1, 0x8D, 0x94, 0x1E } ;

//Characteristic Value
uint8_t txPayload[TXRX_BUF_LEN] = {0,};
uint8_t rxPayload[TXRX_BUF_LEN] = {0,};

//Characteristic Property Setting etc
GattCharacteristic  txCharacteristic (tx_uuid, txPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);
GattCharacteristic  rxCharacteristic (rx_uuid, rxPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY| GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);
GattCharacteristic *myChars[] = {&txCharacteristic, &rxCharacteristic};

//Service Setting
GattService         myService(base_uuid, myChars, sizeof(myChars) / sizeof(GattCharacteristic *));


//======================================================================
//onDisconnection
//======================================================================
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    ble.startAdvertising();
}


//======================================================================
//onDataWritten
//======================================================================
void WrittenHandler(const GattWriteCallbackParams *Handler)
{   
    uint8_t buf[TXRX_BUF_LEN];
    uint16_t bytesRead;
    
    if (Handler->handle == txCharacteristic.getValueAttribute().getHandle()) 
    {
        ble.readCharacteristicValue(txCharacteristic.getValueAttribute().getHandle(), buf, &bytesRead);
        memset(txPayload, 0, TXRX_BUF_LEN);
        memcpy(txPayload, buf, TXRX_BUF_LEN); 
       
        if(buf[0] == 1)
            LED_SET = 1;
        else
            LED_SET = 0;
    }
}


//======================================================================
//onTimeout
//======================================================================


void m_status_check_handle(void)
{   
        //uint8_t x,y,z;
        uint8_t x[1];
        x[0] = adxl362.scanx_u8();
        //y=adxl362.scany_u8();
        //z=adxl362.scanz_u8();
        //printf("x = %x y = %x z = %x\r\n",x,y,z);
        //wait_ms(10);
    
        //Send out
        ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), x, 1); 
}


//======================================================================
//convert reverse UUID
//======================================================================
void reverseUUID(const uint8_t* src, uint8_t* dst)
{
    int i;
    
    for(i=0;i<16;i++)
        dst[i] = src[15 - i];
}


//======================================================================
//main
//======================================================================
int main(void)
{
    //ADXL362 Reset
    adxl362.reset();
    wait_ms(600);      // we need to wait at least 500ms after ADXL362 reset
    adxl362.set_mode(ADXL362::MEASUREMENT);
   
    uint8_t base_uuid_rev[16];

    //Timer Setting [us]
    Ticker ticker;
    ticker.attach_us(m_status_check_handle, TICKER_TIME);
    
    //BLE init
    ble.init();
    
    //EventListener
    ble.onDisconnection(disconnectionCallback);
    ble.onDataWritten(WrittenHandler);  

    //------------------------------------------------------------
    //setup advertising 
    //------------------------------------------------------------
    //Classic BT not support
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    
    //Connectable to Central
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    
    //Local Name
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME,
                                    (const uint8_t *)DEVICE_LOCAL_NAME, sizeof(DEVICE_LOCAL_NAME) - 1);
    
    //GAP AdvertisingData                                
    reverseUUID(base_uuid, base_uuid_rev);  
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                    (uint8_t *)base_uuid_rev, sizeof(base_uuid));
                                    
    //Advertising Interval 
    ble.setAdvertisingInterval(ADVERTISING_INTERVAL);

    //Add Service
    ble.addService(myService);
    
    //Start Advertising
    ble.startAdvertising(); 
    
    //------------------------------------------------------------
    //Loop
    //------------------------------------------------------------
    while(1)
    {
        ble.waitForEvent(); 
    }
}