#include "mbed.h"
#include "HIDScope.h"
#include "BiQuad.h"
#include "math.h"


Serial pc(USBTX, USBRX);


//Defining all in- and outputs
//EMG input
AnalogIn    emgBR( A0 );    //Right Biceps
AnalogIn    emgBL( A1 );    //Left Biceps

//LED output
DigitalOut  led_B( LED_BLUE );
DigitalOut  led_R( LED_RED );
DigitalOut  led_G( LED_GREEN );

//Setting Tickers for sampling EMG and determing if the threshold is met
Ticker      sample_timer;
Ticker      threshold_timer;

// Defining HIDScope ports needed
HIDScope    scope( 2 );


/* Defining all the different BiQuad filters, which contain a Notch filter,
High-pass filter and Low-pass filter. The Notch filter cancels all frequencies
between 49 and 51 Hz, the High-pass filter cancels all frequencies below 20 Hz
and the Low-pass filter cancels out all frequencies below 4 Hz */

/* Defining all the normalized values of b and a in the Notch filter for the
creation of the Notch BiQuad */

BiQuad      bqNotch1( 0.9876, -1.5981, 0.9876, -1.5981, 0.9752 );
BiQuad      bqNotch2( 0.9876, -1.5981, 0.9876, -1.5981, 0.9752 );

/* Defining all the normalized values of b and a in the High-pass filter for the
creation of the High-pass BiQuad */

BiQuad      bqHigh1( 0.8371, -1.6742, 0.8371, -1.6475, 0.7009 );
BiQuad      bqHigh2( 0.8371, -1.6742, 0.8371, -1.6475, 0.7009 );

/* Defining all the normalized values of b and a in the Low-pass filter for the
creation of the Low-pass BiQuad */

BiQuad      bqLow1( 6.0985e-4, 0.0012, 6.0985e-4, -1.9289, 0.9314 );
BiQuad      bqLow2( 6.0985e-4, 0.0012, 6.0985e-4, -1.9289, 0.9314 );

// Creating a variable needed for the creation of the BiQuadChain
BiQuadChain bqChain1;
BiQuadChain bqChain2;


//Defining all doubles needed in the filtering process
double emgBRfiltered;   //Right biceps Notch+High pass filter
double emgBRrectified;  //Right biceps rectified
double emgBRcomplete;   //Right biceps low-pass filter, filtering complete

double emgBLfiltered;   //Left biceps Notch+High pass filter
double emgBLrectified;  //Left biceps rectified
double emgBLcomplete;   //Left biceps low-pass filter, filtering complete


double threshold = 0.03;


/** Sample function
 * this function samples the BR EMG, filters the EMG and sends it to HIDScope
 **/
void EMG_sample()
{
    emgBRfiltered = bqChain1.step( emgBR.read() );   //Notch+High-pass
    emgBRrectified = fabs(emgBRfiltered);            //Rectification
    emgBRcomplete = bqLow1.step(emgBRrectified);     //Low-pass
    
    
   
    emgBLfiltered = bqChain2.step( emgBL.read() );    //Notch+High-pass
    emgBLrectified = fabs( emgBLfiltered );           //Rectification
    emgBLcomplete = bqLow2.step( emgBLrectified );    //Low-pass
    
    /* Set the sampled emg values in channel 0 (the first channel) and 1 
    (the second channel) in the 'HIDScope' instance named 'scope' */
    
    scope.set(1, emgBLcomplete );
    scope.set(0, emgBRcomplete );
    
       
    /*  Finally, send all channels to the PC at once */
    scope.send();
    /* To indicate that the function is working, the LED is toggled */
    led_B = !led_B;
    
    
}

double numsamples = 500;
double emgBRarray[500];
double emgBRsum;
double emgBRmeanMVC;
double thresholdBR;

double getThresholdBR()
{
    for (int i=0; i<numsamples; i++) {
        emgBRarray[i] = emgBRcomplete;
        emgBRsum = emgBRsum + emgBRarray[i];
        }
    
    emgBRmeanMVC = emgBRsum / numsamples;
    
    thresholdBR = emgBRmeanMVC * 0.3;
    return thresholdBR;
}
    
double getThresholdBL()
{
    double numsamples = 500;
    double emgBLarray[500];
    double emgBLsum;
    for (int i=0; i<numsamples; i++) {
        emgBLarray[i] = emgBLcomplete;
        emgBLsum = emgBLsum + emgBLarray[i];
        }
    
    double emgBLmeanMVC = emgBLsum / numsamples;
    
    double thresholdBL = emgBLmeanMVC * 0.3;
    return thresholdBL;
}
    

// Function to make the BiQuadChain for the Notch and High pass filter for all three filters    
void getbqChain()
{
    bqChain1.add(&bqNotch1).add(&bqHigh1);                 //Making the BiQuadChain
    bqChain2.add(&bqNotch2).add(&bqHigh2);
}


// Function to check if the threshold is met, with LED feedback for the user
void ThresholdReached()
{
    if (emgBRcomplete > thresholdBR) 
    {
        led_G = 0;
        led_R = 1;
        led_B = 1;
    }
    
    else if (emgBLcomplete > threshold)
    {
        led_G = 1;
        led_R = 0;
        led_B = 1;
    }
        
        
    else {
        led_G = 1;
        led_R = 1;
        led_B = 0; 
        }
        
}
  

int main()
{   
    /*Attach the 'sample' function to the timer 'sample_timer'.
    this ensures that 'sample' is executed every 0.002 seconds, to get a 
    sample frequency of 500 Hz
    */
    pc.baud(115200);
    getbqChain();
    sample_timer.attach(&EMG_sample, 0.002);
    thresholdBR = getThresholdBR();
    pc.printf("Threshold is %f.2\n", threshold);
    threshold_timer.attach(&ThresholdReached, 0.002);
            
    
    while(1) {}
}