/*------------------------------------------------------------------------------
Creator : Jonathan Wheadon
Date : 28/11/2018
Module : ELEC351
Project : ELEC351_GroupA
Dependencies : "SDReader.hpp" ; "Button.hpp" ; "General.hpp"
Purpose : To allow for uploading and downloading blocks of DATA to a SD card
connected via SPI
------------------------------------------------------------------------------*/

#include "SDReader.hpp"
#include "Terminal.hpp"

// Create switch to dismount the SD card
InterruptIn onBoardSwitch(USER_BUTTON);
Timeout onBoardSwitchTimeOut;             //Used to prevent switch bounce
int onBoardSwitchState = EDGE_FALLEN;     //Initial state for switch 1

// Create object SD of the SDReader class with set SPI pins
SDReader SD(PB_5, D12, D13, D10); // mosi, miso, sclk, cs   

// queue for SD reader events
EventQueue SDQueue(32 * EVENTS_EVENT_SIZE);

// Needed to print messgaes to terminal
extern EventQueue TerminalQueue;
extern Terminal PC;

extern Mail<mail_t, 32>SDreader_mail;

// Thread for saving DATA to the SD card
void SDThread(void)
{ 
    //Initial attempt of init SD to check if SD is pluged in
    if(SD.INIT() == false){
        TerminalQueue.call(&PC, &Terminal::updateSDstate, false);
    } else {
        TerminalQueue.call(&PC, &Terminal::updateSDstate, true); 
    } 
    
    while(1)
    {
        SDQueue.dispatch_forever();
    }
}

bool SDReader::INIT(void)
{    
    // call the SDBlockDevice instance initialisation method.
    if(sd.init() != 0){
        SDmounted = false;
    } else {
        SDmounted = true;
    }
    
    onBoardSwitch.rise(&SD, &SDReader::onBoardSwitchRisingEdge);
    
    if(SDmounted == false){
        return false;
    } else {           
        //Remove existing file
        remove( "/sd/ELEC351.csv" );
        
        FILE* fp = fopen("/sd/ELEC351.csv","a+");
        //Check file handle (stream)
        if (fp == NULL) {
            return true;
        }
            
        //Put some text in the file...
        fprintf(fp, " TIME STAMP, light, temperature, pressure\n");
        
        //Close the file
        fclose(fp);
        
        return true;
    }
}

void SDReader::upload(BYTE DATETIME[20], BYTE LIGHT[8], BYTE TEMP[8], BYTE PRESS[8])
{   
    FILE* fp = fopen("/sd/ELEC351.csv","a+");
    //Check file handle (stream)
    if (fp == NULL) {
        return;
    }
    
    //Put some text in the file...
    fprintf(fp, " %s, %s, %s, %s\n", DATETIME, LIGHT, TEMP, PRESS);
    
    //Close the file
    fclose(fp);
}

void SDReader::addDATA(void)
{
    //open mailbox and print to line in terminal
    osEvent evt = SDreader_mail.get();
    if (evt.status == osEventMail) {
        mail_t *mail = (mail_t*)evt.value.p;

        upload(mail->MDate, mail->MLIGHT, mail->MTEMP, mail->MPRESS);

        SDreader_mail.free(mail);
    }
}

void SDReader::unmount(void)
{
    sd.deinit();
    return;
}

void SDReader::update_SD_state(void){
    switch(SDmounted)
    {
        case false :
            if(SD.INIT() == false){
                SDmounted = false;
                TerminalQueue.call(&PC, &Terminal::updateSDstate, false);
            } else {
                SDmounted = true; 
                TerminalQueue.call(&PC, &Terminal::updateSDstate, true); 
            }  
        break;
        case true :
            SDmounted = false;
            SDQueue.call(&SD, &SDReader::unmount);
            TerminalQueue.call(&PC, &Terminal::updateSDstate, false);
        break;
    }
}

//Interrupt service routine for handling the timeout
void SDReader::onBoardSwitchTimeOutHandler() {
    onBoardSwitchTimeOut.detach();            //Stop the timeout counter firing

    //Which event does this follow?
    switch (onBoardSwitchState) {
        case EDGE_RISEN:    
            onBoardSwitch.fall(&SD, &SDReader::onBoardSwitchFallingEdge);  //Now wait for a falling edge
            SDQueue.call(&SD, &SDReader::update_SD_state); 
        break;
        case EDGE_FALLEN:
            onBoardSwitch.rise(&SD, &SDReader::onBoardSwitchRisingEdge);   //Now wait for a rising edge
        break;
    } //end switch 
}

//Interrupt service routine for a rising edge (press)
void SDReader::onBoardSwitchRisingEdge() {
    onBoardSwitch.rise(NULL);             //Disable detecting more rising edges
    onBoardSwitchState = EDGE_RISEN;      //Flag state
    onBoardSwitchTimeOut.attach(&SD, &SDReader::onBoardSwitchTimeOutHandler, 0.2);    //Start timeout timer
}

//Interrupt service routive for SW1 falling edge (release)
void SDReader::onBoardSwitchFallingEdge() {
    onBoardSwitch.fall(NULL);                         //Disable this interrupt
    onBoardSwitchState = EDGE_FALLEN;                 //Flag state
    onBoardSwitchTimeOut.attach(&SD, &SDReader::onBoardSwitchTimeOutHandler, 0.2);    //Start timeout counter    
}
