#ifndef MBED_CIRC_BUFFER_H
#define MBED_CIRC_BUFFER_H

#include "mbed.h"

#define OVERWRITE_ENABLE    1
#define SEEK_ENABLE         2

/** Classes circular_buffer_f and circular_buffer_c
*
*   Example use:-
* @code
* #include "mbed.h"
* #include "circular_buffer.h"
*  Serial pc(USBTX, USBRX); // Comms to Tera Term on pc
*  Serial gizmo(p28, p27); //  Serial port for gizmo on pins 27, 28
*  circular_buffer_c   chatter;   //  create  instance of circular_buffer_c for chars
*
* void    Rx_divert (void)  {          //  Gets attached to Serial::RxIrq interrupt handler
*    while   (gizmo.readable())         //  diverting rec'd chars into purpose provided,
*        chatter.write(gizmo.getc()); //  sufficiently sized, circular buffer
* }
*
*   main    ()  {
*   char buff[BUF_SIZE];           //  allocate memory for circular buffer_c
*   char x;
*       //  Use init function to initialise (no constructor functions here)
*   chatter.init(BUF_SIZE, buff, SEEK_ENABLE | OVERWRITE_ENABLE);
*
*   gizmo.attach   (Rx_divert, Serial::RxIrq); //  gizmo rx now diverted to circ buff
*   //  serial_in buffer now ready to use
*   while   (1) {
*       if  (chatter.readable())    {
*           chatter.read(&x);
*           pc.putc(x);
*       }
*   }
*  }
* @endcode
*/
class circular_buffer_f   {     //  Circular buffer of floats
    float   *buffbase, *Onbuff, *Offbuff, *buffend;
    int     buffsize;
    bool    emptyf, fullf,      // Buffer full and empty flags
            overwrite_enable,   // To allow new data to overwrite old, unread data
            seek_enable;        // To allow read pointer repositioning 'n' behind current newest
    public:

/** Returns true if stuff is in buffer to be read
*
*/
    bool    readable    ()  {return !emptyf;}

/** Returns true if space exists in buffer to write more
*
*/
    bool    writeable   ()  {return !fullf;}
    
/** Use init to setup buffer size, buffer mem address, and OVERWRITE_ENABLE and SEEK_ENABLE flags
*  If OVERWRITE_ENABLE is set, allows new data to overwrite old, unread data.
*  If OVERWRITE_ENABLE is clear, writes to a full buffer return false and new data is lost
*/
    void    init    (int size, float *buffstart, int flags);  // Size and address of buffer to work with

/** If SEEK_ENABLE flag set, sets read pointer to some distance_back in prep for multiple read using get_samps
*
*/
    bool    seek    (int distance_back) ;   // Assumes at least that much has been written to buff beforehand

/** Write places one sample into buffer. Returns false if buffer already full, true otherwise
*
*/
    bool    write   (float a)  ;     // Put value into circular buffer

/** Read one sample out from buffer. Returns false if nothing to be read, true otherwise
*
*/
    bool    read    (float *rd)  ;


/** get_samps reads len samples from buffer into destination array
*
*/
    bool    get_samps (float *dest, int len) ;
}  ; // end of class circular_buffer_f



class circular_buffer_c   {     //  Circular buffer of char
    char *buffbase, *Onbuff, *Offbuff, *buffend;
    int buffsize;
    bool    emptyf, fullf,      // Buffer full and empty flags
            overwrite_enable,   // To allow new data to overwrite old, unread data
            seek_enable;        // To allow read pointer repositioning 'n' behind current newest
    public:
    bool    readable    ()  {return !emptyf;}
    bool    writeable   ()  {return !fullf;}
    
    void    init    (int size, char *buffstart, int flags);  // Size and address of buffer to work with
    bool    seek    (int distance_back) ;   // Assumes at least that much has been written to buff beforehand
    bool    write   (char a)  ;     // Put value into circular buffer
    bool    read    (char *rd)  ;
    bool    get_samps (char *dest, int len) ;
}  ; // end of class circular_buffer_c

#endif
