/* mbed library for touchscreen connected to 4 mbed pins
 * derive from SPI_TFT lib 
 * Copyright (c) 2011 Peter Drescher - DC2PD
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef MBED_TOUCH_H
#define MBED_TOUCH_H

#include "mbed.h"
#include "SPI_TFT.h"

struct point{
       unsigned short x;
       unsigned short y;
       };


/** touchscreen control class, based on SPI_TFT
 *
 * Example:
 * @code
 * 
 * #include "mbed.h"
 * #include "SPI_TFT.h"
 * #include "Arial12x12.h"
 * #include "Arial28x28.h"
 * #include "touch_tft.h"
 * // the TFT is connected to SPI pin 5-7
 * // the touch is connected to 19,20,16,17
 *
 * touch_tft tt(p19,p20,p16,p17,p5, p6, p7, p8, p15,"TFT"); // x+,x-,y+,y-,mosi, miso, sclk, cs, reset
 *
 * int main() {
 * point p;
 *
 *  tt.claim(stdout);        // send stdout to the TFT display
 *  tt.background(Black);    // set background to black
 *  tt.foreground(White);    // set chars to white
 *  tt.cls();                // clear the screen
 *  tt.set_font((unsigned char*) Arial12x12);  // select the font
 *  tt.set_orientation(1);
 *
 *  tt.calibrate();          // calibrate the touch
 * while (1) {
 *       p = tt.get_touch();   // read analog pos.
 *       if (tt.is_touched(p)) {  // test if touched
 *           p = tt.to_pixel(p);             // convert to pixel pos
 *           tt.fillcircle(p.x,p.y,3,Blue);  // print a blue dot on the screen
 *     }
 * }
 * @endcode
 */
class touch_tft : public  SPI_TFT{
public:
    /** create a TFT with touch object connected to the pins:
     *
     * @param pin xp resistiv touch x+
     * @param pin xm resistiv touch x-
     * @param pin yp resistiv touch y+
     * @param pin ym resistiv touch y-
     * @param mosi,miso,sclk SPI connection to TFT
     * @param cs pin connected to CS of display
     * @param reset pin connected to RESET of display 
     * based on my SPI_TFT lib
     */
    touch_tft(PinName xp, PinName xm, PinName yp, PinName ym,PinName mosi, PinName miso, PinName sclk, PinName cs, PinName reset,const char* name ="TFT");
    
    /** calibrate the touch display
     *
     * User is asked to touch on two points on the screen
     */   
    void calibrate(void);
    void noncalibrate(void);
    
    /** read x and y analog samples
     *
     * @returns point(x,y)
     */
    point get_touch(void);
    
    /** calculate coord on screen
     *
     * @param a_point point(analog x, analog y)
     * @returns point(pixel x, pixel y)
     *
     */
    point to_pixel(point a_point);
    
    /** test if screen is touched
     * 
     * @param point analog x,y
     * @returns true is touched
     *
     */   
    bool is_touched(point a);    
    
protected:
    DigitalInOut _xp;
    DigitalInOut _xm;
    DigitalInOut _yp;
    DigitalInOut _ym;
    AnalogIn     _ax;
    AnalogIn     _ay;
    PinName xa;
    PinName ya;
    
    
    unsigned short x_a,y_a;
    unsigned short x_off,y_off;
    unsigned short pp_tx,pp_ty;
    
    
           
    };

#endif