//Engine4/main.c J.Miller 03/22/2015
// 8-Cylinder Solenoid Engine Ignition controller


#include "mbed.h"

#define EncRes              256
#define DefaultGap          5

#define k_Quad1             57      // Ideal: 64  ~EncRes/4
#define k_Quad2             126     // Ideal: 128 ~EncRes/2
#define k_Quad3             192     // Ideal: 192 ~EncRes*3/4
#define k_Quad4             255     // ~0 (or 255)

// Modes
#define OPENLOOP            1       // Green indicator
#define CLOSEDLOOP          2       // Blue
#define TUNE                3       // Red

#define PWMPeriod           500     //microseconds
#define DefaultPWMPercent   0.25f

#define LOOP_RATE           0.03333 // 30 Hz
#define Kp_initial          0.01    // Proportional Speed ERROR Gain Constant     
#define Ki_initial          0.001   // Intergral Speed ERROR Gain Constant
#define Kd_initial          0.001   // Derivative Speed Gain Constant (not Error)
#define Kpff_initial        0.0     // ThrottlePostion Feedforward

#define THROTTLE_INCREMENT  4
#define MIN_THROTTLE_POS    0       // 4 is about 0.5 RPS, zero stops motor
#define MAX_THROTTLE_POS    150     // ~1000 RPM

#define LED_On              0       // LEDs hava common Anode connection
#define LED_Off             1       // 
#define DEBOUNCE_COUNT      500     // Loop interations to debounce user push button switch



Ticker DriveSignal_Update;      // Real-time interrupt at LOOP_RATE Intervals
Serial pc(PTE0, PTE1);
DigitalOut SerialCommon(PTD7);

// on-board stuff
DigitalOut LEDRed(PTB18);
DigitalOut LEDGreen(PTB19);
DigitalOut LEDBlue(PTD1);
DigitalOut TestPoint(PTB8);

// Engine Cyclider Outputs
// using only Port D&E ended PWM conflicts
PwmOut Cyl_1(PTE31); 
PwmOut Cyl_2(PTD3);
PwmOut Cyl_3(PTD2);
PwmOut Cyl_4(PTD0);
PwmOut Cyl_5(PTD5); 
PwmOut Cyl_6(PTE22);   // (PTA13)was not a team player; 
PwmOut Cyl_7(PTE20);   //(PTC9)was not a team player;;
PwmOut Cyl_8(PTE21);   //(PTC8)was not a team player;; 


// User Rotary Encoder definition
InterruptIn     UserEnc1A(PTD4);
DigitalIn       UserEnc1B(PTC3);
DigitalIn       PB1(PTC4);
DigitalOut      Red(PTA4);
DigitalOut      Green(PTA5);
DigitalOut      Blue(PTA12);
DigitalOut      Anode(PTC6);
DigitalOut      Common(PTC5);
//--------------------------------


//--- Crank Encode Definition ----
InterruptIn  CrankAngleEncA(PTA1);
DigitalIn    CrankAngleEncB(PTC7);
//--------------------------------


Timer t;

char     Mode = OPENLOOP;
char     CylinderFired = 0;
uint8_t  CrankAngle, CrankAngleChange;
uint8_t  Gap = DefaultGap;

uint8_t Quad1 = k_Quad1;
uint8_t Quad2 = k_Quad2;
uint8_t Quad3 = k_Quad3;
uint8_t Quad4 = k_Quad4;

// PID Variables
float    ThrottlePosition = 0.0F;
float    PreviousThrottlePosition = 0.0F;
float    Error;
uint16_t Speed;                 // in encoder pulses/LOOP_RATEfloat Kp    = Kp_initial;
float    Kp    = Kp_initial;
float    Ki    = Ki_initial;
float    Kd    = Kd_initial;
float    Kpff  = Kpff_initial;
float    DriveSignal;
float    IntergalError, DeltaSpeed, PreviousError, PreviousSpeed;
uint8_t  DriveUpdate;
// Measured Benchmarks and PID calculations
// Max Speed: Currently about 15RPS, 923RPM, 65ms/rev, 254uS/Encoder pulse or 3937pps
//          that's 131pulses/LOOP_RATE. 150p/LR is 17.57rps or 1054rpm
// Min Speed: Would be good to get down to 0.5rps, 120rpm, 128pps
//          that's 4.3pulses/LOOP_RATE


char     Ramp_Dir = 0;
uint8_t  PreviousShaftAngle1, PreviousShaftAngle2, ShaftAngle1, ShaftAngle2;
uint8_t  PB1State, BottomDeadCenter;
uint16_t DebounceCounter;


// User Encoder wheel interrupt service call
void UserEnc1_pulse(void)
{
    if ((UserEnc1B == 1) && (ThrottlePosition < MAX_THROTTLE_POS) )
        ThrottlePosition += THROTTLE_INCREMENT;
        if (ThrottlePosition > MAX_THROTTLE_POS) ThrottlePosition = MAX_THROTTLE_POS;
        
    else if ((UserEnc1B == 0) && (ThrottlePosition > MIN_THROTTLE_POS ) )
        ThrottlePosition -= THROTTLE_INCREMENT;    
        if (ThrottlePosition < 0.0f) ThrottlePosition = 0.0F;
       
}

// Crank Shaft angle encoder interrupt service call
void CrankAngleEnc_pulse(void)
{
    if (CrankAngleEncB) {
        CrankAngle++;
        Speed++;
        }
    else {
        CrankAngle--;
        Speed--;
        }
            
   CrankAngleChange = true;         
}



// Update Motor Drive Signal. This is a periodic interrupt event call
void Update_DriveSignal(void) {
        switch (Mode) {
            case OPENLOOP: 
            case TUNE: 
                    if  (ThrottlePosition != PreviousThrottlePosition) {
                        DriveSignal = ThrottlePosition / MAX_THROTTLE_POS;
                        PreviousThrottlePosition = ThrottlePosition;
                        DriveUpdate = true;
                    }
               break;
                                
            case CLOSEDLOOP: 
                    if (ThrottlePosition > 0.0F) {
                        
                        // PID Compensation Loop.  Measured 11us to compute solution
                        Error = ThrottlePosition - Speed;
                        
                        if (((PreviousError * Error) < 0.0F) || (ThrottlePosition > 50) ) {
                            IntergalError = 0.0F;      // Reset Intergal at zero-crossing to prevent wind-up related issues
                            TestPoint = 1;
                        }
                        
                            
                        DeltaSpeed = Speed - PreviousSpeed;
                                
                        DriveSignal =       Kp   * Error 
                                        +   Ki   * IntergalError
                                        -   Kd   * DeltaSpeed
                                        +   Kpff * ThrottlePosition ;
                        
                        IntergalError += Error;
                        PreviousError  = Error;    
                        PreviousSpeed  = Speed;
            
                        if (DriveSignal > 1.0F ) DriveSignal = 1.0F;
                        if (DriveSignal < 0.0F ) DriveSignal = 0.0F;
                  
                        Speed = 0;
                        }
                        
                    else    {                       // coast to stop
                        DriveSignal     = 0.0F;
                        IntergalError   = 0.0F;
                        PreviousError   = 0.0F;    
                        PreviousSpeed   = Speed;
                        }
                    DriveUpdate     = true; // forces update
                break;

        }
        
        TestPoint = 0;

//    pc.printf("%d,%f\r\n\r",Ramp_Dir,ThrottlePosition);
}

//void DispUpdate(void) {
//   pc.printf("%d : %d : %d\r\n\r",CrankAngle,CylindersFired1,CylindersFired2);}

int main() {
    
    //-------------- Initialization  -------------------
    Anode = 1 ; Common = 0;         // Creates a local +3.3V/GND near the User Encoder port pins
    SerialCommon = 0;
    CrankAngleChange = true;
    ThrottlePosition = 0;
    char c;                         // Serial character input during tuning mode
    LEDRed = LED_On; LEDGreen = LED_Off; LEDBlue = LED_Off;
    
    Cyl_1.period_us(PWMPeriod) ; 
    Cyl_2.period_us(PWMPeriod) ; 
    Cyl_3.period_us(PWMPeriod) ; 
    Cyl_4.period_us(PWMPeriod) ; 
    Cyl_5.period_us(PWMPeriod) ; 
    Cyl_6.period_us(PWMPeriod) ; 
    Cyl_7.period_us(PWMPeriod) ; 
    Cyl_8.period_us(PWMPeriod) ; 

    pc.baud(57600);
    pc.printf("V8-Solenoid Engine..........");
   
    Cyl_1 = 0; Cyl_2 = 0; Cyl_3 = 0; Cyl_4 = 0; 
    Cyl_5 = 0; Cyl_6 = 1.0f; Cyl_7 = 0; Cyl_8 = 0;  //Cyl-6 ON to help startup
    wait_ms(500);
    LEDRed = LED_Off; LEDGreen = LED_Off; LEDBlue = LED_On;

    Cyl_1 = 0;  Cyl_2 = 1.0F;Cyl_3 = 0;   Cyl_4 = 0;   Cyl_5 = 1.0F;Cyl_6 = 0;   Cyl_7 = 0;   Cyl_8 = 0; 
      wait_ms(1000); 
      CrankAngle = 0;
      Quad2 = 128;

    CrankAngleEncA.rise(&CrankAngleEnc_pulse);

    Cyl_1 = 0;  Cyl_2 = 0;   Cyl_3 = 0;   Cyl_4 = 0;   Cyl_5 = 0;   Cyl_6 = 1.0f;Cyl_7 = 1.0F;Cyl_8 = 0; 
      wait_ms(1000);
      Quad3 = CrankAngle + 128;

    Cyl_1 = 0;  Cyl_2 = 0;   Cyl_3 = 1.0F;Cyl_4 = 0;   Cyl_5 = 0;   Cyl_6 = 0;   Cyl_7 = 0;   Cyl_8 = 1.0F; 
      wait_ms(1000);
      if (CrankAngle > 128) BottomDeadCenter = CrankAngle - 128; else BottomDeadCenter = CrankAngle + 128; 
      Quad4 = CrankAngle + 128;

    Cyl_1 = 1.0F;Cyl_2 = 0;  Cyl_3 = 0;   Cyl_4 = 1.0F;Cyl_5 = 0;   Cyl_6 = 0;   Cyl_7 = 0;   Cyl_8 = 0; 
      wait_ms(1000);
      Quad1 = CrankAngle + 128;                      
    
    pc.printf("Q1:%d, Q2:%d, Q3:%d, Q4:%d \r\n",Quad1,Quad2,Quad3,Quad4);

    //Cyl_2 = 1.0f; Cyl_5 = 1.0f;Cyl_6 = 0;
    //wait_ms(1000);
    //CrankAngle = 0;
    
    LEDRed = LED_Off; LEDGreen = LED_On; LEDBlue = LED_Off;
    Cyl_1 = 0; Cyl_4 = 0;
    DriveUpdate = true;
   
    DriveSignal_Update.attach(&Update_DriveSignal,LOOP_RATE);
    UserEnc1A.rise(&UserEnc1_pulse);

    Red = LED_Off; Green=LED_On; Blue = LED_Off;

    pc.printf("Initialized!\r\n");

    //-------------- Initialization Complete -------------------


    // Main LOOP...............

    while(true) {
        
        

        // Determine which cylinders to fire base on crank angle. To "fire" in this case means that a solenoid
        // will be pulling its crank shaft slug inward.
        // Electric solenoids are equivalent to a two-stroke cylinder therefore they can fire every cycle. 
        // Each solenoid (cylinder) can fire for an entire 1/4 rotation, or 90-degrees. Since there are 8 cylinders
        // two are always firing at any given moment. there are two sets two cylinder arrangements, as follows:
        //  Crank Shaft Angle   Set A cylinders     Set B Cylinders
        //  0                      6&7  1              *OFF* 0
        //  1-89                   6&7  1               3&8  1  
        //  90                    *OFF* 0               3&8  1
        //  91-179                 1&4  2               3&8  1
        //  180                    1&4  2              *OFF* 0
        //  181-269                1&4  2               2&5  2
        //  270                   *OFF* 0               2&5  2
        //  271-359                6&7  1               2&5  2
        // note: CrankAngle variable in this code(Encoder Counts) =  0.711 * real Crank Angle
        
        // The states used below to assign firing order are are follows
        // State 1 is 6&7 ON for Set A and 3&8 ON for Set B
        // State 2 is 1&4 ON for Set A and 2&5 ON for Set B
        // State 0 is all *OFF* (coasting) while these cylinders are cresting at the top of thier stroke angle        
        if(CrankAngleChange){   
             
            if      ( (CrankAngle < (Quad1-Gap))  ||  (CrankAngle>(Quad3+Gap)) ) // Pair#2  270 to 90 deg  
                ShaftAngle1 = 1;
            else if ((CrankAngle>(Quad1+Gap))  &&  (CrankAngle<(Quad3-Gap)))     // Pair#4  90  to 270  
                ShaftAngle1 = 2;         
            else
                ShaftAngle1 = 0;

              
           
            if      ((CrankAngle>Gap)   &&   (CrankAngle<(Quad2-Gap)))          // Pair#3  0   to 180 deg  
                ShaftAngle2 = 1;
            else {
                 if ((Quad4-Gap) < 128)     // Quad4 must have wrapped around the 8-bit field
                    if ((CrankAngle>(Quad2+Gap))   ||   (CrankAngle<(Quad4-Gap)))  // Pair#1  180 to 360 deg  
                        ShaftAngle2 = 2;
                    else
                        ShaftAngle2 = 0;
                 else                     
                    if ((CrankAngle>(Quad2+Gap))   &&   (CrankAngle<(Quad4-Gap)))  // Pair#1  180 to 360 deg  
                        ShaftAngle2 = 2;
                    else
                        ShaftAngle2 = 0;
                 }   
        }

        //Only if states change will the Cylinder assignments change

        if  ( (PreviousShaftAngle1 != ShaftAngle1) || DriveUpdate) {
            if  (ShaftAngle1 == 1) {
                Cyl_1 = 0; Cyl_4 = 0;
                Cyl_7 = DriveSignal; Cyl_6 = DriveSignal;
            }  
            else if (ShaftAngle1 == 2)  {
                Cyl_7 = 0; Cyl_6 = 0;
                Cyl_4 = DriveSignal; Cyl_1 = DriveSignal;
            }            
            else {  
                Cyl_1 = 0; Cyl_4 = 0; Cyl_6 = 0; Cyl_7 = 0;
            }
            PreviousShaftAngle1 = ShaftAngle1;
        }
            
        if  ( (PreviousShaftAngle2 != ShaftAngle2) || DriveUpdate) {
            if      (ShaftAngle2 == 1) {    
                Cyl_2 = 0; Cyl_5 = 0;
                Cyl_8 = DriveSignal; Cyl_3 = DriveSignal;
            }
            else if (ShaftAngle2 == 2) {
                Cyl_3 = 0; Cyl_8 = 0;
                Cyl_2 = DriveSignal; Cyl_5 = DriveSignal;
            }             
            else {
                Cyl_3 = 0; Cyl_8 = 0; Cyl_2 = 0; Cyl_5 = 0;
            }
            PreviousShaftAngle2 = ShaftAngle2;
        }
        
        //if(DriveUpdate) pc.printf("%d%d.",ShaftAngle1,ShaftAngle2); // debug signals
              
        DriveUpdate = false;   
              

 
 
        // Monitor and process user pushbutton input
        if ((PB1 == 1) && (PB1State == 0)) {
            DebounceCounter--;
            if ((PB1 == 1) && (DebounceCounter == 0)){
                                
                switch (Mode) {
                    case OPENLOOP: 
                        Mode = CLOSEDLOOP;
                        Red = LED_Off; Green=LED_Off; Blue = LED_On;
                        pc.printf("Closed Loop Mode\r\n");
                        pc.printf(" Kp         :%f\r\n Ki         :%f\r\n Kd         :%f\r\n Kpff       :%f\r\n Throttle   :%f\r\n Error      :%f\r\n DriveSignal:%f\r\n------------------------------\r\n",Kp,Ki,Kd,Kpff,ThrottlePosition,Error,DriveSignal);
                        ThrottlePosition = 0;
                        break;
                        
                    case CLOSEDLOOP: 
                        Mode = TUNE;
                        Red = LED_On; Green = LED_Off; Blue = LED_Off;
                        pc.printf("Tune Mode\r\n");
                        pc.printf(" Kp         :%f\r\n Ki         :%f\r\n Kd         :%f\r\n Kpff       :%f\r\n Throttle   :%f\r\n Error      :%f\r\n DriveSignal:%f\r\n------------------------------\r\n",Kp,Ki,Kd,Kpff,ThrottlePosition,Error,DriveSignal);
                        DriveSignal      = 0.0F; 
                        ThrottlePosition = 0;
                        DriveUpdate     = true; // forces update
                        break;
                        
                    case TUNE: 
                        Mode = OPENLOOP;
                        Red = LED_Off; Green=LED_On; Blue = LED_Off;
                        pc.printf("Opened Loop Mode\r\n");
                        pc.printf(" Kp         :%f\r\n Ki         :%f\r\n Kd         :%f\r\n Kpff       :%f\r\n Throttle   :%f\r\n Error      :%f\r\n DriveSignal:%f\r\n------------------------------\r\n",Kp,Ki,Kd,Kpff,ThrottlePosition,Error,DriveSignal);
                        ThrottlePosition = 0;
                        break;
                }

                PB1State = 1;
            }        
        }
        else if (PB1 == 0) {
            DebounceCounter = DEBOUNCE_COUNT; // DebounceCounter and PB1State are part of the switches debounce logic
            PB1State = 0;
        }        
 
      
      if ((Mode == TUNE)&& pc.readable()) {
                
            c = pc.getc();

            if((c == 'p') && (Kp > 0.00F)) {
                Kp -= 0.001F;
            }
            if(c == 'P') {
                Kp += 0.001F;
            }
            
            if((c == 'i') && (Ki > 0.0F)) {
                Ki -= 0.001F;
            }
            if(c == 'I') {
                Ki += 0.001F;
            }
            
            if((c == 'd') && (Kd > 0.0F)) {
                Kd -= 0.001F;
            }
            if(c == 'D') {
                Kd += 0.001F;
            }
            
            if((c == 'f') && (Kpff > 0.0F)) {
                Kpff -= 0.001F;
            }
            if(c == 'F') {
                Kpff += 0.001F;
            }    
            if((c == 'g') && (Gap > 0)) {
                Gap -= 1;
            }
            if(c == 'G') {
                Gap += 1;
            }
            pc.printf(" Kp         :%f\r\n Ki         :%f\r\n Kd         :%f\r\n Kpff       :%f\r\n------------------------------\r\n",Kp,Ki,Kd,Kpff);
            pc.printf("                      -1-  -2-  -3-  -4-  -5-  -6-  -7-  -8-\r\n");
            pc.printf(" CrankAngle:%d,  Cyl:%1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f,  ThottlePos:%1.2f\r\n",
                CrankAngle,Cyl_1.read(), Cyl_2.read(), Cyl_3.read(), Cyl_4.read(),Cyl_5.read(),Cyl_6.read(),Cyl_6.read(),Cyl_7.read(),ThrottlePosition);     
                
           if(c == 'Q') {  // Find Top-Dead-Center of each cylinder pair in order to find the ideal firing quadrants
                   Cyl_1 = 1.0F;Cyl_2 = 0;  Cyl_3 = 0;   Cyl_4 = 1.0F;Cyl_5 = 0;   Cyl_6 = 0;   Cyl_7 = 0;   Cyl_8 = 0; 
                      wait_ms(1000);
                      BottomDeadCenter = CrankAngle + 128; 
                      pc.printf("Ideal Quad1 = %d\r\n",BottomDeadCenter);                      
                   Cyl_1 = 0;  Cyl_2 = 1.0F;Cyl_3 = 0;   Cyl_4 = 0;   Cyl_5 = 1.0F;Cyl_6 = 0;   Cyl_7 = 0;   Cyl_8 = 0; 
                      wait_ms(1000); 
                      BottomDeadCenter = CrankAngle + 128; 
                      pc.printf("Ideal Quad2 = %d\r\n",BottomDeadCenter);
                    Cyl_1 = 0;  Cyl_2 = 0;   Cyl_3 = 0;   Cyl_4 = 0;   Cyl_5 = 0;   Cyl_6 = 1.0f;Cyl_7 = 1.0F;Cyl_8 = 0; 
                      wait_ms(1000);
                      BottomDeadCenter = CrankAngle + 128; 
                      pc.printf("Ideal Quad3 = %d\r\n",BottomDeadCenter);
                    Cyl_1 = 0;  Cyl_2 = 0;   Cyl_3 = 1.0F;Cyl_4 = 0;   Cyl_5 = 0;   Cyl_6 = 0;   Cyl_7 = 0;   Cyl_8 = 1.0F; 
                      wait_ms(1000);
                      BottomDeadCenter = CrankAngle + 128; 
                      pc.printf("Ideal Quad4 = %d\r\n",BottomDeadCenter);
           }    
           if (c == 'C') { //Cylinder test
               pc.printf("Cylinder Test Mode\r\n");
               Cyl_1 = 0; Cyl_2 = 0; Cyl_3 = 0; Cyl_4 = 0; Cyl_5 = 0; Cyl_6 = 0; Cyl_7 = 0; Cyl_8 = 0;
               c = ' ';
               while ((Mode == TUNE)&&(c != '0')) {
                
                   if (pc.readable()) {
                       c = pc.getc();
                       switch (c) {
                           case '1':
                               Cyl_1 = 1.0F; Cyl_2 = 0; Cyl_3 = 0; Cyl_4 = 0; Cyl_5 = 0; Cyl_6 = 0; Cyl_7 = 0; Cyl_8 = 0;
                               break;
                           case '2':
                               Cyl_1 = 0; Cyl_2 = 1.0F; Cyl_3 = 0; Cyl_4 = 0; Cyl_5 = 0; Cyl_6 = 0; Cyl_7 = 0; Cyl_8 = 0;
                               break;
                           case '3':
                               Cyl_1 = 0; Cyl_2 = 0; Cyl_3 = 1.0F; Cyl_4 = 0; Cyl_5 = 0; Cyl_6 = 0; Cyl_7 = 0; Cyl_8 = 0;
                               break;
                           case '4':
                               Cyl_1 = 0; Cyl_2 = 0; Cyl_3 = 0; Cyl_4 = 1.0F; Cyl_5 = 0; Cyl_6 = 0; Cyl_7 = 0; Cyl_8 = 0;
                               break;
                           case '5':
                               Cyl_1 = 0; Cyl_2 = 0; Cyl_3 = 0; Cyl_4 = 0; Cyl_5 = 1.0F; Cyl_6 = 0; Cyl_7 = 0; Cyl_8 = 0;
                               break;
                           case '6':
                               Cyl_1 = 0; Cyl_2 = 0; Cyl_3 = 0; Cyl_4 = 0; Cyl_5 = 0; Cyl_6 = 1.0F; Cyl_7 = 0; Cyl_8 = 0;
                               break;
                           case '7':
                               Cyl_1 = 0; Cyl_2 = 0; Cyl_3 = 0; Cyl_4 = 0; Cyl_5 = 0; Cyl_6 = 0; Cyl_7 = 1.0F; Cyl_8 = 0;
                               break;
                           case '8':
                               Cyl_1 = 0; Cyl_2 = 0; Cyl_3 = 0; Cyl_4 = 0; Cyl_5 = 0; Cyl_6 = 0; Cyl_7 = 0; Cyl_8 = 1.0F;                            
                       }
                   }
               }
               Cyl_1 = 0; Cyl_2 = 0; Cyl_3 = 0; Cyl_4 = 0; Cyl_5 = 0; Cyl_6 = 0; Cyl_7 = 0; Cyl_8 = 0;
           }
           
        }
      
      
    }//end of main loop - while(true)
    
}//end of main

