#ifndef MCP9808_HPP
#define MCP9808_HPP

#include <stdint.h>
#include "mbed.h"

/** A driver interface for the Microchip MCP9808 digital temperature sensor
 *
 *  Example:
 *  @code
 *
 *  #include "mbed.h"
 *
 *  MCP9808 sensor(I2C_SDA, I2C_SCL, 0x30);
 *
 *  int main() {
 *      while (true) {
 *          sensor.wake();
 *          osDelay(250);
 *          printf("Current temperature: %f °C\r\n", sensor.get_temperature());
 *          sensor.sleep();
 *          osDelay(1000);
 *      }
 *  }
 *  @endcode
 */
class MCP9808
{
public:

    /** Register definitions
     */
    enum Register {
        MCP9808_REG_RFU =              0x00,
        MCP9808_REG_CFG =              0x01,
        MCP9808_REG_ALERT_TEMP_UPPER = 0x02,
        MCP9808_REG_ALERT_TEMP_LOWER = 0x03,
        MCP9808_REG_CRIT_TEMP =        0x04,
        MCP9808_REG_TEMP =             0x05,
        MCP9808_REG_MFG_ID =           0x06,
        MCP9808_REG_DEV_ID =           0x07,
        MCP9808_REG_RES =              0x08,
    };

    /** Config register flags
     */
    enum ConfigFlag {
        MCP9808_CFG_FLAG_ALERT_MODE_INT =      (1 << 0),
        MCP9808_CFG_FLAG_ALERT_POL_HIGH =      (1 << 1),
        MCP9808_CFG_FLAG_ALERT_CRIT_ONLY =     (1 << 2),
        MCP9808_CFG_FLAG_ALERT_ENABLE =        (1 << 3),
        MCP9808_CFG_FLAG_ALERT_ASSERT_ENABLE = (1 << 4),
        MCP9808_CFG_FLAG_INT_CLR =             (1 << 5),
        MCP9808_CFG_FLAG_WIN_LOCKED =          (1 << 6),
        MCP9808_CFG_FLAG_CRIT_LOCKED =         (1 << 7),
        MCP9808_CFG_FLAG_SHDN_ENABLED =        (1 << 8),
    };

    /** Manufacturer ID from MCP9808_REG_MFG_ID register
     */
    static const uint16_t MCP9808_MFG_ID = 0x0054;

    /** Device ID from MCP9808_REG_DEV_ID register
     */
    static const uint16_t MCP9808_DEV_ID = 0x0400;

    /** Create an MCP9808 using the specified pins for I2C and specified I2C
     *  address
     *
     *  @param sda I2C SDA pin to connect to
     *  @param scl I2C SCL pin to connect to
     *  @param addr I2C address for the sensor
     */
    MCP9808(PinName sda, PinName scl, int addr);

    virtual ~MCP9808();

    /** Detects the sensor's presence
     *
     * @returns true if sensor is detected
     */
    bool is_detected();

    /** Puts the sensor into (low-power) shutdown mode
     *
     *  @returns 0 on success, negative on error
     */
    int sleep();

    /** Puts the sensor into continuous conversion mode
     *
     *  @returns 0 on success, negative on error
     */
    int wake();

    /** Returns the current ambient temperature in degrees Celsius
     *
     *  @returns A floating-point value representing the current ambient
     *           temperature in degrees Celsius
     */
    float get_temperature();

private:
    I2C _i2c;
    int _addr;

    /** Reads a 16-bit register from the sensor
     *
     *  @param reg The register to read from
     *  @param value A pointer to the value to read into
     *  @returns 0 on success, negative on error
     */
    int reg_read(uint8_t reg, uint16_t *value);

    /** Writes a 16-bit register into the sensor
     *
     *  @param reg The register to read into
     *  @param value The value to write
     *  @returns 0 on success, negative on error
     */
    int reg_write(uint8_t reg, uint16_t value);

    /** Sets or clears a flag from the config register
     *
     *  @param flag The flag to modify
     *  @param v The flag state (true = set, false = clear)
     *  @returns 0 on success, negative on error
     */
    int set_cfg_flag(uint16_t flag, bool v);
};

#endif /* MCP9808_HPP */
