#include <stdint.h>
#include "mbed.h"
#include "MCP9808.hpp"

MCP9808::MCP9808(PinName sda, PinName scl, int addr) :
    _i2c(sda, scl), _addr(addr) {}

MCP9808:: ~MCP9808() {}

bool MCP9808::is_detected()
{
    uint16_t v = 0;

    /* check MFG ID */
    if (reg_read(MCP9808_REG_MFG_ID, &v)) {
        return false;
    }
    if (v != MCP9808_MFG_ID) {
        return false;
    }

    /* check DEV ID */
    if (reg_read(MCP9808_REG_DEV_ID, &v)) {
        return false;
    }
    if (v != MCP9808_DEV_ID) {
        return false;
    }

    return true;
}

int MCP9808::sleep()
{
    return set_cfg_flag(MCP9808_CFG_FLAG_SHDN_ENABLED, true);
}

int MCP9808::wake()
{
    return set_cfg_flag(MCP9808_CFG_FLAG_SHDN_ENABLED, false);
}

float MCP9808::get_temperature()
{
    uint16_t v = 0;
    float t = 0.0f;

    reg_read(MCP9808_REG_TEMP, &v);
    t = v & 0x0FFF;
    t /= 16.0f;
    if (v & 0x1000) {
        t -= 256.0f;
    }
    return t;
}

int MCP9808::reg_read(uint8_t reg, uint16_t *value)
{
    uint8_t buffer[sizeof(*value)] = { reg, 0x00 };
    _i2c.write(_addr, (const char *)buffer, 1, true);
    _i2c.read(_addr, (char *)buffer, sizeof(buffer), false);
    *value = (buffer[0] << 8) | buffer[1];
    return 0;
}

int MCP9808::reg_write(uint8_t reg, uint16_t value)
{
    uint8_t buffer[] = { reg, (value >> 8), (value & 0xFF) };
    _i2c.write(_addr, (const char *)buffer, sizeof(buffer), false);
    return 0;
}

int MCP9808::set_cfg_flag(uint16_t flag, bool v)
{
    uint16_t value = 0x00;
    reg_read(MCP9808_REG_CFG, &value);
    if (v) {
        value |= flag;
    } else {
        value &= ~flag;
    }
    return reg_write(MCP9808_REG_CFG, value);
}
