/*
 * Pacemaker MBED code
 *
 * CIS541 Embedded Systems for Life Critical Applications
 * Authors: Jing Qiu
 *          Michael Frederick
 *          Tim Hu
 *          Alex Little
 * 
 */

#include "mbed.h"
#include "LPC17xx.h"
#include "TextLCD.h"
#include "rtos.h"
#include <ctype.h>

#define AVI_h 100
#define AVI_l 30
#define PVARP_h 500
#define PVARP_l 150

#define VRP_h 500
#define VRP_l 150
#define BM(x) (1<<(x))
#define PULSE_WIDTH_V 10
#define PULSE_WIDTH_A 10

TextLCD lcd(p15, p16, p17, p18, p19, p20, TextLCD::LCD16x2);

int LRI_h = 1666;
int LRI_l = 666;
int observation_interval=10000;   //ms
bool setObservation = false;

Serial pc(USBTX, USBRX);

char mode = 'N';

int a_clock;
int v_clock;
bool setobservation = false;

int beat;

InterruptIn atrial_int(p21); // double check me
InterruptIn vent_int(p22); // me too

DigitalOut aPace(p23);
DigitalOut vPace(p24);

bool aSensed = 0; // 0 means that we are expecting Apace or Asense next
DigitalOut led1(LED1);   //apace
DigitalOut led2(LED2);   //vpace
DigitalOut led3(LED3);   //asense
DigitalOut led4(LED4);   //vsense

void initTimer();
void startTimer();

void asense();
void vsense();

void apace();
void vpace();


void apace()
{
    led1 = 1;
    aPace = 1;
    wait_ms(PULSE_WIDTH_A);
    aPace = 0;
    led1 = 0;   
    beat++;
    
}

void vpace()
{
    led2= 1;
    vPace = 1;
    wait_ms(PULSE_WIDTH_V);
    vPace = 0;
    led2 = 0;
}

void initTimer()
{
    // set up OS timer (timer0)
    LPC_SC->PCONP |= BM(1); //power up timer0
    LPC_SC->PCLKSEL0 |= BM(2); // clock = CCLK (96 MHz)
    LPC_TIM0->PR = 48000; // set prescale to 48000 (2048 Hz timer) 
    LPC_TIM0->MR0 = 1; // match0 compare value (32-bit)
    LPC_TIM0->MCR |= BM(0)|BM(1); // interrupt and reset on match0 compare
    NVIC_EnableIRQ(TIMER0_IRQn); // enable timer interrupt
}

void startTimer()
{
    LPC_TIM0->TCR |= BM(1); // reset timer1
    LPC_TIM0->TCR &= ~BM(1); // release reset
    LPC_TIM0->TCR |= BM(0); // start timer
}

void resetTimer()
{
    LPC_TIM0->TCR |= BM(1); // reset timer0
    LPC_TIM0->TCR &= ~BM(1); // release reset
}



void display_thread_handler(void const *args)
{    
    while(1)
    {
        wait_ms(observation_interval);
        pc.printf("BPM: %d\r\n",beat*60000/observation_interval);
        lcd.cls();
        lcd.printf("BPM: %d\r\n", beat*60000/observation_interval);
        beat=0;
    }
}


void asense() {
    if (v_clock >= VRP_l) { //Ignore vSense outside this time interval
        v_clock = 0;
        aSensed = 0;
     }
     led3 = 1;
     wait_ms(PULSE_WIDTH_A);
     led3 = 0;
     
}

void vsense() {
    if ((a_clock >= PVARP_l) && aSensed == 0){ 
        a_clock = 0;
        aSensed = 1;
        beat++;
    }
    led4 = 1;
    wait_ms(PULSE_WIDTH_V);
    led4 = 0;
}

void button_handler(void const *args)
{
    int observation_temp=0;
    while(1)
    {
      char buffer;
      if(pc.readable()) {
        buffer = toupper(pc.getc());
        if (buffer == 'N')
        {
            mode = buffer;
            LRI_h = 1500;
            LRI_l = 600;
        }
        else if (buffer == 'S')
        {
            mode = buffer;
            LRI_h = 2000;
            LRI_l = 1000;
        }
        else if (buffer == 'E')
        {
            mode = buffer;
            LRI_h = 600;
            LRI_l = 342;
        }
        else if (buffer == 'M')
        {
            mode = buffer;
            LRI_h = 2000;
            LRI_l = 342;
        }
        else if (buffer == 'A' && mode == 'M')
            apace();
        else if (buffer == 'V' && mode == 'M')
            vpace();
        else if (buffer == 'O') {
            setObservation = true;
            observation_temp = 0;
        }
        else if (buffer == 13  && setObservation) //newline
        {
            observation_interval = observation_temp*1000;
            setObservation = false;
         //   printf("observation interval now: %d\r\n",observation_interval);
        }
        else if (setObservation)
        {
            observation_temp *= 10;
            observation_temp += buffer-48;
        
        }
      }
    }
}

void pacemaker_thread_handler(void const *args)
{
    
    /**********************************************************
    ************Initialize timer to interrupt every 1 ms*********
    ***********************************************************/
    
    initTimer();
    startTimer();
    
    while(1){}    
    
}

 
/**********************************************************
 ************ timer interrupt every 1 ms*********
 ***********************************************************/
    
extern "C" void TIMER0_IRQHandler (void) {
    if((LPC_TIM0->IR & 0x01) == 0x01) // if MR0 interrupt
    {
        LPC_TIM0->IR |= (1 << 0); // Clear MR0 interrupt flag
       if (v_clock >= (LRI_h-AVI_l) && aSensed == 0) {
            a_clock = 0;
            aSensed = 1;  
            //printf("Apace %d\r\n",v_clk);
            apace();
            /*
            setGPIO(a_pace);
            wait_us(PULSE_WIDTH_A);
            clearGPIO(a_pace);
            */
        }
        if ((a_clock >= AVI_h) && aSensed == 1) {
            v_clock = 0;
            aSensed = 0;
            //led3 = 0;
            //printf("Vpace %d\r\n",a_clk);
            /*
            setGPIO(v_pace);
            wait_us(PULSE_WIDTH_V);
            clearGPIO(v_pace);
            */         
            vpace();
        }
        v_clock++;
        a_clock++;
        
        if(v_clock>500) led3 = 0;
        if(a_clock>500) led4 = 0;
    } 
}


int main (void) {
    //TODO set parameters
    atrial_int.rise(&asense);
    vent_int.rise(&vsense);

    Thread display(display_thread_handler);
    Thread keyboard(button_handler);
    Thread pacemaker(pacemaker_thread_handler);

    while(1){}
    

}