#ifndef MCP4922_H
#define MCP4922_H

#include "DAC_SPI.h"

/** 
* The MCP4922 is a dual package 12 bit DAC.  We should be able to produce two analog output voltages up to twice the externally wired voltage references (VrefA, VrefB).
 The maximum voltage output is limited by the input voltage at V_DD which can go up to about 5.5V.  

* Datasheet: http://ww1.microchip.com/downloads/en/DeviceDoc/22250A.pdf

* All serial commands are 16 bit words.   The highest 4 bits are control bits, while the last 12 are the data bits for the 12-bit DAC MCP4822.
*    + bit 15: select which DAC to use.
*    + bit 14: 0=buffered , 1= unbuffered
*    + bit 13: 0= gain x2, 1= gain x1
*    + bit 12: 0= DAC active, 1= shut down DAC
*    + bit 11-0: Data bits for the DAC.
*/
class MCP4922: public DAC_SPI
{    
    public:
    
    /** Initializes the MCP 4922 DAC 
    *
    * @param SPIchannelNum An int representing the SPI channel used by this DAC
    * channel 0 for p5 and p7
    * channel 1 for p11 and p13
    * @param CS The chip select pin used to identify the device
    * @param LDAC The LDAC pin used to synchronize updates of multiple devices
    */
    MCP4922(int SPIchannelNum, PinName CS,  PinName LDAC);
    
    /** Writes a value between 0-4095 to the currently selected DAC output 
    * @param DACvalue a value from 0-4095 to write to the DAC register
    */
    virtual void write(int DACvalue);
    
    /** Writes a value in mV to the DAC outputs.
    * The output will only be accurate if Vref is set to the appropriate voltage reference scaling factor. 
    * @param millivolte The desired voltage output in millivolts
    */
    virtual void write_mV(int millivolts); 
    
    /** If automatic updating is disabled, this will update the DAC output on command */
    void update();    
    
    /** select whether to use DAC A or DAC B. 
    * @param DACnum 0 to modify DAC A, 1 to modify DAC B
    */
    virtual void select(char DACnum); 
    
    /** If true, the DAC output will update as soon as the output value is modified.  If false, it will wait for an update command*/
    bool autoUpdate;
    
    /** The currently selected DAC channel.  0 for DAC A, 1 for DAC B*/
    char DACselect;
    
    /** The output scaling factor in mV for channel A.
    * The voltage wired to Vref will be used as the scaling factor which the DAC will use in producing the output voltage.
    * If using the write_mV function, it is important that Vref is properly set.  This library uses 5000mV by default.
    */
    int VrefA;
    /** The output scaling factor in mV for channel B.
    * The voltage wired to Vref will be used as the scaling factor which the DAC will use in producing the output voltage.
    * If using the write_mV function, it is important that Vref is properly set.  This library uses 5000mV by default.
    */
    int VrefB;
    
    /** Manually set the gain of the DAC.  The only valid values are 1 and 2.   The default gain is x1 to match the wired reference voltage*/
    void setGain(int gain_value);
    
    /** Turn on or off the output buffer.  The default value for the buffer is off.  See section 4.1.2 of the datasheet for more information. */
    char buffered;
    
    private:
    /** The output gain bit of the DAC.  If set to 0, gain is x2. If set to 1 gain is x1.  The default gain is 1.*/
    bool gain;
};

#endif //MCP4922_H