#ifndef MCP4822_H
#define MCP4822_H
#include "DAC_SPI.h"

/** The MCP4822 is a dual package 12 bit DAC. It has a simple pinout and is pretty easy to use.
* We should be able to produce two analog output voltages up to 4.096V, by using a gain of 2 with the internal reference diode of 2.048V. 
* This is a simple library intended to help beginners get started quickly. 
 
*Datasheet: http://ww1.microchip.com/downloads/en/DeviceDoc/22249A.pdf
  
*All serial commands are 16 bit words.   The highest 4 bits are control bits, while the last 12 are the data bits for the 12-bit DAC MCP4822.
*    + bit 15: select which DAC to use.
*    + bit 14: unused
*    + bit 13: 0= gain x2, 1= gain x1
*    + bit 12: 0= DAC active, 1= shut down DAC
*    + bit 11-0: Data bits for the DAC.
*/
class MCP4822: public DAC_SPI
{

    public:
    /** Initializes the MCP 4822 DAC 
    *
    * @param SPIchannelNum An int representing the SPI channel used by this DAC
    * channel 0 for p5 and p7
    * channel 1 for p11 and p13
    * @param CS The chip select pin used to identify the device
    * @param LDAC The LDAC pin used to synchronize updates of multiple devices
    */
    MCP4822(int SPIchannelNum, PinName CS,  PinName LDAC);
    
    /** Writes a value between 0-4095 to the currently selected DAC output 
    * @param DACvalue a value from 0-4095 to write to the DAC register
    */
    virtual void write(int millivolts);
    
    /** Writes a value between 0-4095mV to the currently selected DAC output.  Unless the gain has been modified this will produce the same result as write(). 
    * @param DACvalue a value from 0-4095mV to write to the DAC register
    */
    virtual void write_mv(int millivolts);
    /** If automatic updating is disabled, this will update the DAC output on command */
    void update();
    /** select whether to use DAC A or DAC B. 
    * @param DACnum 0 to modify DAC A, 1 to modify DAC B
    */    
    void select(char DACnum); //select whether to use DAC A or DAC B.
    /** If true, the DAC output will update as soon as the output value is modified.  If false, it will wait for an update command*/
    

    /** If true, the DAC output will update as soon as the output value is modified.  If false, it will wait for an update command*/
    bool autoUpdate;
    /** The currently selected DAC channel.  0 for DAC A, 1 for DAC B*/
    bool DACselect;
    
    /** Manually set the gain of the DAC.  The only valid values are 1 and 2.   The default gain is x2, allowing an output range of up to 4.096V*/
    void setGain(int gain_value);
    
    private:
    /** The output gain bit of the DAC.  If set to 0, gain is x2. If set to 1 gain is x1.  The default gain is 2, allowing a range of up to 4.096V.*/
    bool gain;
};

#endif