#ifndef MAX500_H
#define MAX500_H

#include "DAC_SPI.h"

/** 
* The MAX500 is a quad package 8 bit DAC.  We should be able to produce four analog output voltages up to the externally wired voltage references (VrefA/B, VrefC, VrefD).
 The maximum voltage output is recommended to be about 4V less than the power supply voltage, although getting a 12V output from a 15V power supply is possible.  
*
* Datasheet: http://datasheets.maxim-ic.com/en/ds/MAX500.pdf
*
* We use the "3-wire" serial interface since it seems to be more compatible with the mbed Serial library timing than the "2-wire" interface.  The 2-wire interface uses a somewhat non-standard timing scheme
* and may require custom code support.
*
* All serial commands are 10 bit words.   The highest 2 bits are control bits, while the last 8 are the data bits for the 8-bit DAC MAX500.   
*    + bit 13-12: choose which DAC to use
*    + bit 11-0: Data bits for the DAC.
*/
class MAX500: public DAC_SPI
{    
    public:
    
    /** Initializes the MAX 500 DAC 
    *
    * @param SPIchannelNum An int representing the SPI channel used by this DAC
    * channel 0 for p5 and p7
    * channel 1 for p11 and p13
    * @param LOAD The chip select pin (CS) used to identify the device
    * @param LDAC The LDAC pin used to synchronize updates of multiple devices
    */
    MAX500(int SPIchannelNum, PinName LOAD,  PinName LDAC);
    
    /** Writes a value between 0-255 to the currently selected DAC output 
    * @param DACvalue a value from 0-255 to write to the DAC register
    */
    virtual void write(int DACvalue);
    
    /** Writes a value in mV to the DAC outputs.
    * The output will only be accurate if Vref is set to the appropriate voltage reference scaling factor. 
    * @param millivolts The desired voltage output in millivolts
    */
    virtual void write_mV(int millivolts); 
    
    /** If automatic updating is disabled, this will update the DAC output on command */
    void update();    
    
    /** select whether to use DAC A or DAC B. 
    * @param DACnum 0 to modify DAC A, 1 to modify DAC B
    */
    virtual void select(char DACnum); 
    
    /** If true, the DAC output will update as soon as the output value is modified.  If false, it will wait for an update command*/
    bool autoUpdate;
    
    /** The currently selected DAC channel.  0 for DAC A, 1 for DAC B*/
    char DACselect;
    
    /** The output scaling factor in mV for channels A and B.
    * The voltage wired to Vref will be used as the scaling factor which the DAC will use in producing the output voltage.
    * If using the write_mV function, it is important that Vref is properly set.  This library uses 5000mV by default.
    */
    int VrefAB;
    /** The output scaling factor in mV for channel C.
    * The voltage wired to Vref will be used as the scaling factor which the DAC will use in producing the output voltage.
    * If using the write_mV function, it is important that Vref is properly set.  This library uses 5000mV by default.
    */
    int VrefC;
    /** The output scaling factor in mV for channel D.
    * The voltage wired to Vref will be used as the scaling factor which the DAC will use in producing the output voltage.
    * If using the write_mV function, it is important that Vref is properly set.  This library uses 5000mV by default.
    */    
    int VrefD;
    
    
    
};

#endif //MAX500_H