/*******************************************************************************
 * Copyright 2016, 2017 ARM Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *******************************************************************************/

#include "pal.h"


typedef struct palAesTestVector{
	unsigned char key[16];
	unsigned char nonce[16];
	unsigned char input[16];
	unsigned char output[16];
} palAesTestVector_t;

const palAesTestVector_t CtrVector =
{
		{
				0xAE, 0x68, 0x52, 0xF8, 0x12, 0x10, 0x67, 0xCC,
				0x4B, 0xF7, 0xA5, 0x76, 0x55, 0x77, 0xF3, 0x9E
		},
		{
				0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
		},
		{
				0x53, 0x69, 0x6E, 0x67, 0x6C, 0x65, 0x20, 0x62,
				0x6C, 0x6F, 0x63, 0x6B, 0x20, 0x6D, 0x73, 0x67
		},
		{
				0xE4, 0x09, 0x5D, 0x4F, 0xB7, 0xA7, 0xB3, 0x79,
				0x2D, 0x61, 0x75, 0xA3, 0x26, 0x13, 0x11, 0xB8
		}
};

const palAesTestVector_t EcbVector =
{
		{
				0xAE, 0x68, 0x52, 0xF8, 0x12, 0x10, 0x67, 0xCC,
				0x4B, 0xF7, 0xA5, 0x76, 0x55, 0x77, 0xF3, 0x9E
		},
		{
				0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
		},
		{
				0x53, 0x69, 0x6E, 0x67, 0x6C, 0x65, 0x20, 0x62,
				0x6C, 0x6F, 0x63, 0x6B, 0x20, 0x6D, 0x73, 0x67
		},
		{
				0x61, 0x5f, 0x09, 0xfb, 0x35, 0x3f, 0x61, 0x3b,
				0xa2, 0x8f, 0xf3, 0xa3, 0x0c, 0x64, 0x75, 0x2d
		}
};



typedef struct palSha256Vector_2nd{
	size_t inLenInBytes;
	size_t outLenInBytes;
	unsigned char output[32];
	unsigned char input[1000];
} palSha256Vector_2nd_t;

palSha256Vector_2nd_t sha256Vectors_2nd ={
		955,
		32,
		{
				0x41, 0x09, 0xcd, 0xbe, 0xc3, 0x24, 0x0a, 0xd7, 0x4c, 0xc6,0xc3, 0x7f, 0x39, 0x30, 0x0f, 0x70,
				0xfe, 0xde, 0x16, 0xe2, 0x1e, 0xfc, 0x77, 0xf7, 0x86, 0x59,0x98, 0x71, 0x4a, 0xad, 0x0b, 0x5e
		},
		{
				0x83, 0x90, 0xcf, 0x0b, 0xe0, 0x76, 0x61, 0xcc, 0x76, 0x69, 0xaa, 0xc5, 0x4c, 0xe0, 0x9a, 0x37,
				0x73, 0x3a, 0x62, 0x9d, 0x45, 0xf5, 0xd9, 0x83, 0xef, 0x20, 0x1f, 0x9b, 0x2d, 0x13, 0x80, 0x0e,
				0x55, 0x5d, 0x9b, 0x10, 0x97, 0xfe, 0xc3, 0xb7, 0x83, 0xd7, 0xa5, 0x0d, 0xcb, 0x5e, 0x2b, 0x64,
				0x4b, 0x96, 0xa1, 0xe9, 0x46, 0x3f, 0x17, 0x7c, 0xf3, 0x49, 0x06, 0xbf, 0x38, 0x8f, 0x36, 0x6d,
				0xb5, 0xc2, 0xde, 0xee, 0x04, 0xa3, 0x0e, 0x28, 0x3f, 0x76, 0x4a, 0x97, 0xc3, 0xb3, 0x77, 0xa0,
				0x34, 0xfe, 0xfc, 0x22, 0xc2, 0x59, 0x21, 0x4f, 0xaa, 0x99, 0xba, 0xba, 0xff, 0x16, 0x0a, 0xb0,
				0xaa, 0xa7, 0xe2, 0xcc, 0xb0, 0xce, 0x09, 0xc6, 0xb3, 0x2f, 0xe0, 0x8c, 0xbc, 0x47, 0x46, 0x94,
				0x37, 0x5a, 0xba, 0x70, 0x3f, 0xad, 0xbf, 0xa3, 0x1c, 0xf6, 0x85, 0xb3, 0x0a, 0x11, 0xc5, 0x7f,
				0x3c, 0xf4, 0xed, 0xd3, 0x21, 0xe5, 0x7d, 0x3a, 0xe6, 0xeb, 0xb1, 0x13, 0x3c, 0x82, 0x60, 0xe7,
				0x5b, 0x92, 0x24, 0xfa, 0x47, 0xa2, 0xbb, 0x20, 0x52, 0x49, 0xad, 0xd2, 0xe2, 0xe6, 0x2f, 0x81,
				0x74, 0x91, 0x48, 0x2a, 0xe1, 0x52, 0x32, 0x2b, 0xe0, 0x90, 0x03, 0x55, 0xcd, 0xcc, 0x8d, 0x42,
				0xa9, 0x8f, 0x82, 0xe9, 0x61, 0xa0, 0xdc, 0x6f, 0x53, 0x7b, 0x7b, 0x41, 0x0e, 0xff, 0x10, 0x5f,
				0x59, 0x67, 0x3b, 0xfb, 0x78, 0x7b, 0xf0, 0x42, 0xaa, 0x07, 0x1f, 0x7a, 0xf6, 0x8d, 0x94, 0x4d,
				0x27, 0x37, 0x1c, 0x64, 0x16, 0x0f, 0xe9, 0x38, 0x27, 0x72, 0x37, 0x25, 0x16, 0xc2, 0x30, 0xc1,
				0xf4, 0x5c, 0x0d, 0x6b, 0x6c, 0xca, 0x7f, 0x27, 0x4b, 0x39, 0x4d, 0xa9, 0x40, 0x2d, 0x3e, 0xaf,
				0xdf, 0x73, 0x39, 0x94, 0xec, 0x58, 0xab, 0x22, 0xd7, 0x18, 0x29, 0xa9, 0x83, 0x99, 0x57, 0x4d,
				0x4b, 0x59, 0x08, 0xa4, 0x47, 0xa5, 0xa6, 0x81, 0xcb, 0x0d, 0xd5, 0x0a, 0x31, 0x14, 0x53, 0x11,
				0xd9, 0x2c, 0x22, 0xa1, 0x6d, 0xe1, 0xea, 0xd6, 0x6a, 0x54, 0x99, 0xf2, 0xdc, 0xeb, 0x4c, 0xae,
				0x69, 0x47, 0x72, 0xce, 0x90, 0x76, 0x2e, 0xf8, 0x33, 0x6a, 0xfe, 0xc6, 0x53, 0xaa, 0x9b, 0x1a,
				0x1c, 0x48, 0x20, 0xb2, 0x21, 0x13, 0x6d, 0xfc, 0xe8, 0x0d, 0xce, 0x2b, 0xa9, 0x20, 0xd8, 0x8a,
				0x53, 0x0c, 0x94, 0x10, 0xd0, 0xa4, 0xe0, 0x35, 0x8a, 0x3a, 0x11, 0x05, 0x2e, 0x58, 0xdd, 0x73,
				0xb0, 0xb1, 0x79, 0xef, 0x8f, 0x56, 0xfe, 0x3b, 0x5a, 0x2d, 0x11, 0x7a, 0x73, 0xa0, 0xc3, 0x8a,
				0x13, 0x92, 0xb6, 0x93, 0x8e, 0x97, 0x82, 0xe0, 0xd8, 0x64, 0x56, 0xee, 0x48, 0x84, 0xe3, 0xc3,
				0x9d, 0x4d, 0x75, 0x81, 0x3f, 0x13, 0x63, 0x3b, 0xc7, 0x9b, 0xaa, 0x07, 0xc0, 0xd2, 0xd5, 0x55,
				0xaf, 0xbf, 0x20, 0x7f, 0x52, 0xb7, 0xdc, 0xa1, 0x26, 0xd0, 0x15, 0xaa, 0x2b, 0x98, 0x73, 0xb3,
				0xeb, 0x06, 0x5e, 0x90, 0xb9, 0xb0, 0x65, 0xa5, 0x37, 0x3f, 0xe1, 0xfb, 0x1b, 0x20, 0xd5, 0x94,
				0x32, 0x7d, 0x19, 0xfb, 0xa5, 0x6c, 0xb8, 0x1e, 0x7b, 0x66, 0x96, 0x60, 0x5f, 0xfa, 0x56, 0xeb,
				0xa3, 0xc2, 0x7a, 0x43, 0x86, 0x97, 0xcc, 0x21, 0xb2, 0x01, 0xfd, 0x7e, 0x09, 0xf1, 0x8d, 0xee,
				0xa1, 0xb3, 0xea, 0x2f, 0x0d, 0x1e, 0xdc, 0x02, 0xdf, 0x0e, 0x20, 0x39, 0x6a, 0x14, 0x54, 0x12,
				0xcd, 0x6b, 0x13, 0xc3, 0x2d, 0x2e, 0x60, 0x56, 0x41, 0xc9, 0x48, 0xb7, 0x14, 0xae, 0xc3, 0x0c,
				0x06, 0x49, 0xdc, 0x44, 0x14, 0x35, 0x11, 0xf3, 0x5a, 0xb0, 0xfd, 0x5d, 0xd6, 0x4c, 0x34, 0xd0,
				0x6f, 0xe8, 0x6f, 0x38, 0x36, 0xdf, 0xe9, 0xed, 0xeb, 0x7f, 0x08, 0xcf, 0xc3, 0xbd, 0x40, 0x95,
				0x68, 0x26, 0x35, 0x62, 0x42, 0x19, 0x1f, 0x99, 0xf5, 0x34, 0x73, 0xf3, 0x2b, 0x0c, 0xc0, 0xcf,
				0x93, 0x21, 0xd6, 0xc9, 0x2a, 0x11, 0x2e, 0x8d, 0xb9, 0x0b, 0x86, 0xee, 0x9e, 0x87, 0xcc, 0x32,
				0xd0, 0x34, 0x3d, 0xb0, 0x1e, 0x32, 0xce, 0x9e, 0xb7, 0x82, 0xcb, 0x24, 0xef, 0xbb, 0xbe, 0xb4,
				0x40, 0xfe, 0x92, 0x9e, 0x8f, 0x2b, 0xf8, 0xdf, 0xb1, 0x55, 0x0a, 0x3a, 0x2e, 0x74, 0x2e, 0x8b,
				0x45, 0x5a, 0x3e, 0x57, 0x30, 0xe9, 0xe6, 0xa7, 0xa9, 0x82, 0x4d, 0x17, 0xac, 0xc0, 0xf7, 0x2a,
				0x7f, 0x67, 0xea, 0xe0, 0xf0, 0x97, 0x0f, 0x8b, 0xde, 0x46, 0xdc, 0xde, 0xfa, 0xed, 0x30, 0x47,
				0xcf, 0x80, 0x7e, 0x7f, 0x00, 0xa4, 0x2e, 0x5f, 0xd1, 0x1d, 0x40, 0xf5, 0xe9, 0x85, 0x33, 0xd7,
				0x57, 0x44, 0x25, 0xb7, 0xd2, 0xbc, 0x3b, 0x38, 0x45, 0xc4, 0x43, 0x00, 0x8b, 0x58, 0x98, 0x0e,
				0x76, 0x8e, 0x46, 0x4e, 0x17, 0xcc, 0x6f, 0x6b, 0x39, 0x39, 0xee, 0xe5, 0x2f, 0x71, 0x39, 0x63,
				0xd0, 0x7d, 0x8c, 0x4a, 0xbf, 0x02, 0x44, 0x8e, 0xf0, 0xb8, 0x89, 0xc9, 0x67, 0x1e, 0x2f, 0x8a,
				0x43, 0x6d, 0xde, 0xef, 0xfc, 0xca, 0x71, 0x76, 0xe9, 0xbf, 0x9d, 0x10, 0x05, 0xec, 0xd3, 0x77,
				0xf2, 0xfa, 0x67, 0xc2, 0x3e, 0xd1, 0xf1, 0x37, 0xe6, 0x0b, 0xf4, 0x60, 0x18, 0xa8, 0xbd, 0x61,
				0x3d, 0x03, 0x8e, 0x88, 0x37, 0x04, 0xfc, 0x26, 0xe7, 0x98, 0x96, 0x9d, 0xf3, 0x5e, 0xc7, 0xbb,
				0xc6, 0xa4, 0xfe, 0x46, 0xd8, 0x91, 0x0b, 0xd8, 0x2f, 0xa3, 0xcd, 0xed, 0x26, 0x5d, 0x0a, 0x3b,
				0x6d, 0x39, 0x9e, 0x42, 0x51, 0xe4, 0xd8, 0x23, 0x3d, 0xaa, 0x21, 0xb5, 0x81, 0x2f, 0xde, 0xd6,
				0x53, 0x61, 0x98, 0xff, 0x13, 0xaa, 0x5a, 0x1c, 0xd4, 0x6a, 0x5b, 0x9a, 0x17, 0xa4, 0xdd, 0xc1,
				0xd9, 0xf8, 0x55, 0x44, 0xd1, 0xd1, 0xcc, 0x16, 0xf3, 0xdf, 0x85, 0x80, 0x38, 0xc8, 0xe0, 0x71,
				0xa1, 0x1a, 0x7e, 0x15, 0x7a, 0x85, 0xa6, 0xa8, 0xdc, 0x47, 0xe8, 0x8d, 0x75, 0xe7, 0x00, 0x9a,
				0x8b, 0x26, 0xfd, 0xb7, 0x3f, 0x33, 0xa2, 0xa7, 0x0f, 0x1e, 0x0c, 0x25, 0x9f, 0x8f, 0x95, 0x33,
				0xb9, 0xb8, 0xf9, 0xaf, 0x92, 0x88, 0xb7, 0x27, 0x4f, 0x21, 0xba, 0xee, 0xc7, 0x8d, 0x39, 0x6f,
				0x8b, 0xac, 0xdc, 0xc2, 0x24, 0x71, 0x20, 0x7d, 0x9b, 0x4e, 0xfc, 0xcd, 0x3f, 0xed, 0xc5, 0xc5,
				0xa2, 0x21, 0x4f, 0xf5, 0xe5, 0x1c, 0x55, 0x3f, 0x35, 0xe2, 0x1a, 0xe6, 0x96, 0xfe, 0x51, 0xe8,
				0xdf, 0x73, 0x3a, 0x8e, 0x06, 0xf5, 0x0f, 0x41, 0x9e, 0x59, 0x9e, 0x9f, 0x9e, 0x4b, 0x37, 0xce,
				0x64, 0x3f, 0xc8, 0x10, 0xfa, 0xaa, 0x47, 0x98, 0x97, 0x71, 0x50, 0x9d, 0x69, 0xa1, 0x10, 0xac,
				0x91, 0x62, 0x61, 0x42, 0x70, 0x26, 0x36, 0x9a, 0x21, 0x26, 0x3a, 0xc4, 0x46, 0x0f, 0xb4, 0xf7,
				0x08, 0xf8, 0xae, 0x28, 0x59, 0x98, 0x56, 0xdb, 0x7c, 0xb6, 0xa4, 0x3a, 0xc8, 0xe0, 0x3d, 0x64,
				0xa9, 0x60, 0x98, 0x07, 0xe7, 0x6c, 0x5f, 0x31, 0x2b, 0x9d, 0x18, 0x63, 0xbf, 0xa3, 0x04, 0xe8,
				0x95, 0x36, 0x47, 0x64, 0x8b, 0x4f, 0x4a, 0xb0, 0xed, 0x99, 0x5e
		}
};

typedef struct palSha256Vector{
	size_t inLenInBytes;
	size_t outLenInBytes;
	unsigned char output[32];
	unsigned char input[5];
} palSha256Vector_t;

palSha256Vector_t sha256Vectors[6] =
{
		{
				0,
				32,
				{
						0xe3, 0xb0, 0xc4, 0x42, 0x98, 0xfc, 0x1c, 0x14, 0x9a, 0xfb, 0xf4, 0xc8, 0x99, 0x6f, 0xb9, 0x24,
						0x27, 0xae, 0x41, 0xe4, 0x64, 0x9b, 0x93, 0x4c, 0xa4, 0x95, 0x99, 0x1b, 0x78, 0x52, 0xb8, 0x55
				},
				{0x00}
		},
		{
				1,
				32,
				{
						0x68, 0x32, 0x57, 0x20, 0xaa, 0xbd, 0x7c, 0x82, 0xf3, 0x0f, 0x55, 0x4b, 0x31, 0x3d, 0x05, 0x70,
						0xc9, 0x5a, 0xcc, 0xbb, 0x7d, 0xc4, 0xb5, 0xaa, 0xe1, 0x12, 0x04, 0xc0, 0x8f, 0xfe, 0x73, 0x2b
				},
				{0xbd}
		},
		{
				2,
				32,
				{
						0x7c, 0x4f, 0xbf, 0x48, 0x44, 0x98, 0xd2, 0x1b, 0x48, 0x7b, 0x9d, 0x61, 0xde, 0x89,0x14, 0xb2,
						0xea, 0xda, 0xf2, 0x69, 0x87, 0x12, 0x93, 0x6d, 0x47, 0xc3, 0xad, 0xa2, 0x55, 0x8f,0x67, 0x88
				},
				{0x5f, 0xd4}
		},
		{
				3,
				32,
				{
						0x40, 0x96, 0x80, 0x42, 0x21, 0x09, 0x3d, 0xdc, 0xcf, 0xbf, 0x46, 0x83, 0x14, 0x90, 0xea, 0x63,
						0xe9, 0xe9, 0x94, 0x14, 0x85, 0x8f, 0x8d, 0x75, 0xff, 0x7f, 0x64, 0x2c, 0x7c, 0xa6, 0x18, 0x03
				},
				{0xb0, 0xbd, 0x69}
		},
		{
				4,
				32,
				{
						0x7a, 0xbc, 0x22, 0xc0, 0xae, 0x5a, 0xf2, 0x6c, 0xe9, 0x3d, 0xbb, 0x94, 0x43, 0x3a, 0x0e, 0x0b,
						0x2e, 0x11, 0x9d, 0x01, 0x4f, 0x8e, 0x7f, 0x65, 0xbd, 0x56, 0xc6, 0x1c,0xcc, 0xcd, 0x95, 0x04
				},
				{0xc9, 0x8c, 0x8e, 0x55	}
		},
		{
				5,
				32,
				{
						0x75, 0x16, 0xfb, 0x8b, 0xb1, 0x13, 0x50, 0xdf, 0x2b, 0xf3, 0x86, 0xbc, 0x3c, 0x33, 0xbd, 0x0f,
						0x52, 0xcb, 0x4c, 0x67, 0xc6, 0xe4, 0x74, 0x5e, 0x04, 0x88, 0xe6, 0x2c, 0x2a, 0xea, 0x26, 0x05
				},
				{0x81, 0xa7, 0x23, 0xd9, 0x66}
		}
};

typedef struct palAesCcmVector{
	const unsigned char key[16];
	const unsigned char iv[16];
	const unsigned char ad[20];
	const unsigned char in[24];
	const unsigned char out[32];
	size_t ivLen;
	size_t adLen;
	size_t inLen;
	size_t tagLen;
	//	size_t outLen;

}palAesCcmVector_t;

static const palAesCcmVector_t aesCcmVectors[3] =
{
		{
				{
						0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
						0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f
				},
				{ 	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16 },
				{	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 },
				{	0x20, 0x21, 0x22, 0x23 },
				{   0x71, 0x62, 0x01, 0x5b, 0x4d, 0xac, 0x25, 0x5d },
				7, 8, 4, 4
		},
		{
				{
						0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
						0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f
				},
				{ 	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17 },
				{
						0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
						0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
				},
				{
						0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
						0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f
				},
				{
						0xd2, 0xa1, 0xf0, 0xe0, 0x51, 0xea, 0x5f, 0x62,
						0x08, 0x1a, 0x77, 0x92, 0x07, 0x3d, 0x59, 0x3d,
						0x1f, 0xc6, 0x4f, 0xbf, 0xac, 0xcd
				},
				8, 16, 16, 6
		},
		{
				{
						0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
						0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f
				},
				{
						0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
						0x18, 0x19, 0x1a, 0x1b
				},
				{
						0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
						0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
						0x10, 0x11, 0x12, 0x13
				},
				{
						0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
						0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
						0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
				},
				{
						0xe3, 0xb2, 0x01, 0xa9, 0xf5, 0xb7, 0x1a, 0x7a,
						0x9b, 0x1c, 0xea, 0xec, 0xcd, 0x97, 0xe7, 0x0b,
						0x61, 0x76, 0xaa, 0xd9, 0xa4, 0x42, 0x8a, 0xa5,
						0x48, 0x43, 0x92, 0xfb, 0xc1, 0xb0, 0x99, 0x51
				},
				12, 20, 24, 8
		}
};

static const unsigned char ctr_drbg_nonce_pers[16] =
{
		0x1b, 0x54, 0xb8, 0xff, 0x06, 0x42, 0xbf, 0xf5,
		0x21, 0xf1, 0x5c, 0x1c, 0x0b, 0x66, 0x5f, 0x3f
};

unsigned char ctr_drbg_buf[100][16];

typedef struct palAesCMACVector{
	const unsigned char key[16];
	const unsigned char in[64];
	const unsigned char out[16];
	size_t inLen;
}palAesCMACVector_t;

static const palAesCMACVector_t cmacSingleUseVector[4] =
{
		{
				{
						0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
						0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
				},
				{
						0x00
				},
				{
						0xbb, 0x1d, 0x69, 0x29, 0xe9, 0x59, 0x37, 0x28,
						0x7f, 0xa3, 0x7d, 0x12, 0x9b, 0x75, 0x67, 0x46
				},
				0
		},
		{
				{
						0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
						0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
				},
				{
						0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
						0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a
				},
				{
						0x07, 0x0a, 0x16, 0xb4, 0x6b, 0x4d, 0x41, 0x44,
						0xf7, 0x9b, 0xdd, 0x9d, 0xd0, 0x4a, 0x28, 0x7c
				},
				16
		},
		{
				{
						0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
						0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
				},
				{
						0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
						0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
						0xae, 0x2d, 0x8a, 0x57
				},
				{
						0x7d, 0x85, 0x44, 0x9e, 0xa6, 0xea, 0x19, 0xc8,
						0x23, 0xa7, 0xbf, 0x78, 0x83, 0x7d, 0xfa, 0xde
				},
				20
		},
		{
				{
						0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
						0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
				},
				{
						0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
						0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
						0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
						0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
						0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
						0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
						0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
						0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
				},
				{
						0x51, 0xf0, 0xbe, 0xbf, 0x7e, 0x3b, 0x9d, 0x92,
						0xfc, 0x49, 0x74, 0x17, 0x79, 0x36, 0x3c, 0xfe
				},
				64
		}
};

typedef struct palCMACMultipleBlockVector{
	palCipherID_t cipher_type;
	unsigned char key_string[64];
	unsigned char block1_string[64];
	unsigned char block2_string[64];
	unsigned char block3_string[64];
	unsigned char block4_string[64];
	unsigned char expected_result_string[64];
	uint32_t keybits;
	uint32_t block_size;
	int32_t block1_len;
	int32_t block2_len;
	int32_t block3_len;
	int32_t block4_len;
}palCMACMultipleBlockVector_t;

static const palCMACMultipleBlockVector_t cmacIterativeUseVector[] =
{
		{
				PAL_CIPHER_ID_AES,
				{
						0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
				},
				{	0x00	},
				{	0x00	},
				{	0x00	},
				{	0x00	},
				{
						0xbb, 0x1d, 0x69, 0x29, 0xe9, 0x59, 0x37, 0x28, 0x7f, 0xa3, 0x7d, 0x12, 0x9b, 0x75, 0x67, 0x46
				}, 128, 16,	-1,	-1,	-1,	-1
		},
		{
				PAL_CIPHER_ID_AES,
				{
						0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
				},
				{
						0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a
				},
				{	0x00	},
				{	0x00	},
				{	0x00	},
				{
						0x07, 0x0a, 0x16, 0xb4, 0x6b, 0x4d, 0x41, 0x44, 0xf7, 0x9b, 0xdd, 0x9d, 0xd0, 0x4a, 0x28, 0x7c
				}, 128,	16,	16,	-1,	-1,	-1
		},
		{
				PAL_CIPHER_ID_AES,
				{
						0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
				},
				{
						0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
						0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
						0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
						0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
				},
				{	0x00	},
				{	0x00	},
				{	0x00	},
				{
						0x51, 0xf0, 0xbe, 0xbf, 0x7e, 0x3b, 0x9d, 0x92, 0xfc, 0x49, 0x74, 0x17, 0x79, 0x36, 0x3c, 0xfe
				},	128, 16, 64, -1, -1, -1
		},
		{
				PAL_CIPHER_ID_AES,
				{
						0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
				},
				{
						0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96
				},
				{
						0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a
				},
				{	0x00	},
				{	0x00 	},
				{
						0x07, 0x0a, 0x16, 0xb4, 0x6b, 0x4d, 0x41, 0x44, 0xf7, 0x9b, 0xdd, 0x9d, 0xd0, 0x4a, 0x28, 0x7c
				}, 128,	16,	8, 8, -1, -1
		},
		{
				PAL_CIPHER_ID_AES,
				{
						0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
				},
				{
						0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a
				},
				{
						0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51
				},
				{
						0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef
				},
				{
						0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
				},
				{
						0x51, 0xf0, 0xbe, 0xbf, 0x7e, 0x3b, 0x9d, 0x92, 0xfc, 0x49, 0x74, 0x17, 0x79, 0x36, 0x3c, 0xfe
				}, 128,	16,	16,	16,	16,	16
		},
		{
				PAL_CIPHER_ID_AES,
				{
						0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
				},
				{
						0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96
				},
				{
						0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c
				},
				{
						0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51, 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
						0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef
				},
				{
						0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
				},
				{
						0x51, 0xf0, 0xbe, 0xbf, 0x7e, 0x3b, 0x9d, 0x92, 0xfc, 0x49, 0x74, 0x17, 0x79, 0x36, 0x3c, 0xfe
				},128, 16, 8, 16, 24, 16
		},
		{
				PAL_CIPHER_ID_AES,
				{
						0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
				},
				{	0x00	},
				{
						0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96
				},
				{	0x00	},
				{
						0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a
				},
				{
						0x07, 0x0a, 0x16, 0xb4, 0x6b, 0x4d, 0x41, 0x44, 0xf7, 0x9b, 0xdd, 0x9d, 0xd0, 0x4a, 0x28, 0x7c
				}, 128,	16, 0, 8, 0, 8
		}
};

typedef struct palASN1TestVector{
	size_t start;
	size_t end;
	size_t dataLen;
	uint8_t type;
}palASN1TestVector_t;

static const palASN1TestVector_t ASN1TestVector[11] = {
		{ 0, 879, 875, PAL_ASN1_SEQUENCE | PAL_ASN1_CONSTRUCTED },
		{ 10, 13, 1, PAL_ASN1_INTEGER },
		{ 26, 37, 9, PAL_ASN1_OID },
		{ 37, 39, 0, PAL_ASN1_NULL },
		{ 41, 54, 11, PAL_ASN1_SET | PAL_ASN1_CONSTRUCTED },
		{ 50, 54, 2, PAL_ASN1_PRINTABLE_STRING },
		{ 63, 75, 10, PAL_ASN1_UTF8_STRING },
		{ 119, 134, 13, PAL_ASN1_UTC_TIME },
		{ 532, 556, 22, PAL_ASN1_OCTET_STRING },
		{ 600, 603, 1, PAL_ASN1_BOOLEAN },
		{ 618, 879, 257, PAL_ASN1_BIT_STRING }
};


static const unsigned char asn1_data[879] =
{
		0x30, 0x82, 0x03, 0x6B, 0x30, 0x82, 0x02, 0x53, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x09, 0x00,
		0xAA, 0xF6, 0x26, 0x55, 0x10, 0xA4, 0x58, 0x1E, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
		0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x30, 0x4C, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55,
		0x04, 0x06, 0x13, 0x02, 0x49, 0x4C, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C,
		0x0A, 0x53, 0x6F, 0x6D, 0x65, 0x2D, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x0C, 0x30, 0x0A, 0x06,
		0x03, 0x55, 0x04, 0x07, 0x0C, 0x03, 0x4E, 0x65, 0x74, 0x31, 0x0C, 0x30, 0x0A, 0x06, 0x03, 0x55,
		0x04, 0x0A, 0x0C, 0x03, 0x41, 0x52, 0x4D, 0x31, 0x0C, 0x30, 0x0A, 0x06, 0x03, 0x55, 0x04, 0x0B,
		0x0C, 0x03, 0x50, 0x41, 0x4C, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x36, 0x31, 0x31, 0x32, 0x33, 0x31,
		0x33, 0x35, 0x30, 0x31, 0x39, 0x5A, 0x17, 0x0D, 0x31, 0x37, 0x31, 0x31, 0x32, 0x33, 0x31, 0x33,
		0x35, 0x30, 0x31, 0x39, 0x5A, 0x30, 0x4C, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x49, 0x4C, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C, 0x0A, 0x53,
		0x6F, 0x6D, 0x65, 0x2D, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x0C, 0x30, 0x0A, 0x06, 0x03, 0x55,
		0x04, 0x07, 0x0C, 0x03, 0x4E, 0x65, 0x74, 0x31, 0x0C, 0x30, 0x0A, 0x06, 0x03, 0x55, 0x04, 0x0A,
		0x0C, 0x03, 0x41, 0x52, 0x4D, 0x31, 0x0C, 0x30, 0x0A, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x03,
		0x50, 0x41, 0x4C, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
		0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A, 0x02,
		0x82, 0x01, 0x01, 0x00, 0xCE, 0xD5, 0x2F, 0x89, 0x97, 0xD5, 0xC1, 0xDF, 0x2D, 0xDB, 0x1B, 0xF5,
		0x4B, 0x0F, 0xAF, 0xD0, 0xF1, 0xC5, 0xD4, 0x45, 0xD4, 0x6D, 0x55, 0x73, 0x73, 0xD5, 0xBB, 0x7D,
		0x2E, 0x56, 0x8E, 0xF4, 0xCB, 0xAC, 0x0A, 0x78, 0x91, 0x9E, 0x6C, 0x72, 0xA6, 0x0F, 0x65, 0xFE,
		0x86, 0x91, 0x2D, 0x96, 0xE0, 0x40, 0xF4, 0x3A, 0xBB, 0x82, 0x29, 0x69, 0x23, 0x92, 0x93, 0xC8,
		0x20, 0x83, 0xD3, 0x7A, 0xFF, 0x71, 0xB4, 0x11, 0x43, 0xF0, 0x30, 0xE8, 0x32, 0xF8, 0x4C, 0xCB,
		0xE6, 0xF1, 0xE2, 0xBC, 0x01, 0xE6, 0xB8, 0x17, 0xBF, 0x82, 0xA2, 0xBF, 0x75, 0xFF, 0x88, 0x44,
		0x15, 0x86, 0x88, 0x4A, 0xDF, 0xEF, 0x48, 0x46, 0xD8, 0xFA, 0x81, 0xBB, 0xAA, 0xED, 0x16, 0xC7,
		0x7C, 0xE8, 0xCE, 0x1E, 0x79, 0x6C, 0x2D, 0x66, 0x88, 0x4C, 0xB7, 0x88, 0xA5, 0x35, 0xD7, 0x4A,
		0xE8, 0x6D, 0x89, 0xD1, 0x84, 0xBF, 0x23, 0x9B, 0xA2, 0xF4, 0xF3, 0x0D, 0x3E, 0xB8, 0x12, 0x48,
		0x38, 0xC6, 0x76, 0xD3, 0xEF, 0xAE, 0xA1, 0xD9, 0xC9, 0xA8, 0x32, 0xE7, 0x5E, 0xD5, 0x7C, 0x2A,
		0x26, 0xAD, 0x94, 0x60, 0xF6, 0x73, 0x73, 0xD1, 0x8B, 0xA3, 0x2A, 0x42, 0xA6, 0x86, 0x47, 0x0F,
		0x61, 0x59, 0xCD, 0x13, 0x87, 0xFB, 0x14, 0x16, 0x0E, 0x22, 0xBA, 0x7E, 0xC4, 0xF9, 0xC5, 0x59,
		0x31, 0xDF, 0xDA, 0x6A, 0xBB, 0x3F, 0x8B, 0xE7, 0xF7, 0x22, 0x7E, 0x35, 0xF2, 0xE3, 0x8A, 0xAC,
		0xFA, 0xCF, 0x49, 0xE0, 0x64, 0xC0, 0xC8, 0x24, 0x96, 0x69, 0x23, 0x3A, 0xC7, 0x04, 0x0A, 0x99,
		0xA7, 0x21, 0x87, 0x87, 0x9A, 0x01, 0x7B, 0x50, 0xCF, 0xE1, 0x7B, 0x0E, 0x8A, 0x4A, 0x25, 0xCF,
		0x88, 0xE4, 0x47, 0xAD, 0x85, 0x47, 0x8A, 0xE7, 0x35, 0xD9, 0x6B, 0x13, 0xA4, 0x77, 0x3F, 0x58,
		0x8F, 0x55, 0x65, 0xFF, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x50, 0x30, 0x4E, 0x30, 0x1D, 0x06,
		0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0x53, 0x43, 0x7D, 0x59, 0x2A, 0xF5, 0x93, 0xA5,
		0xA5, 0x02, 0x43, 0xD9, 0x7B, 0xC4, 0x2C, 0xCA, 0xBD, 0xDE, 0x96, 0x0D, 0x30, 0x1F, 0x06, 0x03,
		0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x53, 0x43, 0x7D, 0x59, 0x2A, 0xF5, 0x93,
		0xA5, 0xA5, 0x02, 0x43, 0xD9, 0x7B, 0xC4, 0x2C, 0xCA, 0xBD, 0xDE, 0x96, 0x0D, 0x30, 0x0C, 0x06,
		0x03, 0x55, 0x1D, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xFF, 0x30, 0x0D, 0x06, 0x09, 0x2A,
		0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x15,
		0x9A, 0xFC, 0xBF, 0x26, 0x7A, 0x6A, 0xF9, 0x3F, 0xD4, 0x88, 0xF9, 0x83, 0x6C, 0xF6, 0x68, 0xC6,
		0x99, 0xDD, 0x55, 0x62, 0x89, 0xFB, 0x80, 0x74, 0x5B, 0x14, 0x1A, 0x82, 0x4F, 0xFF, 0x5E, 0xCC,
		0x20, 0x97, 0x68, 0x8C, 0x7B, 0x85, 0x02, 0xDE, 0x5A, 0x02, 0x4C, 0x72, 0x57, 0xE8, 0x01, 0x0A,
		0x08, 0x0F, 0xA2, 0xAB, 0xA4, 0x57, 0xC8, 0x4C, 0x8A, 0x3D, 0xCF, 0x23, 0xA6, 0xE4, 0x8D, 0xE3,
		0x1E, 0x9F, 0x50, 0xB3, 0x2C, 0xDD, 0xC6, 0x11, 0x98, 0xAA, 0x71, 0xA9, 0xF3, 0x02, 0x5C, 0x16,
		0xDB, 0xC2, 0x28, 0xA3, 0x9E, 0x51, 0xB1, 0xE7, 0xEC, 0x60, 0xE4, 0x59, 0x62, 0x1B, 0xC0, 0x4F,
		0xE4, 0xF3, 0xD7, 0x10, 0x12, 0x38, 0x6D, 0x2A, 0xF8, 0x96, 0x4E, 0x25, 0xE5, 0x05, 0xE1, 0x77,
		0x1F, 0xC7, 0xAB, 0x54, 0xDA, 0x0D, 0x09, 0xE5, 0xC3, 0xB0, 0x61, 0x50, 0x11, 0xC8, 0x5C, 0x7B,
		0x6B, 0x96, 0x71, 0x2E, 0xC9, 0x7C, 0x8E, 0x34, 0xA9, 0xB9, 0x25, 0x0A, 0x35, 0x0F, 0xA1, 0x77,
		0x6C, 0xC3, 0x30, 0x96, 0x7C, 0x40, 0x12, 0x9D, 0xBB, 0x62, 0xA7, 0xD6, 0xE6, 0x07, 0xCB, 0xAB,
		0xC6, 0xD7, 0xD1, 0x3C, 0x4D, 0x7F, 0xAE, 0x62, 0x22, 0xBD, 0x88, 0x33, 0x43, 0x15, 0xFF, 0x63,
		0x80, 0x95, 0x29, 0xCD, 0x00, 0x76, 0x53, 0xF6, 0xA6, 0xA1, 0xD0, 0x07, 0xE4, 0xF0, 0xC2, 0x03,
		0xBA, 0x7B, 0x25, 0x9B, 0x75, 0xA1, 0xB6, 0xA8, 0x4C, 0x44, 0x72, 0x14, 0x48, 0x29, 0x75, 0x76,
		0x66, 0xB1, 0xBB, 0x5A, 0x2D, 0x7C, 0x21, 0xBB, 0xEA, 0x70, 0x45, 0x8E, 0x6E, 0xFA, 0xCE, 0xCE,
		0x26, 0xDE, 0xF7, 0x36, 0xB7, 0x52, 0xBC, 0x9F, 0x21, 0x35, 0x2F, 0x5B, 0xAF, 0x0A, 0xAE, 0xD8,
		0xE1, 0x0C, 0x69, 0x27, 0x30, 0xFF, 0xCF, 0x13, 0xE8, 0x57, 0x1F, 0x5B, 0x38, 0x13, 0x02
};


typedef struct palCheckEcKeyTestVector
{
	const unsigned char key[200];
	size_t keyLen;
	palGroupIndex_t index;
	uint32_t keyType;
	palStatus_t parsePrvRes;
	palStatus_t checkPrvRes;
	bool verifed;
	palStatus_t parsePubRes;
	palStatus_t checkPubRes;
}palCheckEcKeyTestVector_t;



static const palCheckEcKeyTestVector_t check_ec_key_data[3] =
{
		{
				{ 0x00 }, 0, PAL_ECP_DP_SECP256R1, PAL_CHECK_PRIVATE_KEY, PAL_ERR_PK_KEY_INVALID_FORMAT, PAL_ERR_PARSING_PRIVATE_KEY, false, PAL_ERR_PARSING_PUBLIC_KEY, PAL_ERR_PARSING_PUBLIC_KEY
		},
		{
				{
						0x30, 0x77, 0x02, 0x01, 0x01, 0x04, 0x20, 0x07, 0xED, 0xF8, 0x74, 0xA2, 0x0A, 0x3D, 0xA2, 0xC5,
						0x89, 0x99, 0x8A, 0x28, 0xC5, 0x00, 0x8C, 0x12, 0xD8, 0x9B, 0xC9, 0x74, 0x2F, 0x94, 0x53, 0x40,
						0x82, 0x36, 0x96, 0x04, 0x15, 0x2A, 0xB6, 0xA0, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
						0x03, 0x01, 0x07, 0xA1, 0x44, 0x03, 0x42, 0x00, 0x04, 0xE5, 0xD8, 0x40, 0xEF, 0xE8, 0x3B, 0x0D,
						0xC5, 0x7E, 0x72, 0x21, 0x65, 0xF6, 0x96, 0xF0, 0xC8, 0x16, 0xCC, 0xC5, 0x9D, 0x88, 0x24, 0x41,
						0xF3, 0xB3, 0x39, 0x34, 0xD2, 0xA1, 0x5B, 0xF9, 0xC7, 0xAF, 0xF6, 0x8F, 0x4E, 0x78, 0x01, 0xDB,
						0xB2, 0xAA, 0x7E, 0x7E, 0xC7, 0x41, 0x31, 0xF4, 0x8D, 0xD4, 0x8E, 0x98, 0x7B, 0x16, 0x4E, 0x96,
						0x26, 0x71, 0x9D, 0x1F, 0x84, 0xEC, 0x68, 0x9A, 0x31
				}, 121, PAL_ECP_DP_SECP256R1, PAL_CHECK_PRIVATE_KEY, PAL_SUCCESS, PAL_SUCCESS, true, PAL_ERR_PARSING_PUBLIC_KEY, PAL_ERR_PARSING_PUBLIC_KEY
		},
		{
				{
						0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A,
						0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xE5, 0xD8, 0x40, 0xEF, 0xE8,
						0x3B, 0x0D, 0xC5, 0x7E, 0x72, 0x21, 0x65, 0xF6, 0x96, 0xF0, 0xC8, 0x16, 0xCC, 0xC5, 0x9D, 0x88,
						0x24, 0x41, 0xF3, 0xB3, 0x39, 0x34, 0xD2, 0xA1, 0x5B, 0xF9, 0xC7, 0xAF, 0xF6, 0x8F, 0x4E, 0x78,
						0x01, 0xDB, 0xB2, 0xAA, 0x7E, 0x7E, 0xC7, 0x41, 0x31, 0xF4, 0x8D, 0xD4, 0x8E, 0x98, 0x7B, 0x16,
						0x4E, 0x96, 0x26, 0x71, 0x9D, 0x1F, 0x84, 0xEC, 0x68, 0x9A, 0x31
				}, 91, PAL_ECP_DP_SECP256R1, PAL_CHECK_PUBLIC_KEY, PAL_ERR_PK_KEY_INVALID_FORMAT, PAL_ERR_PARSING_PRIVATE_KEY, true, PAL_SUCCESS, PAL_SUCCESS
		}
};

typedef struct palParseECKeyTestVector{
	const unsigned char key[250];
	size_t len;
	uint32_t type;
	bool isDER;
	bool shouldSucceed;

}palParseECKeyTestVector_t;

static const palParseECKeyTestVector_t parse_ec_key_data[8] =
{
		{
				{0x00, 0x00}, 0, PAL_CHECK_PRIVATE_KEY, true, false
		},
		{
				{0x00, 0x00}, 0, PAL_CHECK_PUBLIC_KEY, true, false
		},
		{
				{
						0x30, 0x77, 0x02, 0x01, 0x01, 0x04, 0x20, 0xBD, 0x42, 0xD6, 0x36, 0x31, 0x2D, 0xF3, 0x2B, 0x31,
						0xEB, 0xE6, 0xE3, 0xC8, 0x63, 0x61, 0xA8, 0x45, 0x92, 0x2C, 0x70, 0xAB, 0x02, 0xC7, 0x45, 0xA7,
						0xBA, 0x7F, 0x39, 0xD3, 0xFD, 0xF0, 0x07, 0xA0, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
						0x03, 0x01, 0x07, 0xA1, 0x44, 0x03, 0x42, 0x00, 0x04, 0x7A, 0xDC, 0x03, 0xFE, 0xEF, 0x2B, 0x2B,
						0xF4, 0xE4, 0x6B, 0xAB, 0xA4, 0xD8, 0xEE, 0x2D, 0xAE, 0xA5, 0xD2, 0x28, 0xC5, 0xC1, 0xB2, 0x3C,
						0x7A, 0xAA, 0x5D, 0xE7, 0x81, 0x09, 0x1F, 0xE5, 0x9D, 0x80, 0xD4, 0xE9, 0xC5, 0x99, 0xF8, 0xBB,
						0xB1, 0x7B, 0xCB, 0x9A, 0x48, 0x2E, 0xF9, 0xEB, 0x01, 0xA3, 0xA4, 0x81, 0x70, 0x29, 0x34, 0xDB,
						0xE0, 0x65, 0x68, 0x48, 0x62, 0x99, 0x6D, 0xEC, 0x2E
				},121, PAL_CHECK_PRIVATE_KEY, true, true
		},
		{
				{
						0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A,
						0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x7A, 0xDC, 0x03, 0xFE, 0xEF,
						0x2B, 0x2B, 0xF4, 0xE4, 0x6B, 0xAB, 0xA4, 0xD8, 0xEE, 0x2D, 0xAE, 0xA5, 0xD2, 0x28, 0xC5, 0xC1,
						0xB2, 0x3C, 0x7A, 0xAA, 0x5D, 0xE7, 0x81, 0x09, 0x1F, 0xE5, 0x9D, 0x80, 0xD4, 0xE9, 0xC5, 0x99,
						0xF8, 0xBB, 0xB1, 0x7B, 0xCB, 0x9A, 0x48, 0x2E, 0xF9, 0xEB, 0x01, 0xA3, 0xA4, 0x81, 0x70, 0x29,
						0x34, 0xDB, 0xE0, 0x65, 0x68, 0x48, 0x62, 0x99, 0x6D, 0xEC, 0x2E
				}, 91, PAL_CHECK_PUBLIC_KEY, true, true
		},
		{
				{
						0x30, 0x77, 0x02, 0x01, 0x01, 0x04, 0x20, 0xBD, 0x42, 0xD6, 0x36, 0x31, 0x2D, 0xF3, 0x2B, 0x31,
						0xEB, 0xE6, 0xE3, 0xC8, 0x63, 0x61, 0xA8, 0x45, 0x92, 0x2C, 0x70, 0xAB, 0x02, 0xC7, 0x45, 0xA7,
						0xBA, 0x7F, 0x39, 0xD3, 0xFD, 0xF0, 0x07, 0xA0, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
						0x03, 0x01, 0x07, 0xA1, 0x44, 0x03, 0x42, 0x00, 0x04, 0x7A, 0xDC, 0x03, 0xFE, 0xEF, 0x2B, 0x2B,
						0xF4, 0xE4, 0x6B, 0xAB, 0xA4, 0xD8, 0xEE, 0x2D, 0xAE, 0xA5, 0xD2, 0x28, 0xC5, 0xC1, 0xB2, 0x3C,
						0x7A, 0xAA, 0x5D, 0xE7, 0x81, 0x09, 0x1F, 0xE5, 0x9D, 0x80, 0xD4, 0xE9, 0xC5, 0x99, 0xF8, 0xBB,
						0xB1, 0x7B, 0xCB, 0x9A, 0x48, 0x2E, 0xF9, 0xEB, 0x01, 0xA3, 0xA4, 0x81, 0x70, 0x29, 0x34, 0xDB,
						0xE0, 0x65, 0x68, 0x48, 0x62, 0x99, 0x6D, 0xEC, 0x2E
				}, 121,	PAL_CHECK_PUBLIC_KEY, true, false
		},
		{
				{
						0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A,
						0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x7A, 0xDC, 0x03, 0xFE, 0xEF,
						0x2B, 0x2B, 0xF4, 0xE4, 0x6B, 0xAB, 0xA4, 0xD8, 0xEE, 0x2D, 0xAE, 0xA5, 0xD2, 0x28, 0xC5, 0xC1,
						0xB2, 0x3C, 0x7A, 0xAA, 0x5D, 0xE7, 0x81, 0x09, 0x1F, 0xE5, 0x9D, 0x80, 0xD4, 0xE9, 0xC5, 0x99,
						0xF8, 0xBB, 0xB1, 0x7B, 0xCB, 0x9A, 0x48, 0x2E, 0xF9, 0xEB, 0x01, 0xA3, 0xA4, 0x81, 0x70, 0x29,
						0x34, 0xDB, 0xE0, 0x65, 0x68, 0x48, 0x62, 0x99, 0x6D, 0xEC, 0x2E
				},91, PAL_CHECK_PRIVATE_KEY, true, false
		},
		{
				"-----BEGIN EC PRIVATE KEY-----\r\n"
				"MHcCAQEEIBnI7FMl8SVLh8u3jUyGtJRxtyzf8WhZ6zVlR+uezRi/oAoGCCqGSM49\r\n"
				"AwEHoUQDQgAE7XbxNHg17rUMJ8nwTSTwSRrskUVWVkrnB7HARGN8eX4vL6v75D/3\r\n"
				"WsseEnEVMAhsbPs7rDx7xaKStJQKKSX84w==\r\n"
				"-----END EC PRIVATE KEY-----\r\n",
				236, PAL_CHECK_PRIVATE_KEY, false, false
		},
		{
				"-----BEGIN PUBLIC KEY-----\r\n"
				"MFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAE7XbxNHg17rUMJ8nwTSTwSRrskUVW\r\n"
				"VkrnB7HARGN8eX4vL6v75D/3WsseEnEVMAhsbPs7rDx7xaKStJQKKSX84w==\r\n"
				"-----END PUBLIC KEY-----\r\n",
				185, PAL_CHECK_PUBLIC_KEY, false, false
		}
};


typedef struct palX509CSRTestVector{
	unsigned char prvkey[300];
	size_t prvkeyLen;
	unsigned char pubkey[300];
	size_t pubkeyLen;
	palMDType_t mdType;
	const char subject_name[50];
	uint32_t keyUsage;
	unsigned char derOut[300];
	size_t derOutLen;
}palX509CSRTestVector_t;

static const palX509CSRTestVector_t CsrTests[2] =
{
		{
				{
						0x30, 0x78, 0x02, 0x01, 0x01, 0x04, 0x21, 0x00, 0xAE, 0xD9, 0xBE, 0xEA, 0x76, 0x78, 0xF2, 0xD9,
						0xEC, 0x11, 0xC5, 0x49, 0x00, 0xDC, 0xB7, 0xB1, 0x06, 0xBD, 0xA5, 0xF6, 0xF7, 0x06, 0xBB, 0xC2,
						0x8A, 0x8D, 0x46, 0xCB, 0xCF, 0xAE, 0x1F, 0x22, 0xA0, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
						0x3D, 0x03, 0x01, 0x07, 0xA1, 0x44, 0x03, 0x42, 0x00, 0x04, 0xC9, 0xEB, 0xAC, 0x6F, 0x9C, 0x39,
						0x1D, 0xFC, 0xFE, 0xBD, 0x25, 0x69, 0x51, 0x56, 0x80, 0xAC, 0xBF, 0x1A, 0x6F, 0xD4, 0xD7, 0x2E,
						0x02, 0x07, 0xBF, 0xF8, 0x4C, 0xEF, 0xEC, 0x60, 0x41, 0xAC, 0xDC, 0x7D, 0xD6, 0xDE, 0xD9, 0xF4,
						0xAA, 0xB8, 0x81, 0x0F, 0x96, 0xB6, 0xB3, 0x4A, 0x47, 0xD3, 0x98, 0x3F, 0x52, 0x87, 0x62, 0x0D,
						0xC7, 0xA0, 0x40, 0xCD, 0x23, 0x03, 0xBD, 0x17, 0x1E, 0xDE
				},122,
				{
						0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A,
						0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xC9, 0xEB, 0xAC, 0x6F, 0x9C,
						0x39, 0x1D, 0xFC, 0xFE, 0xBD, 0x25, 0x69, 0x51, 0x56, 0x80, 0xAC, 0xBF, 0x1A, 0x6F, 0xD4, 0xD7,
						0x2E, 0x02, 0x07, 0xBF, 0xF8, 0x4C, 0xEF, 0xEC, 0x60, 0x41, 0xAC, 0xDC, 0x7D, 0xD6, 0xDE, 0xD9,
						0xF4, 0xAA, 0xB8, 0x81, 0x0F, 0x96, 0xB6, 0xB3, 0x4A, 0x47, 0xD3, 0x98, 0x3F, 0x52, 0x87, 0x62,
						0x0D, 0xC7, 0xA0, 0x40, 0xCD, 0x23, 0x03, 0xBD, 0x17, 0x1E, 0xDE
				}, 91, PAL_SHA256, "C=AU, ST=Some-State, O=Internet Widgits Pty Ltd", PAL_X509_KU_DIGITAL_SIGNATURE,
				{
						0x30, 0x82, 0x01, 0x20, 0x30, 0x81, 0xC5, 0x02, 0x01, 0x00, 0x30, 0x45, 0x31, 0x0B, 0x30, 0x09,
						0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
						0x04, 0x08, 0x13, 0x0A, 0x53, 0x6F, 0x6D, 0x65, 0x2D, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
						0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x18, 0x49, 0x6E, 0x74, 0x65, 0x72, 0x6E, 0x65,
						0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4C, 0x74,
						0x64, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08,
						0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xC9, 0xEB, 0xAC, 0x6F,
						0x9C, 0x39, 0x1D, 0xFC, 0xFE, 0xBD, 0x25, 0x69, 0x51, 0x56, 0x80, 0xAC, 0xBF, 0x1A, 0x6F, 0xD4,
						0xD7, 0x2E, 0x02, 0x07, 0xBF, 0xF8, 0x4C, 0xEF, 0xEC, 0x60, 0x41, 0xAC, 0xDC, 0x7D, 0xD6, 0xDE,
						0xD9, 0xF4, 0xAA, 0xB8, 0x81, 0x0F, 0x96, 0xB6, 0xB3, 0x4A, 0x47, 0xD3, 0x98, 0x3F, 0x52, 0x87,
						0x62, 0x0D, 0xC7, 0xA0, 0x40, 0xCD, 0x23, 0x03, 0xBD, 0x17, 0x1E, 0xDE, 0xA0, 0x1E, 0x30, 0x1C,
						0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x0E, 0x31, 0x0F, 0x30, 0x0D, 0x30,
						0x0B, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x04, 0x04, 0x03, 0x02, 0x01, 0x80, 0x30, 0x0C, 0x06, 0x08,
						0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x05, 0x00, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02,
						0x21, 0x00, 0xEF, 0xA5, 0x91, 0xE8, 0x43, 0x2C, 0x95, 0x54, 0x51, 0xFA, 0x75, 0xAE, 0xF3, 0xE5,
						0x5B, 0xAE, 0x37, 0x28, 0x43, 0x12, 0xBE, 0xCB, 0x54, 0x67, 0x1C, 0xF8, 0x5F, 0x28, 0xD8, 0x87,
						0x23, 0x35, 0x02, 0x20, 0x32, 0xB7, 0x86, 0x38, 0xEF, 0x9F, 0x96, 0x25, 0x4B, 0xB5, 0xCD, 0x11,
						0xBB, 0x23, 0x3F, 0x93, 0x44, 0x31, 0x73, 0xF1, 0x0A, 0xBA, 0x2F, 0x43, 0x21, 0xFB, 0xBD, 0x4B,
						0xE5, 0xD4, 0xF0, 0x42
				}, 292
		},
		{
				{
						0x30, 0x77, 0x02, 0x01, 0x01, 0x04, 0x20, 0x12, 0x02, 0xAF, 0xCF, 0x14, 0xEC, 0xDB, 0x72, 0xDF,
						0x1C, 0x0C, 0xF1, 0xE3, 0x23, 0x97, 0x83, 0x62, 0x31, 0x71, 0x4C, 0xE6, 0x5B, 0x88, 0x9B, 0xF1,
						0x19, 0x40, 0xE4, 0xBE, 0xE9, 0x34, 0x76, 0xA0, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
						0x03, 0x01, 0x07, 0xA1, 0x44, 0x03, 0x42, 0x00, 0x04, 0x6D, 0xE7, 0x22, 0xF3, 0xCB, 0x5F, 0x84,
						0x83, 0xEE, 0x27, 0xE2, 0xA1, 0x24, 0xFF, 0xAC, 0x19, 0xF7, 0x2D, 0xDD, 0xFA, 0x69, 0x20, 0xBE,
						0x06, 0x7E, 0x8A, 0x20, 0x40, 0x26, 0x90, 0xE9, 0xFB, 0xBA, 0x86, 0x0A, 0xE0, 0x9A, 0x39, 0x1B,
						0x0F, 0xB8, 0x53, 0xD5, 0xFC, 0xE1, 0x5E, 0x94, 0xBC, 0x1E, 0x97, 0x9C, 0xC2, 0x7B, 0x4E, 0xF7,
						0x17, 0x36, 0xCA, 0x59, 0xD5, 0x01, 0xBB, 0x56, 0x74
				}, 121,
				{
						0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A,
						0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x6D, 0xE7, 0x22, 0xF3, 0xCB,
						0x5F, 0x84, 0x83, 0xEE, 0x27, 0xE2, 0xA1, 0x24, 0xFF, 0xAC, 0x19, 0xF7, 0x2D, 0xDD, 0xFA, 0x69,
						0x20, 0xBE, 0x06, 0x7E, 0x8A, 0x20, 0x40, 0x26, 0x90, 0xE9, 0xFB, 0xBA, 0x86, 0x0A, 0xE0, 0x9A,
						0x39, 0x1B, 0x0F, 0xB8, 0x53, 0xD5, 0xFC, 0xE1, 0x5E, 0x94, 0xBC, 0x1E, 0x97, 0x9C, 0xC2, 0x7B,
						0x4E, 0xF7, 0x17, 0x36, 0xCA, 0x59, 0xD5, 0x01, 0xBB, 0x56, 0x74
				}, 91, PAL_SHA256, "C=AU, ST=Some-State, O=Internet Widgits Pty Ltd", PAL_X509_KU_NON_REPUDIATION,
				{
						0x30, 0x82, 0x01, 0x20, 0x30, 0x81, 0xC5, 0x02, 0x01, 0x00, 0x30, 0x45, 0x31, 0x0B, 0x30, 0x09,
						0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
						0x04, 0x08, 0x13, 0x0A, 0x53, 0x6F, 0x6D, 0x65, 0x2D, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
						0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x18, 0x49, 0x6E, 0x74, 0x65, 0x72, 0x6E, 0x65,
						0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4C, 0x74,
						0x64, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08,
						0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x6D, 0xE7, 0x22, 0xF3,
						0xCB, 0x5F, 0x84, 0x83, 0xEE, 0x27, 0xE2, 0xA1, 0x24, 0xFF, 0xAC, 0x19, 0xF7, 0x2D, 0xDD, 0xFA,
						0x69, 0x20, 0xBE, 0x06, 0x7E, 0x8A, 0x20, 0x40, 0x26, 0x90, 0xE9, 0xFB, 0xBA, 0x86, 0x0A, 0xE0,
						0x9A, 0x39, 0x1B, 0x0F, 0xB8, 0x53, 0xD5, 0xFC, 0xE1, 0x5E, 0x94, 0xBC, 0x1E, 0x97, 0x9C, 0xC2,
						0x7B, 0x4E, 0xF7, 0x17, 0x36, 0xCA, 0x59, 0xD5, 0x01, 0xBB, 0x56, 0x74, 0xA0, 0x1E, 0x30, 0x1C,
						0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x0E, 0x31, 0x0F, 0x30, 0x0D, 0x30,
						0x0B, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x04, 0x04, 0x03, 0x02, 0x01, 0x40, 0x30, 0x0C, 0x06, 0x08,
						0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x05, 0x00, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02,
						0x21, 0x00, 0xEC, 0xA1, 0xDE, 0x83, 0x89, 0x1A, 0x05, 0xA6, 0x38, 0x7B, 0xC1, 0xDB, 0x0D, 0x67,
						0xEC, 0x9E, 0x98, 0x3C, 0x92, 0xB3, 0x58, 0x06, 0x34, 0x87, 0x19, 0xAB, 0x57, 0x2B, 0x70, 0x29,
						0x4C, 0x3D, 0x02, 0x20, 0x6A, 0x9D, 0x7D, 0xFF, 0x8B, 0x00, 0x74, 0x24, 0xA1, 0xD6, 0xD6, 0xEF,
						0xF7, 0x70, 0x86, 0x8D, 0x2D, 0x52, 0x68, 0x55, 0x18, 0x7C, 0x45, 0xFB, 0xEE, 0x49, 0x50, 0x62,
						0x9E, 0x6C, 0x1C, 0xCA
				}, 292
		}
};

const unsigned char cert_not_self_signed[] =
{
		0x30, 0x82, 0x01, 0xc4, 0x30, 0x82, 0x01, 0x69, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x01,
		0x30, 0x0c, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x05, 0x00, 0x30, 0x3d,
		0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x07, 0x49, 0x4f, 0x54, 0x5f, 0x50,
		0x41, 0x4c, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x03, 0x50, 0x41, 0x4c,
		0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x49, 0x4c, 0x31, 0x0e, 0x30,
		0x0c, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x05, 0x49, 0x4f, 0x54, 0x42, 0x55, 0x30, 0x1e, 0x17,
		0x0d, 0x31, 0x33, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0x0d,
		0x31, 0x35, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x42, 0x31,
		0x11, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x08, 0x49, 0x4f, 0x54, 0x5f, 0x54, 0x45,
		0x53, 0x54, 0x31, 0x0d, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x04, 0x50, 0x41, 0x41,
		0x4c, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x49, 0x4c, 0x31, 0x11,
		0x30, 0x0f, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x08, 0x49, 0x4f, 0x54, 0x42, 0x55, 0x5f, 0x49,
		0x4c, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08,
		0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x18, 0xf7, 0x26, 0xc6,
		0x86, 0x30, 0xec, 0xed, 0xd6, 0xb4, 0x3a, 0xd3, 0x86, 0x7a, 0x49, 0xbb, 0xb3, 0x93, 0xee, 0x43,
		0x56, 0x18, 0x71, 0x2a, 0x40, 0xda, 0xc8, 0x2c, 0x4a, 0xc6, 0x3d, 0x09, 0xfc, 0xe0, 0x84, 0x2f,
		0x0d, 0xce, 0xf8, 0x17, 0xcf, 0x28, 0x8f, 0x4f, 0xb4, 0xea, 0xde, 0xf9, 0xe4, 0x9a, 0x16, 0xeb,
		0x8c, 0x67, 0x02, 0xfd, 0x64, 0x7b, 0x62, 0x0e, 0x9b, 0x29, 0x05, 0x22, 0xa3, 0x53, 0x30, 0x51,
		0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01,
		0x00, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x13, 0xde, 0xa2, 0x8b,
		0x8d, 0x40, 0x1e, 0xd6, 0x62, 0xfd, 0x7e, 0x27, 0x53, 0xb8, 0x8f, 0x17, 0x18, 0x76, 0x33, 0x20,
		0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x13, 0xde, 0xa2,
		0x8b, 0x8d, 0x40, 0x1e, 0xd6, 0x62, 0xfd, 0x7e, 0x27, 0x53, 0xb8, 0x8f, 0x17, 0x18, 0x76, 0x33,
		0x20, 0x30, 0x0c, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x05, 0x00, 0x03,
		0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x47, 0xb7, 0xf8, 0x12, 0xb0, 0xe7, 0x7b, 0x33, 0x7f, 0x31,
		0x5e, 0x2f, 0x54, 0x40, 0x8f, 0xff, 0xe1, 0x05, 0x1b, 0xe5, 0xf1, 0xd5, 0xa7, 0x6a, 0xb8, 0x2a,
		0x6f, 0x22, 0x5a, 0xce, 0x31, 0x84, 0x02, 0x20, 0x51, 0x5d, 0x68, 0x41, 0x7f, 0x26, 0xb2, 0x6b,
		0x71, 0x1e, 0xd2, 0xa2, 0x2d, 0xd9, 0x43, 0x69, 0xbf, 0xf0, 0x73, 0xf6, 0x01, 0x7f, 0xf7, 0xac,
		0x0e, 0x5c, 0xb4, 0x4e, 0x56, 0x73, 0x63, 0x3c
};

//Certificate based on unsupported hash – sha512
const uint8_t testdata_x509_Sha512[506] = 
		{ 
			0x30, 0x82, 0x01, 0xf6, 0x30, 0x82, 0x01, 0x9c, 0x02, 0x09, 0x00, 0xb4, 0x60, 0xcc, 0x6f, 0xa2, 
			0x55, 0x52, 0xfe, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x04, 0x30, 
			0x81, 0x82, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x4b, 0x31, 
			0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x4b, 0x66, 0x61, 0x72, 0x2d, 0x6e, 
			0x65, 0x74, 0x65, 0x72, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x4e, 
			0x65, 0x74, 0x61, 0x6e, 0x69, 0x61, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 
			0x03, 0x41, 0x52, 0x4d, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x03, 0x49, 
			0x4f, 0x54, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x06, 0x44, 0x65, 0x76, 
			0x69, 0x63, 0x65, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 
			0x09, 0x01, 0x16, 0x10, 0x64, 0x65, 0x76, 0x69, 0x63, 0x65, 0x30, 0x31, 0x40, 0x61, 0x72, 0x6d, 
			0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x37, 0x30, 0x39, 0x31, 0x30, 0x31, 0x34, 
			0x30, 0x33, 0x33, 0x33, 0x5a, 0x17, 0x0d, 0x32, 0x33, 0x30, 0x33, 0x30, 0x33, 0x31, 0x34, 0x30, 
			0x33, 0x33, 0x33, 0x5a, 0x30, 0x81, 0x82, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 
			0x13, 0x02, 0x55, 0x4b, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x4b, 
			0x66, 0x61, 0x72, 0x2d, 0x6e, 0x65, 0x74, 0x65, 0x72, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 
			0x04, 0x07, 0x0c, 0x07, 0x4e, 0x65, 0x74, 0x61, 0x6e, 0x69, 0x61, 0x31, 0x0c, 0x30, 0x0a, 0x06, 
			0x03, 0x55, 0x04, 0x0a, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 
			0x04, 0x0b, 0x0c, 0x03, 0x49, 0x4f, 0x54, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x03, 
			0x0c, 0x06, 0x44, 0x65, 0x76, 0x69, 0x63, 0x65, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 
			0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x64, 0x65, 0x76, 0x69, 0x63, 0x65, 0x30, 
			0x31, 0x40, 0x61, 0x72, 0x6d, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 
			0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 
			0x03, 0x42, 0x00, 0x04, 0x41, 0xdf, 0x03, 0x36, 0x9c, 0xb3, 0xb2, 0x9e, 0x42, 0xbc, 0x48, 0x37, 
			0x3c, 0x21, 0xc2, 0x99, 0xb1, 0x47, 0x1d, 0x96, 0x84, 0x28, 0xdb, 0x07, 0x30, 0xad, 0x69, 0x86, 
			0x39, 0x10, 0x10, 0x52, 0xe9, 0x3a, 0xe5, 0x4e, 0xd3, 0x3f, 0xd3, 0x0d, 0xfd, 0x9f, 0xd9, 0x4b, 
			0xcb, 0xa1, 0x81, 0xbe, 0x3f, 0xbe, 0x24, 0x68, 0xc3, 0xe9, 0x73, 0xbf, 0x34, 0xb7, 0xa0, 0x61, 
			0x56, 0x96, 0xb5, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x04, 
			0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0xa0, 0xa6, 0x46, 0xaa, 0xef, 0x18, 0xa6, 0xa3, 
			0x7b, 0x81, 0xdb, 0x36, 0xf7, 0x2c, 0xbc, 0xb4, 0x29, 0x7f, 0x3c, 0x35, 0xbd, 0x88, 0xba, 0xa6, 
			0x98, 0x8f, 0x5f, 0x67, 0x35, 0xf8, 0xf2, 0x18, 0x02, 0x20, 0x65, 0x88, 0x9e, 0x5d, 0x5a, 0x84, 
			0xb7, 0x42, 0x96, 0xe4, 0xb7, 0x7d, 0x9b, 0x5c, 0x01, 0xc3, 0x5a, 0x40, 0x11, 0x0d, 0xf5, 0xfe, 
			0x44, 0x9c, 0xbb, 0xd2, 0x2f, 0x37, 0x38, 0xbe, 0x03, 0x56};

//Certificate based on unsupported curve – secp521r1 
const uint8_t testdata_x509_Curve512r1[641] = 
		{ 
			0x30, 0x82, 0x02, 0x7d, 0x30, 0x82, 0x01, 0xdf, 0x02, 0x09, 0x00, 0xe2, 0x16, 0x36, 0x0a, 0x28,
			0x52, 0x3e, 0x81, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30,
			0x81, 0x82, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x4b, 0x31,
			0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x4b, 0x66, 0x61, 0x72, 0x2d, 0x6e,
			0x65, 0x74, 0x65, 0x72, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x4e,
			0x65, 0x74, 0x61, 0x6e, 0x69, 0x61, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c,
			0x03, 0x41, 0x52, 0x4d, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x03, 0x49,
			0x4f, 0x54, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x06, 0x44, 0x65, 0x76,
			0x69, 0x63, 0x65, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
			0x09, 0x01, 0x16, 0x10, 0x64, 0x65, 0x76, 0x69, 0x63, 0x65, 0x30, 0x31, 0x40, 0x61, 0x72, 0x6d,
			0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x37, 0x31, 0x30, 0x30, 0x31, 0x30, 0x38,
			0x35, 0x36, 0x35, 0x31, 0x5a, 0x17, 0x0d, 0x32, 0x33, 0x30, 0x33, 0x32, 0x34, 0x30, 0x38, 0x35,
			0x36, 0x35, 0x31, 0x5a, 0x30, 0x81, 0x82, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
			0x13, 0x02, 0x55, 0x4b, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x4b,
			0x66, 0x61, 0x72, 0x2d, 0x6e, 0x65, 0x74, 0x65, 0x72, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55,
			0x04, 0x07, 0x0c, 0x07, 0x4e, 0x65, 0x74, 0x61, 0x6e, 0x69, 0x61, 0x31, 0x0c, 0x30, 0x0a, 0x06,
			0x03, 0x55, 0x04, 0x0a, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55,
			0x04, 0x0b, 0x0c, 0x03, 0x49, 0x4f, 0x54, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x03,
			0x0c, 0x06, 0x44, 0x65, 0x76, 0x69, 0x63, 0x65, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86,
			0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x64, 0x65, 0x76, 0x69, 0x63, 0x65, 0x30,
			0x31, 0x40, 0x61, 0x72, 0x6d, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x81, 0x9b, 0x30, 0x10, 0x06, 0x07,
			0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x05, 0x2b, 0x81, 0x04, 0x00, 0x23, 0x03, 0x81,
			0x86, 0x00, 0x04, 0x00, 0xf6, 0x6f, 0x0f, 0xbe, 0x33, 0x95, 0xa3, 0xd1, 0x00, 0x47, 0x27, 0xcf,
			0xd6, 0x3d, 0x0b, 0x69, 0x77, 0x40, 0xba, 0xaa, 0x82, 0xf4, 0xc3, 0x81, 0x41, 0x2b, 0xbd, 0x7b,
			0x2a, 0xf5, 0xd3, 0xf2, 0x09, 0x5a, 0x61, 0xb4, 0x5d, 0x6e, 0x32, 0x10, 0x61, 0xfd, 0x6e, 0x5a,
			0x74, 0x89, 0xad, 0xba, 0x23, 0x92, 0x33, 0x37, 0xa0, 0x4f, 0x98, 0x30, 0x4d, 0x8b, 0xae, 0xbf,
			0x5f, 0x49, 0x56, 0x17, 0x69, 0x00, 0x37, 0x1b, 0xde, 0xa0, 0x11, 0x38, 0x17, 0xb5, 0x15, 0x61,
			0xbd, 0xa1, 0xd8, 0x8d, 0x71, 0x48, 0x27, 0x4d, 0xbe, 0x60, 0x74, 0x3a, 0xd1, 0xe1, 0xff, 0xea,
			0x0a, 0xa2, 0x3d, 0xaa, 0x28, 0x39, 0x28, 0x32, 0xa7, 0x0c, 0x9d, 0xfa, 0x52, 0xc1, 0x2c, 0xba,
			0xac, 0x0f, 0x42, 0xa2, 0xcc, 0x78, 0x01, 0x45, 0x11, 0x62, 0xcf, 0x4f, 0x87, 0xcb, 0xf1, 0xff,
			0x2a, 0x1a, 0x30, 0xe9, 0x3d, 0x31, 0x59, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d,
			0x04, 0x03, 0x02, 0x03, 0x81, 0x8b, 0x00, 0x30, 0x81, 0x87, 0x02, 0x41, 0x31, 0x73, 0x8d, 0x4a,
			0x28, 0x86, 0xa9, 0x06, 0xbd, 0x7f, 0xe8, 0x58, 0x43, 0xc2, 0xce, 0xd9, 0xdb, 0x75, 0xae, 0x42,
			0x83, 0x98, 0x29, 0x28, 0x61, 0x02, 0xfc, 0x23, 0x54, 0xe3, 0xcd, 0x07, 0x6f, 0xdc, 0xa0, 0x2e,
			0xdd, 0xd3, 0xc6, 0x2e, 0x11, 0xcc, 0xd9, 0x3c, 0xdf, 0x26, 0xd4, 0x66, 0xdb, 0x03, 0x9e, 0x79,
			0x9a, 0x60, 0x8f, 0x1d, 0x52, 0x6e, 0x3d, 0x16, 0x2f, 0x7d, 0x86, 0x43, 0x98, 0x02, 0x42, 0x01,
			0xaa, 0xf2, 0x7f, 0xad, 0xad, 0xaa, 0x16, 0x35, 0x66, 0xe7, 0x52, 0xcc, 0x60, 0xdd, 0x09, 0xae,
			0x1b, 0x44, 0x33, 0xa1, 0x57, 0x3a, 0xa0, 0xc5, 0xe3, 0x08, 0x84, 0xa7, 0xe3, 0x59, 0xf3, 0x79,
			0x9a, 0xbf, 0xcb, 0x26, 0x36, 0x45, 0x64, 0x4b, 0x92, 0xb8, 0xc7, 0x51, 0x14, 0x57, 0x94, 0x4c,
			0x96, 0x59, 0x2a, 0x5f, 0xb1, 0x38, 0x84, 0xc3, 0x26, 0x51, 0x57, 0x64, 0x4e, 0x54, 0x30, 0x3e,
			0x00};

const unsigned char x509_TI[] =
{
		0x30, 0x82, 0x01, 0xc1, 0x30, 0x82, 0x01, 0x64, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x01,
		0x30, 0x0c, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x05, 0x00, 0x30, 0x3d,
		0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x07, 0x49, 0x4f, 0x54, 0x5f, 0x50,
		0x41, 0x4c, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x03, 0x50, 0x41, 0x4c,
		0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x49, 0x4c, 0x31, 0x0e, 0x30,
		0x0c, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x05, 0x49, 0x4f, 0x54, 0x42, 0x55, 0x30, 0x1e, 0x17,
		0x0d, 0x31, 0x33, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0x0d,
		0x31, 0x35, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x3d, 0x31,
		0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x07, 0x49, 0x4f, 0x54, 0x5f, 0x50, 0x41,
		0x4c, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x03, 0x50, 0x41, 0x4c, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x49, 0x4c, 0x31, 0x0e, 0x30, 0x0c,
		0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x05, 0x49, 0x4f, 0x54, 0x42, 0x55, 0x30, 0x59, 0x30, 0x13,
		0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d,
		0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x18, 0xf7, 0x26, 0xc6, 0x86, 0x30, 0xec, 0xed, 0xd6,
		0xb4, 0x3a, 0xd3, 0x86, 0x7a, 0x49, 0xbb, 0xb3, 0x93, 0xee, 0x43, 0x56, 0x18, 0x71, 0x2a, 0x40,
		0xda, 0xc8, 0x2c, 0x4a, 0xc6, 0x3d, 0x09, 0xfc, 0xe0, 0x84, 0x2f, 0x0d, 0xce, 0xf8, 0x17, 0xcf,
		0x28, 0x8f, 0x4f, 0xb4, 0xea, 0xde, 0xf9, 0xe4, 0x9a, 0x16, 0xeb, 0x8c, 0x67, 0x02, 0xfd, 0x64,
		0x7b, 0x62, 0x0e, 0x9b, 0x29, 0x05, 0x22, 0xa3, 0x53, 0x30, 0x51, 0x30, 0x0f, 0x06, 0x03, 0x55,
		0x1d, 0x13, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x1d, 0x06, 0x03,
		0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x13, 0xde, 0xa2, 0x8b, 0x8d, 0x40, 0x1e, 0xd6, 0x62,
		0xfd, 0x7e, 0x27, 0x53, 0xb8, 0x8f, 0x17, 0x18, 0x76, 0x33, 0x20, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x13, 0xde, 0xa2, 0x8b, 0x8d, 0x40, 0x1e, 0xd6,
		0x62, 0xfd, 0x7e, 0x27, 0x53, 0xb8, 0x8f, 0x17, 0x18, 0x76, 0x33, 0x20, 0x30, 0x0c, 0x06, 0x08,
		0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x05, 0x00, 0x03, 0x49, 0x00, 0x30, 0x46, 0x02,
		0x21, 0x00, 0xb7, 0xbb, 0x7b, 0x05, 0x17, 0x5f, 0x74, 0x0a, 0x91, 0xd7, 0x95, 0x30, 0xa8, 0xb4,
		0x47, 0x26, 0x2d, 0xc4, 0xbf, 0xb4, 0xba, 0x00, 0x54, 0x92, 0xcd, 0xb8, 0xb4, 0x42, 0x4b, 0x05,
		0x95, 0xe6, 0x02, 0x21, 0x00, 0xdb, 0x61, 0x24, 0x78, 0xed, 0x54, 0xcd, 0x8e, 0xfa, 0x31, 0x21,
		0x8b, 0xe7, 0x11, 0x7e, 0x6e, 0xe2, 0x17, 0x63, 0xbe, 0x31, 0x90, 0x2b, 0xf8, 0xd2, 0x11, 0x5d,
		0x9d, 0x8d, 0x5a, 0xaa, 0xdd
};

const unsigned char x509_TI_PEM[] = "-----BEGIN CERTIFICATE-----\r\n"
		"MIIFIzCCBAugAwIBAgIQVYyM7bV85U+m+zUfuEsiLzANBgkqhkiG9w0BAQUFADCB\r\n"
		"tDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\r\n"
		"ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2Ug\r\n"
		"YXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykxMDEuMCwGA1UEAxMl\r\n"
		"VmVyaVNpZ24gQ2xhc3MgMyBDb2RlIFNpZ25pbmcgMjAxMCBDQTAeFw0xNDA0MTUw\r\n"
		"MDAwMDBaFw0xNTA0MTYyMzU5NTlaMHQxCzAJBgNVBAYTAlVTMQ4wDAYDVQQIEwVU\r\n"
		"ZXhhczEPMA0GA1UEBxMGRGFsbGFzMRowGAYDVQQKFBFUZXhhcyBJbnN0cnVtZW50\r\n"
		"czEMMAoGA1UECxQDV0NTMRowGAYDVQQDFBFUZXhhcyBJbnN0cnVtZW50czCCASIw\r\n"
		"DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMuAIvJOnebITQe2qli+t6Juwoy7\r\n"
		"I9qenavTt7D2cYsEBO+Oj/SLWjkieGPKek1Q+DIo66h3ZAYB9mDhxIAxY8bMmO61\r\n"
		"SOwxtcjA7nLgoqacMi61ly5OmjDuTholKjCyFCYmun7YPiI4b0BRNwFIMxOjfKdc\r\n"
		"9Jj1UCl7TH0DFsn1ly583NOlJey7AV5v4sijU5AWSR+JeaV/yFw1MPbY58ekjnS8\r\n"
		"tv5K8r5k5D770jDOZ/nuLpVZ3oNvhP9uMb6abEcUcVnxHkPu8ovFa43T1FwAKYSF\r\n"
		"4FAPpe4P9EQVUBJAupMt91J6iPWeiSjKYMYbd3dgNHNabsoT1gACAhCWwJkCAwEA\r\n"
		"AaOCAW4wggFqMAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoG\r\n"
		"CCsGAQUFBwMDMGYGA1UdIARfMF0wWwYLYIZIAYb4RQEHFwMwTDAjBggrBgEFBQcC\r\n"
		"ARYXaHR0cHM6Ly9kLnN5bWNiLmNvbS9jcHMwJQYIKwYBBQUHAgIwGRoXaHR0cHM6\r\n"
		"Ly9kLnN5bWNiLmNvbS9ycGEwHwYDVR0jBBgwFoAUz5mp6nsm9EvJjo/X8AUm7+PS\r\n"
		"p50wKwYDVR0fBCQwIjAgoB6gHIYaaHR0cDovL3NmLnN5bWNiLmNvbS9zZi5jcmww\r\n"
		"VwYIKwYBBQUHAQEESzBJMB8GCCsGAQUFBzABhhNodHRwOi8vc2Yuc3ltY2QuY29t\r\n"
		"MCYGCCsGAQUFBzAChhpodHRwOi8vc2Yuc3ltY2IuY29tL3NmLmNydDARBglghkgB\r\n"
		"hvhCAQEEBAMCBBAwFgYKKwYBBAGCNwIBGwQIMAYBAQABAf8wDQYJKoZIhvcNAQEF\r\n"
		"BQADggEBACKIttb6LKlpjDHUFo9BHdYw6qxbF3Ham5GPjAn+ofYqpv3KoF7L2Rc9\r\n"
		"jhYDIA5IHmYAjDdHLUBx+DdzMP80cXcRIdBGgOFFw++rRboUi0yOpg47yD0AipNT\r\n"
		"Q00rBdoeEzs3ERiPir20402OmFUBkCcwM85XaXh1hvOKDZaCEj/6vlK/Y6++y1QC\r\n"
		"tQGm08HAAuEz4VsnE/mcVeuaVM8DqFInrp9mpCNiCCwdiG3TEC1UgACjxFmFk0UH\r\n"
		"mrUNww35tofUjAoXIkcegm2d/ld8bZxludWG2sJRydVTAQe1jJJDg+N7IdyH/gv7\r\n"
		"y8i1xf5goGEvMqLiYY5njLCSSpCeNqE=\r\n"
		"-----END CERTIFICATE-----\r\n";

const unsigned char x509_selfsign[] =
{
		0x30, 0x82, 0x01, 0x9a, 0x30, 0x82, 0x01, 0x3e, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x01,
		0x30, 0x0c, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x05, 0x00, 0x30, 0x2d,
		0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x02, 0x43, 0x41, 0x31, 0x11, 0x30,
		0x0f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x08, 0x6d, 0x62, 0x65, 0x64, 0x20, 0x54, 0x4c, 0x53,
		0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x4b, 0x30, 0x1e, 0x17,
		0x0d, 0x30, 0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0x0d,
		0x33, 0x30, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x2d, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x02, 0x43, 0x41, 0x31, 0x11, 0x30, 0x0f,
		0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x08, 0x6d, 0x62, 0x65, 0x64, 0x20, 0x54, 0x4c, 0x53, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x4b, 0x30, 0x59, 0x30, 0x13,
		0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d,
		0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xf7, 0xc6, 0x63, 0x68, 0x8d, 0x48, 0x3a, 0xcc, 0xeb,
		0xc1, 0x01, 0xcb, 0xd5, 0xc9, 0xa9, 0xc8, 0x42, 0x62, 0x2e, 0xd0, 0xb2, 0x34, 0x6a, 0x9f, 0xc9,
		0xce, 0xe0, 0x1c, 0x57, 0xc7, 0x0a, 0x62, 0x6e, 0x8e, 0x2e, 0xc5, 0x9f, 0xef, 0x8e, 0x04, 0x44,
		0x7e, 0xf3, 0xd0, 0xe6, 0x92, 0xc9, 0x0a, 0x49, 0x72, 0x98, 0x7f, 0x73, 0x3e, 0xf6, 0x97, 0x70,
		0x74, 0xb7, 0x9f, 0xe1, 0xb5, 0xef, 0xce, 0xa3, 0x4d, 0x30, 0x4b, 0x30, 0x09, 0x06, 0x03, 0x55,
		0x1d, 0x13, 0x04, 0x02, 0x30, 0x00, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04,
		0x14, 0x0f, 0xc6, 0x27, 0x51, 0xce, 0x06, 0x0b, 0x3b, 0xbd, 0xf1, 0xd1, 0x8e, 0x25, 0x6d, 0xcd,
		0x12, 0x64, 0x8f, 0x73, 0x4d, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16,
		0x80, 0x14, 0x0f, 0xc6, 0x27, 0x51, 0xce, 0x06, 0x0b, 0x3b, 0xbd, 0xf1, 0xd1, 0x8e, 0x25, 0x6d,
		0xcd, 0x12, 0x64, 0x8f, 0x73, 0x4d, 0x30, 0x0c, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04,
		0x03, 0x02, 0x05, 0x00, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0xf1, 0x6a, 0x24, 0x08,
		0x99, 0xf1, 0xe0, 0x55, 0x4f, 0x0a, 0x52, 0x72, 0x68, 0x63, 0x6e, 0xec, 0x2f, 0x9d, 0x54, 0x61,
		0x9d, 0x7f, 0xa3, 0x5d, 0xa5, 0x79, 0x69, 0xc2, 0xab, 0xcf, 0x0f, 0x48, 0x02, 0x20, 0x51, 0x41,
		0x26, 0x4b, 0xdf, 0x1b, 0x5e, 0xda, 0xa4, 0x53, 0x8d, 0xe5, 0x92, 0xa6, 0x1b, 0x8a, 0x0d, 0x58,
		0x59, 0x9e, 0x77, 0x99, 0xd6, 0x81, 0xf1, 0x5d, 0xcd, 0x28, 0xcc, 0x03, 0x75, 0x96
};

const unsigned char x509_verify_ca[] =
{
		0x30, 0x82, 0x01, 0xf6, 0x30, 0x82, 0x01, 0x9c, 0x02, 0x09, 0x00, 0xe3, 0x22, 0x71, 0x24, 0xb8,
		0xa6, 0x9b, 0xce, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30,
		0x81, 0x82, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x4b, 0x31,
		0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x4b, 0x66, 0x61, 0x72, 0x2d, 0x6e,
		0x65, 0x74, 0x65, 0x72, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x4e,
		0x65, 0x74, 0x61, 0x6e, 0x69, 0x61, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c,
		0x03, 0x41, 0x52, 0x4d, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x03, 0x49,
		0x4f, 0x54, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x06, 0x44, 0x65, 0x76,
		0x69, 0x63, 0x65, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
		0x09, 0x01, 0x16, 0x10, 0x64, 0x65, 0x76, 0x69, 0x63, 0x65, 0x30, 0x31, 0x40, 0x61, 0x72, 0x6d,
		0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x37, 0x30, 0x33, 0x30, 0x31, 0x31, 0x37,
		0x34, 0x33, 0x35, 0x33, 0x5a, 0x17, 0x0d, 0x32, 0x32, 0x30, 0x38, 0x32, 0x32, 0x31, 0x37, 0x34,
		0x33, 0x35, 0x33, 0x5a, 0x30, 0x81, 0x82, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x55, 0x4b, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x4b,
		0x66, 0x61, 0x72, 0x2d, 0x6e, 0x65, 0x74, 0x65, 0x72, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55,
		0x04, 0x07, 0x0c, 0x07, 0x4e, 0x65, 0x74, 0x61, 0x6e, 0x69, 0x61, 0x31, 0x0c, 0x30, 0x0a, 0x06,
		0x03, 0x55, 0x04, 0x0a, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55,
		0x04, 0x0b, 0x0c, 0x03, 0x49, 0x4f, 0x54, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x03,
		0x0c, 0x06, 0x44, 0x65, 0x76, 0x69, 0x63, 0x65, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86,
		0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x64, 0x65, 0x76, 0x69, 0x63, 0x65, 0x30,
		0x31, 0x40, 0x61, 0x72, 0x6d, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a,
		0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
		0x03, 0x42, 0x00, 0x04, 0x24, 0xb5, 0x1b, 0xaf, 0x82, 0xd6, 0xc6, 0xc4, 0x66, 0xba, 0xe7, 0xdf,
		0x41, 0xd2, 0x03, 0xdb, 0xbf, 0x49, 0xf0, 0x99, 0x66, 0xf1, 0xe1, 0x15, 0x37, 0x94, 0x66, 0x2b,
		0x22, 0x4c, 0x8e, 0xc6, 0xd4, 0x5e, 0x6f, 0x1a, 0xdb, 0x44, 0xe5, 0x06, 0xd8, 0x34, 0xad, 0xf2,
		0x1e, 0x92, 0x23, 0xd8, 0xd4, 0xd2, 0x01, 0x6a, 0x6e, 0x3c, 0x09, 0x7b, 0x5b, 0xff, 0x0e, 0x6f,
		0xb4, 0x28, 0xe6, 0x28, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02,
		0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0xae, 0xaf, 0x9e, 0xfb, 0xea, 0x53, 0xe7, 0xe3,
		0xd5, 0xe0, 0x90, 0x87, 0x15, 0xaf, 0x48, 0x9f, 0x50, 0x2c, 0x44, 0x4d, 0x2a, 0x68, 0x38, 0xa0,
		0x3c, 0x09, 0xbc, 0x32, 0xb8, 0x78, 0xcf, 0xfa, 0x02, 0x20, 0x0c, 0x0d, 0xc8, 0xda, 0xcd, 0x09,
		0x7a, 0xa9, 0x6d, 0x18, 0x6c, 0xc1, 0x58, 0x1a, 0xec, 0x9a, 0x8b, 0x0d, 0x8a, 0xc4, 0xea, 0x9d,
		0x49, 0x5f, 0xa1, 0x9e, 0xcc, 0xcf, 0xbe, 0xb5, 0x25, 0xfd
};

const unsigned char x509_verify_cert[] =
{
		0x30, 0x82, 0x01, 0xf2, 0x30, 0x82, 0x01, 0x98, 0x02, 0x01, 0x03, 0x30, 0x0a, 0x06, 0x08, 0x2a,
		0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x81, 0x82, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03,
		0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x4b, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08,
		0x0c, 0x0a, 0x4b, 0x66, 0x61, 0x72, 0x2d, 0x6e, 0x65, 0x74, 0x65, 0x72, 0x31, 0x10, 0x30, 0x0e,
		0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x4e, 0x65, 0x74, 0x61, 0x6e, 0x69, 0x61, 0x31, 0x0c,
		0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x31, 0x0c, 0x30, 0x0a,
		0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x03, 0x49, 0x4f, 0x54, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03,
		0x55, 0x04, 0x03, 0x0c, 0x06, 0x44, 0x65, 0x76, 0x69, 0x63, 0x65, 0x31, 0x1f, 0x30, 0x1d, 0x06,
		0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x64, 0x65, 0x76, 0x69,
		0x63, 0x65, 0x30, 0x31, 0x40, 0x61, 0x72, 0x6d, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d,
		0x31, 0x37, 0x30, 0x33, 0x30, 0x31, 0x31, 0x37, 0x34, 0x33, 0x35, 0x33, 0x5a, 0x17, 0x0d, 0x31,
		0x37, 0x30, 0x33, 0x33, 0x31, 0x31, 0x37, 0x34, 0x33, 0x35, 0x33, 0x5a, 0x30, 0x81, 0x86, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x4b, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x4b, 0x66, 0x61, 0x72, 0x2d, 0x6e, 0x65, 0x74, 0x65,
		0x72, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x4e, 0x65, 0x74, 0x61,
		0x6e, 0x69, 0x61, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x03, 0x41, 0x52,
		0x4d, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x03, 0x49, 0x4f, 0x54, 0x31,
		0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0a, 0x43, 0x68, 0x69, 0x6c, 0x64, 0x20,
		0x43, 0x45, 0x52, 0x54, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
		0x01, 0x09, 0x01, 0x16, 0x10, 0x64, 0x65, 0x76, 0x69, 0x63, 0x65, 0x30, 0x31, 0x40, 0x61, 0x72,
		0x6d, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d,
		0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04,
		0x6b, 0x28, 0x07, 0xf4, 0x9a, 0xf7, 0x19, 0x2b, 0x3f, 0x6a, 0xf4, 0xa1, 0xcb, 0x5f, 0x21, 0x18,
		0x46, 0xef, 0xcf, 0x9c, 0xbd, 0x31, 0x89, 0xac, 0x9c, 0xb6, 0xe4, 0x2c, 0xb9, 0x81, 0xae, 0x6d,
		0xb6, 0x57, 0x88, 0x18, 0xde, 0x06, 0x12, 0xe4, 0xfe, 0x63, 0xea, 0x13, 0x8b, 0xf7, 0x76, 0xe3,
		0x97, 0xc9, 0xbd, 0x47, 0xbf, 0x50, 0x92, 0xba, 0x1e, 0x3e, 0x3b, 0x8d, 0x6e, 0x8f, 0x86, 0xfd,
		0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30,
		0x45, 0x02, 0x20, 0x30, 0xc2, 0xd4, 0xbd, 0xa9, 0x5d, 0xe1, 0xd0, 0xc5, 0xcb, 0xb5, 0xb3, 0x31,
		0x8a, 0xad, 0x7a, 0xab, 0xc7, 0x85, 0xf2, 0x1a, 0x0a, 0x4f, 0xd1, 0x63, 0x33, 0x05, 0x20, 0x1f,
		0xb4, 0xd6, 0x6e, 0x02, 0x21, 0x00, 0x8c, 0xcd, 0x36, 0x73, 0x9c, 0xb7, 0xc2, 0xee, 0xf3, 0x0a,
		0xbf, 0x4f, 0xd3, 0x95, 0x08, 0xa6, 0xcd, 0xdd, 0x5b, 0x78, 0x8f, 0x83, 0x90, 0xef, 0x2a, 0x8a,
		0x27, 0x0d, 0x73, 0x0b, 0xa0, 0xfa
};

unsigned char cert_sha224_der[] = {
  		0x30, 0x82, 0x01, 0xf5, 0x30, 0x82, 0x01, 0x9c, 0x02, 0x09, 0x00, 0x9c, 0xe8, 0x23, 0xd2, 0x32,
		0xa3, 0x6a, 0x44, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x01, 0x30, 
		0x81, 0x82, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x4b, 0x31, 
		0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x4b, 0x66, 0x61, 0x72, 0x2d, 0x6e,
		0x65, 0x74, 0x65, 0x72, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x4e,
		0x65, 0x74, 0x61, 0x6e, 0x69, 0x61, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c,
		0x03, 0x41, 0x52, 0x4d, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x03, 0x49,
		0x4f, 0x54, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x06, 0x44, 0x65, 0x76,
		0x69, 0x63, 0x65, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
		0x09, 0x01, 0x16, 0x10, 0x64, 0x65, 0x76, 0x69, 0x63, 0x65, 0x30, 0x31, 0x40, 0x61, 0x72, 0x6d,
		0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x37, 0x30, 0x39, 0x31, 0x30, 0x31, 0x34,
		0x30, 0x33, 0x33, 0x33, 0x5a, 0x17, 0x0d, 0x32, 0x33, 0x30, 0x33, 0x30, 0x33, 0x31, 0x34, 0x30,
		0x33, 0x33, 0x33, 0x5a, 0x30, 0x81, 0x82, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x55, 0x4b, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x4b,
		0x66, 0x61, 0x72, 0x2d, 0x6e, 0x65, 0x74, 0x65, 0x72, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55,
		0x04, 0x07, 0x0c, 0x07, 0x4e, 0x65, 0x74, 0x61, 0x6e, 0x69, 0x61, 0x31, 0x0c, 0x30, 0x0a, 0x06,
		0x03, 0x55, 0x04, 0x0a, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55,
		0x04, 0x0b, 0x0c, 0x03, 0x49, 0x4f, 0x54, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x03,
		0x0c, 0x06, 0x44, 0x65, 0x76, 0x69, 0x63, 0x65, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86,
		0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x64, 0x65, 0x76, 0x69, 0x63, 0x65, 0x30,
		0x31, 0x40, 0x61, 0x72, 0x6d, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a,
		0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
		0x03, 0x42, 0x00, 0x04, 0x41, 0xdf, 0x03, 0x36, 0x9c, 0xb3, 0xb2, 0x9e, 0x42, 0xbc, 0x48, 0x37,
		0x3c, 0x21, 0xc2, 0x99, 0xb1, 0x47, 0x1d, 0x96, 0x84, 0x28, 0xdb, 0x07, 0x30, 0xad, 0x69, 0x86,
		0x39, 0x10, 0x10, 0x52, 0xe9, 0x3a, 0xe5, 0x4e, 0xd3, 0x3f, 0xd3, 0x0d, 0xfd, 0x9f, 0xd9, 0x4b,
		0xcb, 0xa1, 0x81, 0xbe, 0x3f, 0xbe, 0x24, 0x68, 0xc3, 0xe9, 0x73, 0xbf, 0x34, 0xb7, 0xa0, 0x61,
		0x56, 0x96, 0xb5, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x01,
		0x03, 0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x15, 0x93, 0xc8, 0xc3, 0x11, 0x3d, 0x66, 0x90, 0xf4,
		0xff, 0x9a, 0xb2, 0x1f, 0x97, 0x98, 0xb4, 0xde, 0xec, 0xef, 0x3a, 0xf1, 0x70, 0xd5, 0x04, 0x3b,
		0x64, 0x3b, 0x66, 0x5a, 0x31, 0x86, 0x6e, 0x02, 0x20, 0x37, 0xf7, 0xcf, 0x21, 0x30, 0xe7, 0xf8,
		0xff, 0x0c, 0x2a, 0x4a, 0xa7, 0xc3, 0x45, 0xee, 0x90, 0xd8, 0xda, 0xaf, 0x57, 0x9c, 0x32, 0xbd,
		0x2c, 0x4f, 0x3f, 0x3b, 0x43, 0x97, 0x18, 0xaa, 0x3d
};

typedef struct palX509VertifyTestVector
{
	const unsigned char* ca;
	size_t ca_size;
	const unsigned char* crt;
	size_t crt_size;
	palStatus_t result;
}palX509VertifyTestVector_t;

#if (PAL_USE_SECURE_TIME == 1)

static const palX509VertifyTestVector_t x509_verify_data[] =
{
		{ x509_selfsign, sizeof(x509_selfsign), x509_selfsign, sizeof(x509_selfsign), 0 },
		{ cert_sha224_der, sizeof(cert_sha224_der), cert_sha224_der, sizeof(cert_sha224_der), 0},
		{ NULL, 0, x509_selfsign, sizeof(x509_selfsign), PAL_ERR_X509_BADCERT_NOT_TRUSTED },
		{ x509_verify_ca, sizeof(x509_verify_ca), x509_verify_cert, sizeof(x509_verify_cert), PAL_ERR_X509_BADCERT_EXPIRED },
		{ x509_verify_cert, sizeof(x509_verify_cert), x509_verify_ca, sizeof(x509_verify_ca), PAL_ERR_X509_BADCERT_NOT_TRUSTED }
		//should simulate one of the cert verify tests to be future error
};

#else

static const palX509VertifyTestVector_t x509_verify_data[] = 
{
		{ x509_selfsign, sizeof(x509_selfsign), x509_selfsign, sizeof(x509_selfsign), 0 },
		{ cert_sha224_der, sizeof(cert_sha224_der), cert_sha224_der, sizeof(cert_sha224_der),0},
		{ NULL, 0, x509_selfsign, sizeof(x509_selfsign), PAL_ERR_X509_BADCERT_NOT_TRUSTED },
		{ x509_verify_ca, sizeof(x509_verify_ca), x509_verify_cert, sizeof(x509_verify_cert), 0 },
		{ x509_verify_cert, sizeof(x509_verify_cert), x509_verify_ca, sizeof(x509_verify_ca), PAL_ERR_X509_BADCERT_NOT_TRUSTED }
};


#endif //PAL_USE_SECURE_TIME


typedef struct palMdHMACTestVector
{
	const unsigned char key[131];
	const unsigned char input[152];
	unsigned char output[32];
	size_t keyLen;
	size_t inputLen;
	size_t outputLen;
} palMdHMACTestVector_t;

const palMdHMACTestVector_t mdHMACVector[7] =
{
		{
				{
						0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
						0x0b, 0x0b, 0x0b, 0x0b
				},
				{
						0x48, 0x69, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65
				},
				{
						0xb0, 0x34, 0x4c, 0x61, 0xd8, 0xdb, 0x38, 0x53, 0x5c, 0xa8, 0xaf, 0xce, 0xaf, 0x0b, 0xf1, 0x2b,
						0x88, 0x1d, 0xc2, 0x00, 0xc9, 0x83, 0x3d, 0xa7,	0x26, 0xe9, 0x37, 0x6c, 0x2e, 0x32, 0xcf, 0xf7
				}, 20, 8, 32
		},
		{
				{
						0x4a, 0x65, 0x66, 0x65
				},
				{
						0x77, 0x68, 0x61, 0x74, 0x20, 0x64, 0x6f, 0x20, 0x79, 0x61, 0x20, 0x77, 0x61, 0x6e, 0x74, 0x20,
						0x66, 0x6f, 0x72, 0x20, 0x6e, 0x6f, 0x74, 0x68, 0x69, 0x6e, 0x67, 0x3f
				},
				{
						0x5b, 0xdc, 0xc1, 0x46, 0xbf, 0x60, 0x75, 0x4e, 0x6a, 0x04, 0x24, 0x26, 0x08, 0x95, 0x75, 0xc7,
						0x5a, 0x00, 0x3f, 0x08, 0x9d, 0x27, 0x39, 0x83,	0x9d, 0xec, 0x58, 0xb9, 0x64, 0xec, 0x38, 0x43
				}, 4, 28, 32
		},
		{
				{
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa
				},
				{
						0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
						0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
						0xdd, 0xdd, 0xdd, 0xdd,	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
						0xdd, 0xdd
				},
				{
						0x77, 0x3e, 0xa9, 0x1e, 0x36, 0x80, 0x0e, 0x46, 0x85, 0x4d, 0xb8, 0xeb,	0xd0, 0x91, 0x81, 0xa7,
						0x29, 0x59, 0x09, 0x8b, 0x3e, 0xf8, 0xc1, 0x22,	0xd9, 0x63, 0x55, 0x14, 0xce, 0xd5, 0x65, 0xfe
				}, 20, 50, 32
		},
		{
				{
						0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
						0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19
				},
				{
						0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
						0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
						0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
						0xcd, 0xcd
				},
				{
						0x82, 0x55, 0x8a, 0x38, 0x9a, 0x44, 0x3c, 0x0e, 0xa4, 0xcc, 0x81, 0x98, 0x99, 0xf2, 0x08, 0x3a,
						0x85, 0xf0, 0xfa, 0xa3, 0xe5, 0x78, 0xf8, 0x07, 0x7a, 0x2e, 0x3f, 0xf4, 0x67, 0x29, 0x66, 0x5b
				}, 25, 50, 32
		},
		{
				{
						0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
						0x0c, 0x0c, 0x0c, 0x0c
				},
				{
						0x54, 0x65, 0x73, 0x74, 0x20, 0x57, 0x69, 0x74, 0x68, 0x20, 0x54, 0x72, 0x75, 0x6e, 0x63, 0x61,
						0x74, 0x69, 0x6f, 0x6e
				},
				{
						0xa3, 0xb6, 0x16, 0x74, 0x73, 0x10, 0x0e, 0xe0, 0x6e, 0x0c, 0x79, 0x6c, 0x29, 0x55, 0x55, 0x2b
				}, 20, 20, 16
		},
		{
				{
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa
				},

				{
						0x54, 0x65, 0x73, 0x74, 0x20, 0x55, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x4c, 0x61, 0x72, 0x67, 0x65,
						0x72, 0x20, 0x54, 0x68, 0x61, 0x6e, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x2d, 0x53, 0x69, 0x7a,
						0x65, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x2d, 0x20, 0x48, 0x61, 0x73, 0x68, 0x20, 0x4b, 0x65, 0x79,
						0x20, 0x46, 0x69, 0x72, 0x73, 0x74
				},

				{
						0x60, 0xe4, 0x31, 0x59, 0x1e, 0xe0, 0xb6, 0x7f, 0x0d, 0x8a, 0x26, 0xaa, 0xcb, 0xf5, 0xb7, 0x7f,
						0x8e, 0x0b, 0xc6, 0x21, 0x37, 0x28, 0xc5, 0x14, 0x05, 0x46, 0x04, 0x0f, 0x0e, 0xe3, 0x7f, 0x54
				}, 131, 54, 32
		},
		{
				{
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
						0xaa, 0xaa, 0xaa
				},
				{
						0x54, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x75,
						0x73, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x6c, 0x61, 0x72, 0x67, 0x65, 0x72, 0x20, 0x74, 0x68,
						0x61, 0x6e, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x2d, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x6b, 0x65,
						0x79, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x20, 0x6c, 0x61, 0x72, 0x67, 0x65, 0x72, 0x20, 0x74,
						0x68, 0x61, 0x6e, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x2d, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x64,
						0x61, 0x74, 0x61, 0x2e, 0x20, 0x54, 0x68, 0x65, 0x20, 0x6b, 0x65, 0x79, 0x20, 0x6e, 0x65, 0x65,
						0x64, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x62, 0x65, 0x20, 0x68, 0x61, 0x73, 0x68, 0x65, 0x64, 0x20,
						0x62, 0x65, 0x66, 0x6f, 0x72, 0x65, 0x20, 0x62, 0x65, 0x69, 0x6e, 0x67, 0x20, 0x75, 0x73, 0x65,
						0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x20, 0x48, 0x4d, 0x41, 0x43, 0x20, 0x61, 0x6c,
						0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x2e
				},
				{
						0x9b, 0x09, 0xff, 0xa7, 0x1b, 0x94, 0x2f, 0xcb, 0x27, 0x63, 0x5f, 0xbc, 0xd5, 0xb0, 0xe9, 0x44,
						0xbf, 0xdc, 0x63, 0x64, 0x4f, 0x07, 0x13, 0x93, 0x8a, 0x7f, 0x51, 0x53, 0x5c, 0x3a, 0x35, 0xe2
				}, 131, 152, 32
		}
};
