/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include "MMA8451Q.h"
#include "MotionSensor.h"

#define REG_WHOAMI      0x0D
#define REG_CTRL_REG_1    0x2A
#define REG_DR_STATUS     0x00
#define REG_OUT_X_MSB     0x01
#define REG_OUT_Y_MSB     0x03
#define REG_OUT_Z_MSB     0x05

#define UINT14_MAX        16383

float countsperG = 4096.0f; // default 2G range

MMA8451Q::MMA8451Q(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr) {
    // activate the peripheral
    uint8_t data[2] = {REG_CTRL_REG_1, 0x00};
    writeRegs(data, 2);
}

MMA8451Q::~MMA8451Q() { }

void MMA8451Q::enable(void) {
    uint8_t data[2];
    readRegs( REG_CTRL_REG_1, &data[1], 1);
    data[1] |= 0x01;
    data[0] = REG_CTRL_REG_1;
    writeRegs(data, 2);
}

void MMA8451Q::disable(void) {
    uint8_t data[2];
    readRegs( REG_CTRL_REG_1, &data[1], 1);
    data[1] &= 0xFE;
    data[0] = REG_CTRL_REG_1;
    writeRegs(data, 2);
}

uint32_t MMA8451Q::whoAmI() {
    uint8_t who_am_i = 0;
    readRegs(REG_WHOAMI, &who_am_i, 1);
    return (uint32_t) who_am_i;
}

uint32_t MMA8451Q::dataReady(void) {
    uint8_t stat = 0;
    readRegs(REG_DR_STATUS, &stat, 1);
    return (uint32_t) stat;
}

uint32_t MMA8451Q::sampleRate(uint32_t f) {
    return(50); // for now sample rate is fixed at 50Hz
}

void MMA8451Q::getX(float * x) {
    *x = (float(getAccAxis(REG_OUT_X_MSB)) / countsperG);
}

void MMA8451Q::getY(float * y) {
    *y = (float(getAccAxis(REG_OUT_Y_MSB)) / countsperG);
}

void MMA8451Q::getZ(float * z) {
    *z = (float(getAccAxis(REG_OUT_Z_MSB)) / countsperG);
}

void MMA8451Q::getX(int16_t * d) {
    *d = getAccAxis(REG_OUT_X_MSB);
}

void MMA8451Q::getY(int16_t * d) {
    *d = getAccAxis(REG_OUT_Y_MSB);
}

void MMA8451Q::getZ(int16_t * d) {
    *d = getAccAxis(REG_OUT_Z_MSB);
}

int16_t MMA8451Q::getAccAxis(uint8_t addr) {
    int16_t acc;
    uint8_t res[2];
    readRegs(addr, res, 2);

    acc = (res[0] << 8) | res[1];
    acc = acc >> 2;
    return acc;
}


void MMA8451Q::getAxis(MotionSensorDataUnits &data) {
    int16_t t[3];
    uint8_t res[6];

    readRegs(REG_OUT_X_MSB, res, 6);
    t[0] = (res[0] << 8) | res[1];
    t[1] = (res[2] << 8) | res[3];
    t[2] = (res[4] << 8) | res[5];
    data.x = ((float) (t[0] >> 2))  / countsperG;
    data.y = ((float) (t[1] >> 2))  / countsperG;
    data.z = ((float) (t[2] >> 2))  / countsperG;
}


void MMA8451Q::getAxis(MotionSensorDataCounts &data) {
    int16_t acc;
    uint8_t res[6];
    readRegs(REG_OUT_X_MSB, res, 6);


    acc = (res[0] << 8) | res[1];
    data.x = acc >> 2;
    acc = (res[2] << 8) | res[3];
    data.y = acc >> 2;
    acc = (res[4] << 8) | res[5];
    data.z = acc >> 2;
}

void MMA8451Q::readRegs(int addr, uint8_t * data, int len) {
    char t[1] = {addr};
    m_i2c.write(m_addr, t, 1, true);
    m_i2c.read(m_addr, (char *)data, len);
}

void MMA8451Q::writeRegs(uint8_t * data, int len) {
    m_i2c.write(m_addr, (char *)data, len);
}
