// ----------------------------------------------------------------------------
// Copyright 2016-2018 ARM Ltd.
//
// SPDX-License-Identifier: Apache-2.0
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// ----------------------------------------------------------------------------
#ifndef MBED_TEST_MODE

#include "mbed.h"
#include "simple-mbed-cloud-client.h"
#include "LittleFileSystem.h"

// Default network interface object. Don't forget to change the WiFi SSID/password in mbed_app.json if you're using WiFi.
NetworkInterface *net;

// Default block device available on the target board
BlockDevice* bd = BlockDevice::get_default_instance();
SlicingBlockDevice sd(bd, 0, 2*1024*1024);

#if COMPONENT_SD || COMPONENT_NUSD
// Use FATFileSystem for SD card type blockdevices
FATFileSystem fs("fs");
#else
// Use LittleFileSystem for non-SD block devices to enable wear leveling and other functions
LittleFileSystem fs("fs");
#endif

// Default User button for GET example and for resetting the storage
InterruptIn button(BUTTON1);
// Default LED to use for PUT/POST example
DigitalOut DoorActivate(PD_3, 0);

// How often to fetch sensor data (in seconds)
#define SENSORS_POLL_INTERVAL 15.0
int dflag = 1;
// Sensors related includes and initialization
#include "HTS221Sensor.h"
#include "LPS22HBSensor.h"
#include "VL53L0X.h"

static DevI2C devI2c(PB_11,PB_10);
static HTS221Sensor sen_hum_temp(&devI2c);
static LPS22HBSensor sen_press_temp(&devI2c);

static DigitalOut shutdown_pin(PC_6);
static VL53L0X sen_distance(&devI2c, &shutdown_pin, PC_7);

// Declaring pointers for access to Pelion Client resources outside of main()
MbedCloudClientResource *res_button;
MbedCloudClientResource *DoorControl;


// Additional resources for sensor readings

MbedCloudClientResource *res_humidity;
MbedCloudClientResource *res_temperature;
MbedCloudClientResource *res_pressure;
MbedCloudClientResource *res_distance;

// define your personal password
#define DoorPassword "MyPassword"

    char DoorCode[32];
    DigitalOut ConnectTrue(LED1);
    DigitalOut CodeError(LED2);
    int ErrorCount = 0;
    int ErrorLockoutTime = 0;
    
// An event queue is a very useful structure to debounce information between contexts (e.g. ISR and normal threads)
// This is great because things such as network operations are illegal in ISR, so updating a resource in a button's fall() function is not allowed
EventQueue eventQueue;

// When the device is registered, this variable will be used to access various useful information, like device ID etc.
static const ConnectorClientEndpointInfo* endpointInfo;

/**
 * PUT handler
 * @param resource The resource that triggered the callback
 * @param newValue Updated value for the resource
 */
void put_callback(MbedCloudClientResource *resource, m2m::String newValue) {

    if(ErrorLockoutTime != 0) { /* Ignore password attempts while locked */
        printf("Password Lockout!\r\n");
        return;
        }
    printf("*** PUT received, new value: %s                             \n", newValue.c_str());
    strcpy(DoorCode, newValue.c_str());
    printf("%s\r\n", DoorCode);
    if(!strcmp(DoorCode, DoorPassword )) {
        DoorActivate = 1; /* Activate relay for 1500ms */
        wait_ms(1500);
        DoorActivate = 0;
        DoorCode[0] = NULL;
        // clear password error data
        ErrorCount = 0;
        ErrorLockoutTime = 0;
        } else { /* bad password attempt */
        DoorCode[0] = NULL;
        ErrorCount++;
        if(ErrorCount >= 3) { /* If password submitted is wring lockout the remote control on the third error */
            ErrorLockoutTime = (ErrorCount - 2) * 2; /* Double the lockout time after every failed password */
            CodeError = 1; /* Indicate lockout on LED */
            printf("PasswordLocked %d Minutes\r\n", ErrorLockoutTime);
            }
        }
            
        
}



/**
 * Button function triggered by the physical button press.
 */
void button_press() {
    int v = res_button->get_value_int() + 1;
    res_button->set_value(v);
    printf("*** Button clicked %d times                                 \n", v);
}

/**
 * Notification callback handler
 * @param resource The resource that triggered the callback
 * @param status The delivery status of the notification
 */
void button_callback(MbedCloudClientResource *resource, const NoticationDeliveryStatus status) {
    printf("*** Button notification, status %s (%d)                     \n", MbedCloudClientResource::delivery_status_to_string(status), status);
}

/**
 * Registration callback handler
 * @param endpoint Information about the registered endpoint such as the name (so you can find it back in portal)
 */
void registered(const ConnectorClientEndpointInfo *endpoint) {
    printf("Registered to Pelion Device Management. Endpoint Name: %s\n", endpoint->internal_endpoint_name.c_str());
    endpointInfo = endpoint;
}

/**
 * Initialize sensors
 */
void sensors_init() {
    uint8_t id1, id2;

    printf ("\nSensors configuration:\n");
    // Initialize sensors
    sen_hum_temp.init(NULL);
    sen_press_temp.init(NULL);
    sen_distance.init_sensor(VL53L0X_DEFAULT_ADDRESS);

    /// Call sensors enable routines
    sen_hum_temp.enable();
    sen_press_temp.enable();


    sen_hum_temp.read_id(&id1);
    sen_press_temp.read_id(&id2);

    printf("HTS221  humidity & temperature    = 0x%X\n", id1);
    printf("LPS22HB pressure & temperature    = 0x%X\n", id2);

    printf("\n"); ;
}

/**
 * Update sensors and report their values.
 * This function is called periodically.
 */
void sensors_update() {
    float temp1_value, humid_value, pressure_value;
    uint32_t distance_value, distance_reading;

    if(!dflag++) { /* Only update the sensors once a minute */
        sen_hum_temp.get_humidity(&humid_value);
        sen_hum_temp.get_temperature(&temp1_value);
        sen_press_temp.get_pressure(&pressure_value);
        res_humidity->set_value(humid_value);
        res_temperature->set_value(temp1_value);
        res_pressure->set_value(pressure_value);
        if(ErrorLockoutTime != 0) {
            ErrorLockoutTime--;
            if(ErrorLockoutTime == 0) {
                printf("Unlocked\r\n");
                CodeError = 0;
                }
            }
        }
    if(dflag > 3) dflag = 0;
    distance_reading = sen_distance.get_distance(&distance_value);


    if (distance_reading == VL53L0X_ERROR_NONE) {
//        printf("VL53L0X dist: %7ld mm\n", distance_value);
        res_distance->set_value((int)distance_value);
    } else {
//        printf("VL53L0X dist:        --       \n");
        distance_value = 1999;
        res_distance->set_value((int)distance_value);
    }
}

int main(void) {
    printf("\nStarting Simple Pelion Device Management Client example\n");
    ConnectTrue = 0;
    CodeError = 0;
    DoorActivate = 0;
    int storage_status = fs.mount(&sd);
    if (storage_status != 0) {
        printf("Storage mounting failed.\n");
    }
    // If the User button is pressed ons start, then format storage.
    bool btn_pressed = (button.read() == MBED_CONF_APP_BUTTON_PRESSED_STATE);
    if (btn_pressed) {
        printf("User button is pushed on start...\n");
    }

    if (storage_status || btn_pressed) {
        printf("Formatting the storage...\n");
        int storage_status = StorageHelper::format(&fs, &sd);
        if (storage_status != 0) {
            printf("ERROR: Failed to reformat the storage (%d).\n", storage_status);
        }
    } else {
        printf("You can hold the user button during boot to format the storage and change the device identity.\n");
    }

    sensors_init();

    // Connect to the internet (DHCP is expected to be on)
    printf("Connecting to the network using Wifi...\n");
    net = NetworkInterface::get_default_instance();

    nsapi_error_t net_status = -1;
    for (int tries = 0; tries < 3; tries++) {
        net_status = net->connect();
        if (net_status == NSAPI_ERROR_OK) {
            break;
        } else {
            printf("Unable to connect to network. Retrying...\n");
        }
    }

    if (net_status != NSAPI_ERROR_OK) {
        printf("ERROR: Connecting to the network failed (%d)!\n", net_status);
        return -1;
    }

    printf("Connected to the network successfully. IP address: %s\n", net->get_ip_address());

    printf("Initializing Pelion Device Management Client...\n");

    // SimpleMbedCloudClient handles registering over LwM2M to Pelion DM
    SimpleMbedCloudClient client(net, bd, &fs);
    int client_status = client.init();
    if (client_status != 0) {
        printf("ERROR: Pelion Client initialization failed (%d)\n", client_status);
        return -1;
    }

    // Creating resources, which can be written or read from the cloud
    res_button = client.create_resource("3200/0/5501", "Button Count");
    res_button->set_value(0);
    res_button->methods(M2MMethod::GET);
    res_button->observable(true);
    //res_button->max_age(6000);
    res_button->attach_notification_callback(button_callback);

    DoorControl = client.create_resource("3201/0/5853", "Door Control");
    DoorControl->set_value(1);
    DoorControl->methods(M2MMethod::GET | M2MMethod::PUT);
    DoorControl->attach_put_callback(put_callback);

    // Sensor resources
    res_temperature = client.create_resource("3303/0/5700", "Temperature HTS221 (C)");
    res_temperature->set_value(0);
    res_temperature->methods(M2MMethod::GET);
    res_temperature->observable(true);

    res_humidity = client.create_resource("3304/0/5700", "Humidity");
    res_humidity->set_value(0);
    res_humidity->methods(M2MMethod::GET);
    res_humidity->observable(true);

    res_pressure = client.create_resource("3323/0/5700", "Pressure");
    res_pressure->set_value(0);
    res_pressure->methods(M2MMethod::GET);
    res_pressure->observable(true);

    res_distance = client.create_resource("3330/0/5700", "Distance");
    res_distance->set_value((float)999.9);
    res_distance->methods(M2MMethod::GET);
    res_distance->observable(true);

    printf("Initialized Pelion Client. Registering...\n");

    // Callback that fires when registering is complete
    client.on_registered(&registered);

    // Register with Pelion DM
    client.register_and_connect();

    int i = 6000; // wait up 600 seconds before attaching sensors and button events
    while (i-- > 0 && !client.is_client_registered()) {
        wait_ms(100);
    }
    ConnectTrue = 1;
    button.fall(eventQueue.event(&button_press));
    res_button->set_value(0);
    sensors_update();
    // The timer fires on an interrupt context, but debounces it to the eventqueue, so it's safe to do network operations
    Ticker timer;
    timer.attach(eventQueue.event(&sensors_update), SENSORS_POLL_INTERVAL);

    // You can easily run the eventQueue in a separate thread if required
    eventQueue.dispatch_forever();
}

#endif
