/**********************************
MBED1: Contains code for ->
1) Servo for Door Lock
2) JPG Camera (URL: http://mbed.org/cookbook/Camera_LS_Y201)
3) Distance Sensor
4) Lock LED / Pushbutton

Sends to Atom w/o Command:
1) BreakIn -> Signal and Pic
2) Door Open
3) Door Close
***********************************/

#include "mbed.h"
#include "Camera_LS_Y201.h"
#include "SDFileSystem.h"
#include "Servo.h"

//Setup
Serial pc(USBTX,USBRX);
Servo servo(p21);
AnalogIn door(p20);
DigitalOut lockLED(p11);    //1 = locked

//Pushbuttons
InterruptIn lockPB(p13);

//Globals
int isDoorOpen;
int isDoorLocked;

//File System
#define FILENAME    "/sd/IMG_%04d.jpg"
SDFileSystem fs(p5, p6, p7, p8, "sd");
int cnt = 0;
int sentCnt = 0;

//Camera
Camera_LS_Y201 cam1(p9, p10);
FILE* work;
FILE* fp;
int signalPic = 0;
 
 
 
 
 
 
 /***************************************/
/*           HELPER FUNCTIONS          */
/***************************************/
inline void servoLockDoor() {servo = .2;lockLED = 1;}
inline void servoUnlockDoor() {servo = .7;lockLED = 0;}
inline void sendID() {pc.printf("1\n");}
inline int checkLock() {return isDoorLocked;}
inline float checkDist() {return door;}
 
void lockPushed() {
    char out[50];
    if(checkLock()) {
        servoUnlockDoor();
        isDoorLocked = 0;
        sprintf(out, "X:0\n");
    }
    else {
        servoLockDoor();
        isDoorLocked = 1;
        sprintf(out, "X:1\n");
    }
    pc.printf(out);
}

void lockDoor() {
    char out[50];
    if(isDoorLocked) {
        sprintf(out, "X:1\n");
    }
    else {
        servoLockDoor();
        isDoorLocked = 1;
        sprintf(out, "X:1\n");
    }
    pc.printf(out);
}

void unlockDoor() {
    char out[50];
    if(!isDoorLocked) {
        sprintf(out, "X:0\n");
    }
    else {
        servoUnlockDoor();
        isDoorLocked = 0;
        sprintf(out, "X:0\n");
    }
    pc.printf(out);
}

void sendCheckLock() {
    char out[50];
    sprintf(out,"X:%d\n", checkLock());
    pc.printf(out);
}

void sendCheckDist() {

    char out[50];
    sprintf(out,"D:%d\n",(int) isDoorOpen);
    pc.printf(out);
}

void sendFullStatus() {
    char out[50];
    sprintf(out,"X:%d\nD:%d\n", checkLock(), (int)isDoorOpen);
    pc.printf(out);
}
 
 
 
 
 
/***************************************/
/*           CAMERA FUNCTIONS          */
/***************************************/
void callback_func(int done, int total, uint8_t *buf, size_t siz) {
    fwrite(buf, siz, 1, work);
}
 
int capture(Camera_LS_Y201 *cam, char *filename) {
    if (cam->takePicture() != 0) {
        return -1;
    }
 
    work = fopen(filename, "wb");
    if (work == NULL) {
        return -2;
    }
 
    if (cam->readJpegFileContent(callback_func) != 0) {
        fclose(work);
        return -3;
    }
    fclose(work);
 
    cam->stopTakingPictures();
    return 0;
}

void sendCurrPic() {
    char fname[64];
    snprintf(fname, sizeof(fname) - 1, FILENAME, cnt);
    int r = capture(&cam1, fname);
    cnt++;
}

void breakIn() {
    char out[50];
    sprintf(out, "B:0\n");
    pc.printf(out);
    sendCurrPic();
    //while(1);
}
 

/***************/
/* GET MESSAGE */
/***************/
void getMessage() {
    //Check for Message
    char signal = pc.getc();
    
    //Switch on Buffer
        switch(signal) {
            case 'o':
                sendID();
                break;
            case 'p':
                sendFullStatus();
                break;
            case 'q':
                sendCheckDist();
                break;
            case 'w':
                sendCheckLock();
                break;
            case 's':
                lockDoor();
                break;
            case 'x':
                unlockDoor();
                break;
            case 't':
                signalPic = 1;
                break;
            default:
                pc.printf("E\n");
                break;
        }
}
 
 
 


/***************************************/
/*                MAIN                 */
/***************************************/
int main(void) {

    //Setup
    lockLED = 1;
    isDoorLocked = 1;
    isDoorOpen = 0;
    servo = .2;
    
    //Serial Setup Conenction
    pc.baud(115200);
    pc.attach(&getMessage);
    
    //Setup Interrupts
    lockPB.mode(PullUp);
    wait(0.1);
    lockPB.fall(&lockPushed);
    
    //Setup Camera
    wait(1);
    if (cam1.reset() != 0) {
        return 1;
    }
    wait(1);
    
    //Variables
    float DOOR_DISTANCE = float(door) + .08;

 
    //Program
    while(1) {
        wait_ms(100);
        
        //Check Camera
        if(signalPic) {
            sendCurrPic();
            signalPic = 0;
        }
        
        //Check Door Dist and Send if needed
        if(isDoorOpen) {
            if(float(door) < DOOR_DISTANCE) {
                isDoorOpen = 0;
                pc.printf("D:0\n");
            }
        }
        else {
            if(float(door) > DOOR_DISTANCE) {
                isDoorOpen = 1;
                if(isDoorLocked) {
                    breakIn(); //Break In!
                }
                else {
                    pc.printf("D:1\n");
                }
            }
        }
        
        
    }//End While
}