/**********************************
MBED1: Contains code for ->
1) Servo for Door Lock
2) JPG Camera (URL: http://mbed.org/cookbook/Camera_LS_Y201)
3) Distance Sensor
4) Lock LED / Pushbutton

Sends to Atom w/o Command:
1) BreakIn -> Signal and Pic
2) Door Open
3) Door Close
***********************************/
#include "mbed.h"
#include "Camera_LS_Y201.h"
#include "Servo.h"

Serial pc(USBTX,USBRX);
Servo servo(p21);
AnalogIn door(p20);
Camera_LS_Y201 cam(p9, p10);
DigitalOut lockLED(p11);    //1 = locked

//Pushbuttons
InterruptIn lockPB(p12);

//Globals
FILE *picture;
char signal;
int isDoorOpen;
int isDoorLocked;








/***************************************/
/*           HELPER FUNCTIONS          */
/***************************************/
void servoLockDoor() {
    servo = 0;
    lockLED = 1;
}
void servoUnlockDoor() {
    servo = .5;
    lockLED = 0;
}


inline void sendID() {pc.putc('1');}
inline int checkLock() {return isDoorLocked;}
inline float checkDist() {return door;}

void writePic(int done, int total, uint8_t *buf, size_t siz) {
    fwrite(buf, siz, 1, picture);
}

void capture() {
    cam.takePicture();
    picture = freopen("pic.jpg", "w", picture);
    cam.readJpegFileContent(writePic);
    cam.stopTakingPictures();
    fclose(picture);
}

void lockPushed() {
    char out[50];
    if(checkLock()) {
        servoUnlockDoor();
        isDoorLocked = 0;
        sprintf(out, "USERUNLOCK\n\r");
    }
    else {
        servoLockDoor();
        isDoorLocked = 1;
        sprintf(out, "USERLOCK\n\r");
    }
}

void sendCurrPic() {
    char buff[200];
    capture();
    picture = freopen("pic.jpg", "r", picture);
    pc.printf("SENDINGPIC\n\r");
    while(fgets(buff, 200, picture)) {
        pc.printf(buff);
    }
    pc.printf("PICSENT\n\r");
}

void breakIn() {
    char out[50];
    sprintf(out, "BREAKIN\n\r");
    pc.printf(out);
    sendCurrPic();
}

void lockDoor() {
    char out[50];
    if(isDoorLocked) {
        sprintf(out, "ALREADYLOCKED\n\r");
    }
    else {
        servoLockDoor();
        isDoorLocked = 1;
        sprintf(out, "NOWLOCKED\n\r");
    }
    pc.printf(out);
}

void unlockDoor() {
    char out[50];
    if(!isDoorLocked) {
        sprintf(out, "ALREADYUNLOCKED\n\r");
    }
    else {
        servoUnlockDoor();
        isDoorLocked = 0;
        sprintf(out, "NOWUNLOCKED\n\r");
    }
    pc.printf(out);
}

void sendCheckLock() {
    char out[50];
    sprintf(out,"LOCK:%d\n\r", checkLock());
    pc.printf(out);
}

void sendCheckDist() {
    char out[50];
    sprintf(out,"DIST:%0.3f\n\r", checkDist());
    pc.printf(out);
}

void sendFullStatus() {
    char out[50];
    sprintf(out,"LOCK:%d|DIST:%0.3f\n\r", checkLock(), checkDist());
    pc.printf(out);
}

void getMessage() {
    //Check for Message
    signal = pc.getc();
    
    //Switch on Buffer
    switch(signal) {
        case 'Z':
            sendID();
            break;
        case '~':
            sendFullStatus();
            break;
        case 'a':
            sendCheckDist();
            break;
        case 'b':
            sendCheckLock();
            break;
        case 'c':
            lockDoor();
            break;
        case 'd':
            unlockDoor();
            break;
        case 'p':
            sendCurrPic();
            break;
        default:
            pc.printf("BADCOMMAND\n\r");
            break;
    }
}










/***************************************/
/*                MAIN                 */
/***************************************/
int main() {
    //Variables
    int DOOR_DISTANCE = door + .05;

    //Setup
    lockLED = 0;
    cam.reset();
    servo = 0;
    
    //Setup for freopen
    picture = fopen("pic.jpg", "w");
    fclose(picture);
    
    //Setup Interrupts
    pc.attach(&getMessage);
    lockPB.rise(&lockPushed);
    
    while(1) {
    
        //Check Door Dist and Send if needed
        if(isDoorOpen) {
            if(door < DOOR_DISTANCE) {
                isDoorOpen = 0;
                pc.printf("DOORCLOSE");
            }
        }
        else {
            if(door > DOOR_DISTANCE) {
                isDoorOpen = 1;
                if(isDoorLocked) {
                    breakIn(); //Break In!
                }
                else {
                    pc.printf("DOOROPEN");
                }
            }
        }
        wait_ms(50);
        
    } // End While(1)
}